/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from './output/output_ast';
const CONSTANT_PREFIX = '_c';
/**
 * `ConstantPool` tries to reuse literal factories when two or more literals are identical.
 * We determine whether literals are identical by creating a key out of their AST using the
 * `KeyVisitor`. This constant is used to replace dynamic expressions which can't be safely
 * converted into a key. E.g. given an expression `{foo: bar()}`, since we don't know what
 * the result of `bar` will be, we create a key that looks like `{foo: <unknown>}`. Note
 * that we use a variable, rather than something like `null` in order to avoid collisions.
 */
const UNKNOWN_VALUE_KEY = o.variable('<unknown>');
/**
 * Context to use when producing a key.
 *
 * This ensures we see the constant not the reference variable when producing
 * a key.
 */
const KEY_CONTEXT = {};
/**
 * Generally all primitive values are excluded from the `ConstantPool`, but there is an exclusion
 * for strings that reach a certain length threshold. This constant defines the length threshold for
 * strings.
 */
const POOL_INCLUSION_LENGTH_THRESHOLD_FOR_STRINGS = 50;
/**
 * A node that is a place-holder that allows the node to be replaced when the actual
 * node is known.
 *
 * This allows the constant pool to change an expression from a direct reference to
 * a constant to a shared constant. It returns a fix-up node that is later allowed to
 * change the referenced expression.
 */
class FixupExpression extends o.Expression {
    constructor(resolved) {
        super(resolved.type);
        this.resolved = resolved;
        this.shared = false;
        this.original = resolved;
    }
    visitExpression(visitor, context) {
        if (context === KEY_CONTEXT) {
            // When producing a key we want to traverse the constant not the
            // variable used to refer to it.
            return this.original.visitExpression(visitor, context);
        }
        else {
            return this.resolved.visitExpression(visitor, context);
        }
    }
    isEquivalent(e) {
        return e instanceof FixupExpression && this.resolved.isEquivalent(e.resolved);
    }
    isConstant() {
        return true;
    }
    clone() {
        throw new Error(`Not supported.`);
    }
    fixup(expression) {
        this.resolved = expression;
        this.shared = true;
    }
}
/**
 * A constant pool allows a code emitter to share constant in an output context.
 *
 * The constant pool also supports sharing access to ivy definitions references.
 */
export class ConstantPool {
    constructor(isClosureCompilerEnabled = false) {
        this.isClosureCompilerEnabled = isClosureCompilerEnabled;
        this.statements = [];
        this.literals = new Map();
        this.literalFactories = new Map();
        this.sharedConstants = new Map();
        /**
         * Constant pool also tracks claimed names from {@link uniqueName}.
         * This is useful to avoid collisions if variables are intended to be
         * named a certain way- but may conflict. We wouldn't want to always suffix
         * them with unique numbers.
         */
        this._claimedNames = new Map();
        this.nextNameIndex = 0;
    }
    getConstLiteral(literal, forceShared) {
        if ((literal instanceof o.LiteralExpr && !isLongStringLiteral(literal)) ||
            literal instanceof FixupExpression) {
            // Do no put simple literals into the constant pool or try to produce a constant for a
            // reference to a constant.
            return literal;
        }
        const key = GenericKeyFn.INSTANCE.keyOf(literal);
        let fixup = this.literals.get(key);
        let newValue = false;
        if (!fixup) {
            fixup = new FixupExpression(literal);
            this.literals.set(key, fixup);
            newValue = true;
        }
        if ((!newValue && !fixup.shared) || (newValue && forceShared)) {
            // Replace the expression with a variable
            const name = this.freshName();
            let definition;
            let usage;
            if (this.isClosureCompilerEnabled && isLongStringLiteral(literal)) {
                // For string literals, Closure will **always** inline the string at
                // **all** usages, duplicating it each time. For large strings, this
                // unnecessarily bloats bundle size. To work around this restriction, we
                // wrap the string in a function, and call that function for each usage.
                // This tricks Closure into using inline logic for functions instead of
                // string literals. Function calls are only inlined if the body is small
                // enough to be worth it. By doing this, very large strings will be
                // shared across multiple usages, rather than duplicating the string at
                // each usage site.
                //
                // const myStr = function() { return "very very very long string"; };
                // const usage1 = myStr();
                // const usage2 = myStr();
                definition = o.variable(name).set(new o.FunctionExpr([], // Params.
                [
                    // Statements.
                    new o.ReturnStatement(literal),
                ]));
                usage = o.variable(name).callFn([]);
            }
            else {
                // Just declare and use the variable directly, without a function call
                // indirection. This saves a few bytes and avoids an unnecessary call.
                definition = o.variable(name).set(literal);
                usage = o.variable(name);
            }
            this.statements.push(definition.toDeclStmt(o.INFERRED_TYPE, o.StmtModifier.Final));
            fixup.fixup(usage);
        }
        return fixup;
    }
    getSharedConstant(def, expr) {
        const key = def.keyOf(expr);
        if (!this.sharedConstants.has(key)) {
            const id = this.freshName();
            this.sharedConstants.set(key, o.variable(id));
            this.statements.push(def.toSharedConstantDeclaration(id, expr));
        }
        return this.sharedConstants.get(key);
    }
    getLiteralFactory(literal) {
        // Create a pure function that builds an array of a mix of constant and variable expressions
        if (literal instanceof o.LiteralArrayExpr) {
            const argumentsForKey = literal.entries.map((e) => (e.isConstant() ? e : UNKNOWN_VALUE_KEY));
            const key = GenericKeyFn.INSTANCE.keyOf(o.literalArr(argumentsForKey));
            return this._getLiteralFactory(key, literal.entries, (entries) => o.literalArr(entries));
        }
        else {
            const expressionForKey = o.literalMap(literal.entries.map((e) => ({
                key: e.key,
                value: e.value.isConstant() ? e.value : UNKNOWN_VALUE_KEY,
                quoted: e.quoted,
            })));
            const key = GenericKeyFn.INSTANCE.keyOf(expressionForKey);
            return this._getLiteralFactory(key, literal.entries.map((e) => e.value), (entries) => o.literalMap(entries.map((value, index) => ({
                key: literal.entries[index].key,
                value,
                quoted: literal.entries[index].quoted,
            }))));
        }
    }
    // TODO: useUniqueName(false) is necessary for naming compatibility with
    // TemplateDefinitionBuilder, but should be removed once Template Pipeline is the default.
    getSharedFunctionReference(fn, prefix, useUniqueName = true) {
        const isArrow = fn instanceof o.ArrowFunctionExpr;
        for (const current of this.statements) {
            // Arrow functions are saved as variables so we check if the
            // value of the variable is the same as the arrow function.
            if (isArrow && current instanceof o.DeclareVarStmt && current.value?.isEquivalent(fn)) {
                return o.variable(current.name);
            }
            // Function declarations are saved as function statements
            // so we compare them directly to the passed-in function.
            if (!isArrow &&
                current instanceof o.DeclareFunctionStmt &&
                fn instanceof o.FunctionExpr &&
                fn.isEquivalent(current)) {
                return o.variable(current.name);
            }
        }
        // Otherwise declare the function.
        const name = useUniqueName ? this.uniqueName(prefix) : prefix;
        this.statements.push(fn instanceof o.FunctionExpr
            ? fn.toDeclStmt(name, o.StmtModifier.Final)
            : new o.DeclareVarStmt(name, fn, o.INFERRED_TYPE, o.StmtModifier.Final, fn.sourceSpan));
        return o.variable(name);
    }
    _getLiteralFactory(key, values, resultMap) {
        let literalFactory = this.literalFactories.get(key);
        const literalFactoryArguments = values.filter((e) => !e.isConstant());
        if (!literalFactory) {
            const resultExpressions = values.map((e, index) => e.isConstant() ? this.getConstLiteral(e, true) : o.variable(`a${index}`));
            const parameters = resultExpressions
                .filter(isVariable)
                .map((e) => new o.FnParam(e.name, o.DYNAMIC_TYPE));
            const pureFunctionDeclaration = o.arrowFn(parameters, resultMap(resultExpressions), o.INFERRED_TYPE);
            const name = this.freshName();
            this.statements.push(o
                .variable(name)
                .set(pureFunctionDeclaration)
                .toDeclStmt(o.INFERRED_TYPE, o.StmtModifier.Final));
            literalFactory = o.variable(name);
            this.literalFactories.set(key, literalFactory);
        }
        return { literalFactory, literalFactoryArguments };
    }
    /**
     * Produce a unique name in the context of this pool.
     *
     * The name might be unique among different prefixes if any of the prefixes end in
     * a digit so the prefix should be a constant string (not based on user input) and
     * must not end in a digit.
     */
    uniqueName(name, alwaysIncludeSuffix = true) {
        const count = this._claimedNames.get(name) ?? 0;
        const result = count === 0 && !alwaysIncludeSuffix ? `${name}` : `${name}${count}`;
        this._claimedNames.set(name, count + 1);
        return result;
    }
    freshName() {
        return this.uniqueName(CONSTANT_PREFIX);
    }
}
export class GenericKeyFn {
    static { this.INSTANCE = new GenericKeyFn(); }
    keyOf(expr) {
        if (expr instanceof o.LiteralExpr && typeof expr.value === 'string') {
            return `"${expr.value}"`;
        }
        else if (expr instanceof o.LiteralExpr) {
            return String(expr.value);
        }
        else if (expr instanceof o.LiteralArrayExpr) {
            const entries = [];
            for (const entry of expr.entries) {
                entries.push(this.keyOf(entry));
            }
            return `[${entries.join(',')}]`;
        }
        else if (expr instanceof o.LiteralMapExpr) {
            const entries = [];
            for (const entry of expr.entries) {
                let key = entry.key;
                if (entry.quoted) {
                    key = `"${key}"`;
                }
                entries.push(key + ':' + this.keyOf(entry.value));
            }
            return `{${entries.join(',')}}`;
        }
        else if (expr instanceof o.ExternalExpr) {
            return `import("${expr.value.moduleName}", ${expr.value.name})`;
        }
        else if (expr instanceof o.ReadVarExpr) {
            return `read(${expr.name})`;
        }
        else if (expr instanceof o.TypeofExpr) {
            return `typeof(${this.keyOf(expr.expr)})`;
        }
        else {
            throw new Error(`${this.constructor.name} does not handle expressions of type ${expr.constructor.name}`);
        }
    }
}
function isVariable(e) {
    return e instanceof o.ReadVarExpr;
}
function isLongStringLiteral(expr) {
    return (expr instanceof o.LiteralExpr &&
        typeof expr.value === 'string' &&
        expr.value.length >= POOL_INCLUSION_LENGTH_THRESHOLD_FOR_STRINGS);
}
//# sourceMappingURL=data:application/json;base64,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