/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as core from '../../core';
import { DEFAULT_INTERPOLATION_CONFIG } from '../../ml_parser/defaults';
import * as o from '../../output/output_ast';
import { ParseLocation, ParseSourceFile, ParseSourceSpan } from '../../parse_util';
import { RecursiveVisitor, visitAll } from '../r3_ast';
import { Identifiers as R3 } from '../r3_identifiers';
import { generateForwardRef } from '../util';
import { R3TemplateDependencyKind, } from '../view/api';
import { createComponentType } from '../view/compiler';
import { DefinitionMap } from '../view/util';
import { createDirectiveDefinitionMap } from './directive';
import { toOptionalLiteralArray } from './util';
/**
 * Compile a component declaration defined by the `R3ComponentMetadata`.
 */
export function compileDeclareComponentFromMetadata(meta, template, additionalTemplateInfo) {
    const definitionMap = createComponentDefinitionMap(meta, template, additionalTemplateInfo);
    const expression = o.importExpr(R3.declareComponent).callFn([definitionMap.toLiteralMap()]);
    const type = createComponentType(meta);
    return { expression, type, statements: [] };
}
/**
 * Gathers the declaration fields for a component into a `DefinitionMap`.
 */
export function createComponentDefinitionMap(meta, template, templateInfo) {
    const definitionMap = createDirectiveDefinitionMap(meta);
    const blockVisitor = new BlockPresenceVisitor();
    visitAll(blockVisitor, template.nodes);
    definitionMap.set('template', getTemplateExpression(template, templateInfo));
    if (templateInfo.isInline) {
        definitionMap.set('isInline', o.literal(true));
    }
    // Set the minVersion to 17.0.0 if the component is using at least one block in its template.
    // We don't do this for templates without blocks, in order to preserve backwards compatibility.
    if (blockVisitor.hasBlocks) {
        definitionMap.set('minVersion', o.literal('17.0.0'));
    }
    definitionMap.set('styles', toOptionalLiteralArray(meta.styles, o.literal));
    definitionMap.set('dependencies', compileUsedDependenciesMetadata(meta));
    definitionMap.set('viewProviders', meta.viewProviders);
    definitionMap.set('animations', meta.animations);
    if (meta.changeDetection !== null) {
        if (typeof meta.changeDetection === 'object') {
            throw new Error('Impossible state! Change detection flag is not resolved!');
        }
        definitionMap.set('changeDetection', o
            .importExpr(R3.ChangeDetectionStrategy)
            .prop(core.ChangeDetectionStrategy[meta.changeDetection]));
    }
    if (meta.encapsulation !== core.ViewEncapsulation.Emulated) {
        definitionMap.set('encapsulation', o.importExpr(R3.ViewEncapsulation).prop(core.ViewEncapsulation[meta.encapsulation]));
    }
    if (meta.interpolation !== DEFAULT_INTERPOLATION_CONFIG) {
        definitionMap.set('interpolation', o.literalArr([o.literal(meta.interpolation.start), o.literal(meta.interpolation.end)]));
    }
    if (template.preserveWhitespaces === true) {
        definitionMap.set('preserveWhitespaces', o.literal(true));
    }
    if (meta.defer.mode === 0 /* DeferBlockDepsEmitMode.PerBlock */) {
        const resolvers = [];
        let hasResolvers = false;
        for (const deps of meta.defer.blocks.values()) {
            // Note: we need to push a `null` even if there are no dependencies, because matching of
            // defer resolver functions to defer blocks happens by index and not adding an array
            // entry for a block can throw off the blocks coming after it.
            if (deps === null) {
                resolvers.push(o.literal(null));
            }
            else {
                resolvers.push(deps);
                hasResolvers = true;
            }
        }
        // If *all* the resolvers are null, we can skip the field.
        if (hasResolvers) {
            definitionMap.set('deferBlockDependencies', o.literalArr(resolvers));
        }
    }
    else {
        throw new Error('Unsupported defer function emit mode in partial compilation');
    }
    return definitionMap;
}
function getTemplateExpression(template, templateInfo) {
    // If the template has been defined using a direct literal, we use that expression directly
    // without any modifications. This is ensures proper source mapping from the partially
    // compiled code to the source file declaring the template. Note that this does not capture
    // template literals referenced indirectly through an identifier.
    if (templateInfo.inlineTemplateLiteralExpression !== null) {
        return templateInfo.inlineTemplateLiteralExpression;
    }
    // If the template is defined inline but not through a literal, the template has been resolved
    // through static interpretation. We create a literal but cannot provide any source span. Note
    // that we cannot use the expression defining the template because the linker expects the template
    // to be defined as a literal in the declaration.
    if (templateInfo.isInline) {
        return o.literal(templateInfo.content, null, null);
    }
    // The template is external so we must synthesize an expression node with
    // the appropriate source-span.
    const contents = templateInfo.content;
    const file = new ParseSourceFile(contents, templateInfo.sourceUrl);
    const start = new ParseLocation(file, 0, 0, 0);
    const end = computeEndLocation(file, contents);
    const span = new ParseSourceSpan(start, end);
    return o.literal(contents, null, span);
}
function computeEndLocation(file, contents) {
    const length = contents.length;
    let lineStart = 0;
    let lastLineStart = 0;
    let line = 0;
    do {
        lineStart = contents.indexOf('\n', lastLineStart);
        if (lineStart !== -1) {
            lastLineStart = lineStart + 1;
            line++;
        }
    } while (lineStart !== -1);
    return new ParseLocation(file, length, line, length - lastLineStart);
}
function compileUsedDependenciesMetadata(meta) {
    const wrapType = meta.declarationListEmitMode !== 0 /* DeclarationListEmitMode.Direct */
        ? generateForwardRef
        : (expr) => expr;
    if (meta.declarationListEmitMode === 3 /* DeclarationListEmitMode.RuntimeResolved */) {
        throw new Error(`Unsupported emit mode`);
    }
    return toOptionalLiteralArray(meta.declarations, (decl) => {
        switch (decl.kind) {
            case R3TemplateDependencyKind.Directive:
                const dirMeta = new DefinitionMap();
                dirMeta.set('kind', o.literal(decl.isComponent ? 'component' : 'directive'));
                dirMeta.set('type', wrapType(decl.type));
                dirMeta.set('selector', o.literal(decl.selector));
                dirMeta.set('inputs', toOptionalLiteralArray(decl.inputs, o.literal));
                dirMeta.set('outputs', toOptionalLiteralArray(decl.outputs, o.literal));
                dirMeta.set('exportAs', toOptionalLiteralArray(decl.exportAs, o.literal));
                return dirMeta.toLiteralMap();
            case R3TemplateDependencyKind.Pipe:
                const pipeMeta = new DefinitionMap();
                pipeMeta.set('kind', o.literal('pipe'));
                pipeMeta.set('type', wrapType(decl.type));
                pipeMeta.set('name', o.literal(decl.name));
                return pipeMeta.toLiteralMap();
            case R3TemplateDependencyKind.NgModule:
                const ngModuleMeta = new DefinitionMap();
                ngModuleMeta.set('kind', o.literal('ngmodule'));
                ngModuleMeta.set('type', wrapType(decl.type));
                return ngModuleMeta.toLiteralMap();
        }
    });
}
class BlockPresenceVisitor extends RecursiveVisitor {
    constructor() {
        super(...arguments);
        this.hasBlocks = false;
    }
    visitDeferredBlock() {
        this.hasBlocks = true;
    }
    visitDeferredBlockPlaceholder() {
        this.hasBlocks = true;
    }
    visitDeferredBlockLoading() {
        this.hasBlocks = true;
    }
    visitDeferredBlockError() {
        this.hasBlocks = true;
    }
    visitIfBlock() {
        this.hasBlocks = true;
    }
    visitIfBlockBranch() {
        this.hasBlocks = true;
    }
    visitForLoopBlock() {
        this.hasBlocks = true;
    }
    visitForLoopBlockEmpty() {
        this.hasBlocks = true;
    }
    visitSwitchBlock() {
        this.hasBlocks = true;
    }
    visitSwitchBlockCase() {
        this.hasBlocks = true;
    }
}
//# sourceMappingURL=data:application/json;base64,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