/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../output/output_ast';
import { Identifiers as R3 } from './r3_identifiers';
import { devOnlyGuardedExpression } from './util';
export function compileClassMetadata(metadata) {
    const fnCall = internalCompileClassMetadata(metadata);
    return o.arrowFn([], [devOnlyGuardedExpression(fnCall).toStmt()]).callFn([]);
}
/** Compiles only the `setClassMetadata` call without any additional wrappers. */
function internalCompileClassMetadata(metadata) {
    return o
        .importExpr(R3.setClassMetadata)
        .callFn([
        metadata.type,
        metadata.decorators,
        metadata.ctorParameters ?? o.literal(null),
        metadata.propDecorators ?? o.literal(null),
    ]);
}
/**
 * Wraps the `setClassMetadata` function with extra logic that dynamically
 * loads dependencies from `@defer` blocks.
 *
 * Generates a call like this:
 * ```
 * setClassMetadataAsync(type, () => [
 *   import('./cmp-a').then(m => m.CmpA);
 *   import('./cmp-b').then(m => m.CmpB);
 * ], (CmpA, CmpB) => {
 *   setClassMetadata(type, decorators, ctorParameters, propParameters);
 * });
 * ```
 *
 * Similar to the `setClassMetadata` call, it's wrapped into the `ngDevMode`
 * check to tree-shake away this code in production mode.
 */
export function compileComponentClassMetadata(metadata, dependencies) {
    if (dependencies === null || dependencies.length === 0) {
        // If there are no deferrable symbols - just generate a regular `setClassMetadata` call.
        return compileClassMetadata(metadata);
    }
    return internalCompileSetClassMetadataAsync(metadata, dependencies.map((dep) => new o.FnParam(dep.symbolName, o.DYNAMIC_TYPE)), compileComponentMetadataAsyncResolver(dependencies));
}
/**
 * Identical to `compileComponentClassMetadata`. Used for the cases where we're unable to
 * analyze the deferred block dependencies, but we have a reference to the compiled
 * dependency resolver function that we can use as is.
 * @param metadata Class metadata for the internal `setClassMetadata` call.
 * @param deferResolver Expression representing the deferred dependency loading function.
 * @param deferredDependencyNames Names of the dependencies that are being loaded asynchronously.
 */
export function compileOpaqueAsyncClassMetadata(metadata, deferResolver, deferredDependencyNames) {
    return internalCompileSetClassMetadataAsync(metadata, deferredDependencyNames.map((name) => new o.FnParam(name, o.DYNAMIC_TYPE)), deferResolver);
}
/**
 * Internal logic used to compile a `setClassMetadataAsync` call.
 * @param metadata Class metadata for the internal `setClassMetadata` call.
 * @param wrapperParams Parameters to be set on the callback that wraps `setClassMetata`.
 * @param dependencyResolverFn Function to resolve the deferred dependencies.
 */
function internalCompileSetClassMetadataAsync(metadata, wrapperParams, dependencyResolverFn) {
    // Omit the wrapper since it'll be added around `setClassMetadataAsync` instead.
    const setClassMetadataCall = internalCompileClassMetadata(metadata);
    const setClassMetaWrapper = o.arrowFn(wrapperParams, [setClassMetadataCall.toStmt()]);
    const setClassMetaAsync = o
        .importExpr(R3.setClassMetadataAsync)
        .callFn([metadata.type, dependencyResolverFn, setClassMetaWrapper]);
    return o.arrowFn([], [devOnlyGuardedExpression(setClassMetaAsync).toStmt()]).callFn([]);
}
/**
 * Compiles the function that loads the dependencies for the
 * entire component in `setClassMetadataAsync`.
 */
export function compileComponentMetadataAsyncResolver(dependencies) {
    const dynamicImports = dependencies.map(({ symbolName, importPath, isDefaultImport }) => {
        // e.g. `(m) => m.CmpA`
        const innerFn = 
        // Default imports are always accessed through the `default` property.
        o.arrowFn([new o.FnParam('m', o.DYNAMIC_TYPE)], o.variable('m').prop(isDefaultImport ? 'default' : symbolName));
        // e.g. `import('./cmp-a').then(...)`
        return new o.DynamicImportExpr(importPath).prop('then').callFn([innerFn]);
    });
    // e.g. `() => [ ... ];`
    return o.arrowFn([], o.literalArr(dynamicImports));
}
//# sourceMappingURL=data:application/json;base64,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