/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../../output/output_ast';
import { Identifiers as R3 } from '../r3_identifiers';
import { CONTEXT_NAME, RENDER_FLAGS, TEMPORARY_NAME, temporaryAllocator } from './util';
//  if (rf & flags) { .. }
function renderFlagCheckIfStmt(flags, statements) {
    return o.ifStmt(o.variable(RENDER_FLAGS).bitwiseAnd(o.literal(flags), null, false), statements);
}
/**
 * Translates query flags into `TQueryFlags` type in
 * packages/core/src/render3/interfaces/query.ts
 * @param query
 */
function toQueryFlags(query) {
    return ((query.descendants ? 1 /* QueryFlags.descendants */ : 0 /* QueryFlags.none */) |
        (query.static ? 2 /* QueryFlags.isStatic */ : 0 /* QueryFlags.none */) |
        (query.emitDistinctChangesOnly ? 4 /* QueryFlags.emitDistinctChangesOnly */ : 0 /* QueryFlags.none */));
}
export function getQueryPredicate(query, constantPool) {
    if (Array.isArray(query.predicate)) {
        let predicate = [];
        query.predicate.forEach((selector) => {
            // Each item in predicates array may contain strings with comma-separated refs
            // (for ex. 'ref, ref1, ..., refN'), thus we extract individual refs and store them
            // as separate array entities
            const selectors = selector.split(',').map((token) => o.literal(token.trim()));
            predicate.push(...selectors);
        });
        return constantPool.getConstLiteral(o.literalArr(predicate), true);
    }
    else {
        // The original predicate may have been wrapped in a `forwardRef()` call.
        switch (query.predicate.forwardRef) {
            case 0 /* ForwardRefHandling.None */:
            case 2 /* ForwardRefHandling.Unwrapped */:
                return query.predicate.expression;
            case 1 /* ForwardRefHandling.Wrapped */:
                return o.importExpr(R3.resolveForwardRef).callFn([query.predicate.expression]);
        }
    }
}
export function createQueryCreateCall(query, constantPool, queryTypeFns, prependParams) {
    const parameters = [];
    if (prependParams !== undefined) {
        parameters.push(...prependParams);
    }
    if (query.isSignal) {
        parameters.push(new o.ReadPropExpr(o.variable(CONTEXT_NAME), query.propertyName));
    }
    parameters.push(getQueryPredicate(query, constantPool), o.literal(toQueryFlags(query)));
    if (query.read) {
        parameters.push(query.read);
    }
    const queryCreateFn = query.isSignal ? queryTypeFns.signalBased : queryTypeFns.nonSignal;
    return o.importExpr(queryCreateFn).callFn(parameters);
}
const queryAdvancePlaceholder = Symbol('queryAdvancePlaceholder');
/**
 * Collapses query advance placeholders in a list of statements.
 *
 * This allows for less generated code because multiple sibling query advance
 * statements can be collapsed into a single call with the count as argument.
 *
 * e.g.
 *
 * ```ts
 *   bla();
 *   queryAdvance();
 *   queryAdvance();
 *   bla();
 * ```
 *
 *   --> will turn into
 *
 * ```
 *   bla();
 *   queryAdvance(2);
 *   bla();
 * ```
 */
function collapseAdvanceStatements(statements) {
    const result = [];
    let advanceCollapseCount = 0;
    const flushAdvanceCount = () => {
        if (advanceCollapseCount > 0) {
            result.unshift(o
                .importExpr(R3.queryAdvance)
                .callFn(advanceCollapseCount === 1 ? [] : [o.literal(advanceCollapseCount)])
                .toStmt());
            advanceCollapseCount = 0;
        }
    };
    // Iterate through statements in reverse and collapse advance placeholders.
    for (let i = statements.length - 1; i >= 0; i--) {
        const st = statements[i];
        if (st === queryAdvancePlaceholder) {
            advanceCollapseCount++;
        }
        else {
            flushAdvanceCount();
            result.unshift(st);
        }
    }
    flushAdvanceCount();
    return result;
}
// Define and update any view queries
export function createViewQueriesFunction(viewQueries, constantPool, name) {
    const createStatements = [];
    const updateStatements = [];
    const tempAllocator = temporaryAllocator((st) => updateStatements.push(st), TEMPORARY_NAME);
    viewQueries.forEach((query) => {
        // creation call, e.g. r3.viewQuery(somePredicate, true) or
        //                r3.viewQuerySignal(ctx.prop, somePredicate, true);
        const queryDefinitionCall = createQueryCreateCall(query, constantPool, {
            signalBased: R3.viewQuerySignal,
            nonSignal: R3.viewQuery,
        });
        createStatements.push(queryDefinitionCall.toStmt());
        // Signal queries update lazily and we just advance the index.
        if (query.isSignal) {
            updateStatements.push(queryAdvancePlaceholder);
            return;
        }
        // update, e.g. (r3.queryRefresh(tmp = r3.loadQuery()) && (ctx.someDir = tmp));
        const temporary = tempAllocator();
        const getQueryList = o.importExpr(R3.loadQuery).callFn([]);
        const refresh = o.importExpr(R3.queryRefresh).callFn([temporary.set(getQueryList)]);
        const updateDirective = o
            .variable(CONTEXT_NAME)
            .prop(query.propertyName)
            .set(query.first ? temporary.prop('first') : temporary);
        updateStatements.push(refresh.and(updateDirective).toStmt());
    });
    const viewQueryFnName = name ? `${name}_Query` : null;
    return o.fn([new o.FnParam(RENDER_FLAGS, o.NUMBER_TYPE), new o.FnParam(CONTEXT_NAME, null)], [
        renderFlagCheckIfStmt(1 /* core.RenderFlags.Create */, createStatements),
        renderFlagCheckIfStmt(2 /* core.RenderFlags.Update */, collapseAdvanceStatements(updateStatements)),
    ], o.INFERRED_TYPE, null, viewQueryFnName);
}
// Define and update any content queries
export function createContentQueriesFunction(queries, constantPool, name) {
    const createStatements = [];
    const updateStatements = [];
    const tempAllocator = temporaryAllocator((st) => updateStatements.push(st), TEMPORARY_NAME);
    for (const query of queries) {
        // creation, e.g. r3.contentQuery(dirIndex, somePredicate, true, null) or
        //                r3.contentQuerySignal(dirIndex, propName, somePredicate, <flags>, <read>).
        createStatements.push(createQueryCreateCall(query, constantPool, { nonSignal: R3.contentQuery, signalBased: R3.contentQuerySignal }, 
        /* prependParams */ [o.variable('dirIndex')]).toStmt());
        // Signal queries update lazily and we just advance the index.
        if (query.isSignal) {
            updateStatements.push(queryAdvancePlaceholder);
            continue;
        }
        // update, e.g. (r3.queryRefresh(tmp = r3.loadQuery()) && (ctx.someDir = tmp));
        const temporary = tempAllocator();
        const getQueryList = o.importExpr(R3.loadQuery).callFn([]);
        const refresh = o.importExpr(R3.queryRefresh).callFn([temporary.set(getQueryList)]);
        const updateDirective = o
            .variable(CONTEXT_NAME)
            .prop(query.propertyName)
            .set(query.first ? temporary.prop('first') : temporary);
        updateStatements.push(refresh.and(updateDirective).toStmt());
    }
    const contentQueriesFnName = name ? `${name}_ContentQueries` : null;
    return o.fn([
        new o.FnParam(RENDER_FLAGS, o.NUMBER_TYPE),
        new o.FnParam(CONTEXT_NAME, null),
        new o.FnParam('dirIndex', null),
    ], [
        renderFlagCheckIfStmt(1 /* core.RenderFlags.Create */, createStatements),
        renderFlagCheckIfStmt(2 /* core.RenderFlags.Update */, collapseAdvanceStatements(updateStatements)),
    ], o.INFERRED_TYPE, null, contentQueriesFnName);
}
//# sourceMappingURL=data:application/json;base64,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