/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { I18nContextKind, Namespace, OpKind, } from '../enums';
import { SlotHandle } from '../handle';
import { OpList } from '../operations';
import { TRAIT_CONSUMES_SLOT, TRAIT_CONSUMES_VARS, } from '../traits';
import { NEW_OP } from './shared';
/**
 * The set of OpKinds that represent the creation of an element or container
 */
const elementContainerOpKinds = new Set([
    OpKind.Element,
    OpKind.ElementStart,
    OpKind.Container,
    OpKind.ContainerStart,
    OpKind.Template,
    OpKind.RepeaterCreate,
]);
/**
 * Checks whether the given operation represents the creation of an element or container.
 */
export function isElementOrContainerOp(op) {
    return elementContainerOpKinds.has(op.kind);
}
/**
 * Create an `ElementStartOp`.
 */
export function createElementStartOp(tag, xref, namespace, i18nPlaceholder, startSourceSpan, wholeSourceSpan) {
    return {
        kind: OpKind.ElementStart,
        xref,
        tag,
        handle: new SlotHandle(),
        attributes: null,
        localRefs: [],
        nonBindable: false,
        namespace,
        i18nPlaceholder,
        startSourceSpan,
        wholeSourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
/**
 * Create a `TemplateOp`.
 */
export function createTemplateOp(xref, templateKind, tag, functionNameSuffix, namespace, i18nPlaceholder, startSourceSpan, wholeSourceSpan) {
    return {
        kind: OpKind.Template,
        xref,
        templateKind,
        attributes: null,
        tag,
        handle: new SlotHandle(),
        functionNameSuffix,
        decls: null,
        vars: null,
        localRefs: [],
        nonBindable: false,
        namespace,
        i18nPlaceholder,
        startSourceSpan,
        wholeSourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
export function createRepeaterCreateOp(primaryView, emptyView, tag, track, varNames, emptyTag, i18nPlaceholder, emptyI18nPlaceholder, startSourceSpan, wholeSourceSpan) {
    return {
        kind: OpKind.RepeaterCreate,
        attributes: null,
        xref: primaryView,
        handle: new SlotHandle(),
        emptyView,
        track,
        trackByFn: null,
        tag,
        emptyTag,
        emptyAttributes: null,
        functionNameSuffix: 'For',
        namespace: Namespace.HTML,
        nonBindable: false,
        localRefs: [],
        decls: null,
        vars: null,
        varNames,
        usesComponentInstance: false,
        i18nPlaceholder,
        emptyI18nPlaceholder,
        startSourceSpan,
        wholeSourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
        ...TRAIT_CONSUMES_VARS,
        numSlotsUsed: emptyView === null ? 2 : 3,
    };
}
/**
 * Create an `ElementEndOp`.
 */
export function createElementEndOp(xref, sourceSpan) {
    return {
        kind: OpKind.ElementEnd,
        xref,
        sourceSpan,
        ...NEW_OP,
    };
}
export function createDisableBindingsOp(xref) {
    return {
        kind: OpKind.DisableBindings,
        xref,
        ...NEW_OP,
    };
}
export function createEnableBindingsOp(xref) {
    return {
        kind: OpKind.EnableBindings,
        xref,
        ...NEW_OP,
    };
}
/**
 * Create a `TextOp`.
 */
export function createTextOp(xref, initialValue, icuPlaceholder, sourceSpan) {
    return {
        kind: OpKind.Text,
        xref,
        handle: new SlotHandle(),
        initialValue,
        icuPlaceholder,
        sourceSpan,
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
/**
 * Create a `ListenerOp`. Host bindings reuse all the listener logic.
 */
export function createListenerOp(target, targetSlot, name, tag, handlerOps, animationPhase, eventTarget, hostListener, sourceSpan) {
    const handlerList = new OpList();
    handlerList.push(handlerOps);
    return {
        kind: OpKind.Listener,
        target,
        targetSlot,
        tag,
        hostListener,
        name,
        handlerOps: handlerList,
        handlerFnName: null,
        consumesDollarEvent: false,
        isAnimationListener: animationPhase !== null,
        animationPhase,
        eventTarget,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Create a `TwoWayListenerOp`.
 */
export function createTwoWayListenerOp(target, targetSlot, name, tag, handlerOps, sourceSpan) {
    const handlerList = new OpList();
    handlerList.push(handlerOps);
    return {
        kind: OpKind.TwoWayListener,
        target,
        targetSlot,
        tag,
        name,
        handlerOps: handlerList,
        handlerFnName: null,
        sourceSpan,
        ...NEW_OP,
    };
}
export function createPipeOp(xref, slot, name) {
    return {
        kind: OpKind.Pipe,
        xref,
        handle: slot,
        name,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
    };
}
export function createNamespaceOp(namespace) {
    return {
        kind: OpKind.Namespace,
        active: namespace,
        ...NEW_OP,
    };
}
export function createProjectionDefOp(def) {
    return {
        kind: OpKind.ProjectionDef,
        def,
        ...NEW_OP,
    };
}
export function createProjectionOp(xref, selector, i18nPlaceholder, fallbackView, sourceSpan) {
    return {
        kind: OpKind.Projection,
        xref,
        handle: new SlotHandle(),
        selector,
        i18nPlaceholder,
        fallbackView,
        projectionSlotIndex: 0,
        attributes: null,
        localRefs: [],
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
        numSlotsUsed: fallbackView === null ? 1 : 2,
    };
}
/**
 * Create an `ExtractedAttributeOp`.
 */
export function createExtractedAttributeOp(target, bindingKind, namespace, name, expression, i18nContext, i18nMessage, securityContext) {
    return {
        kind: OpKind.ExtractedAttribute,
        target,
        bindingKind,
        namespace,
        name,
        expression,
        i18nContext,
        i18nMessage,
        securityContext,
        trustedValueFn: null,
        ...NEW_OP,
    };
}
export function createDeferOp(xref, main, mainSlot, ownResolverFn, resolverFn, sourceSpan) {
    return {
        kind: OpKind.Defer,
        xref,
        handle: new SlotHandle(),
        mainView: main,
        mainSlot,
        loadingView: null,
        loadingSlot: null,
        loadingConfig: null,
        loadingMinimumTime: null,
        loadingAfterTime: null,
        placeholderView: null,
        placeholderSlot: null,
        placeholderConfig: null,
        placeholderMinimumTime: null,
        errorView: null,
        errorSlot: null,
        ownResolverFn,
        resolverFn,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
        numSlotsUsed: 2,
    };
}
export function createDeferOnOp(defer, trigger, prefetch, sourceSpan) {
    return {
        kind: OpKind.DeferOn,
        defer,
        trigger,
        prefetch,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Creates a `DeclareLetOp`.
 */
export function createDeclareLetOp(xref, declaredName, sourceSpan) {
    return {
        kind: OpKind.DeclareLet,
        xref,
        declaredName,
        sourceSpan,
        handle: new SlotHandle(),
        ...TRAIT_CONSUMES_SLOT,
        ...NEW_OP,
    };
}
/**
 * Create an `ExtractedMessageOp`.
 */
export function createI18nMessageOp(xref, i18nContext, i18nBlock, message, messagePlaceholder, params, postprocessingParams, needsPostprocessing) {
    return {
        kind: OpKind.I18nMessage,
        xref,
        i18nContext,
        i18nBlock,
        message,
        messagePlaceholder,
        params,
        postprocessingParams,
        needsPostprocessing,
        subMessages: [],
        ...NEW_OP,
    };
}
/**
 * Create an `I18nStartOp`.
 */
export function createI18nStartOp(xref, message, root, sourceSpan) {
    return {
        kind: OpKind.I18nStart,
        xref,
        handle: new SlotHandle(),
        root: root ?? xref,
        message,
        messageIndex: null,
        subTemplateIndex: null,
        context: null,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
    };
}
/**
 * Create an `I18nEndOp`.
 */
export function createI18nEndOp(xref, sourceSpan) {
    return {
        kind: OpKind.I18nEnd,
        xref,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Creates an ICU start op.
 */
export function createIcuStartOp(xref, message, messagePlaceholder, sourceSpan) {
    return {
        kind: OpKind.IcuStart,
        xref,
        message,
        messagePlaceholder,
        context: null,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Creates an ICU end op.
 */
export function createIcuEndOp(xref) {
    return {
        kind: OpKind.IcuEnd,
        xref,
        ...NEW_OP,
    };
}
/**
 * Creates an ICU placeholder op.
 */
export function createIcuPlaceholderOp(xref, name, strings) {
    return {
        kind: OpKind.IcuPlaceholder,
        xref,
        name,
        strings,
        expressionPlaceholders: [],
        ...NEW_OP,
    };
}
export function createI18nContextOp(contextKind, xref, i18nBlock, message, sourceSpan) {
    if (i18nBlock === null && contextKind !== I18nContextKind.Attr) {
        throw new Error('AssertionError: i18nBlock must be provided for non-attribute contexts.');
    }
    return {
        kind: OpKind.I18nContext,
        contextKind,
        xref,
        i18nBlock,
        message,
        sourceSpan,
        params: new Map(),
        postprocessingParams: new Map(),
        ...NEW_OP,
    };
}
export function createI18nAttributesOp(xref, handle, target) {
    return {
        kind: OpKind.I18nAttributes,
        xref,
        handle,
        target,
        i18nAttributesConfig: null,
        ...NEW_OP,
        ...TRAIT_CONSUMES_SLOT,
    };
}
//# sourceMappingURL=data:application/json;base64,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