/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { OpKind, } from '../enums';
import { TRAIT_CONSUMES_VARS, TRAIT_DEPENDS_ON_SLOT_CONTEXT, } from '../traits';
import { NEW_OP } from './shared';
/**
 * Create an `InterpolationTextOp`.
 */
export function createInterpolateTextOp(xref, interpolation, sourceSpan) {
    return {
        kind: OpKind.InterpolateText,
        target: xref,
        interpolation,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
export class Interpolation {
    constructor(strings, expressions, i18nPlaceholders) {
        this.strings = strings;
        this.expressions = expressions;
        this.i18nPlaceholders = i18nPlaceholders;
        if (i18nPlaceholders.length !== 0 && i18nPlaceholders.length !== expressions.length) {
            throw new Error(`Expected ${expressions.length} placeholders to match interpolation expression count, but got ${i18nPlaceholders.length}`);
        }
    }
}
/**
 * Create a `BindingOp`, not yet transformed into a particular type of binding.
 */
export function createBindingOp(target, kind, name, expression, unit, securityContext, isTextAttribute, isStructuralTemplateAttribute, templateKind, i18nMessage, sourceSpan) {
    return {
        kind: OpKind.Binding,
        bindingKind: kind,
        target,
        name,
        expression,
        unit,
        securityContext,
        isTextAttribute,
        isStructuralTemplateAttribute,
        templateKind,
        i18nContext: null,
        i18nMessage,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Create a `PropertyOp`.
 */
export function createPropertyOp(target, name, expression, isAnimationTrigger, securityContext, isStructuralTemplateAttribute, templateKind, i18nContext, i18nMessage, sourceSpan) {
    return {
        kind: OpKind.Property,
        target,
        name,
        expression,
        isAnimationTrigger,
        securityContext,
        sanitizer: null,
        isStructuralTemplateAttribute,
        templateKind,
        i18nContext,
        i18nMessage,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create a `TwoWayPropertyOp`.
 */
export function createTwoWayPropertyOp(target, name, expression, securityContext, isStructuralTemplateAttribute, templateKind, i18nContext, i18nMessage, sourceSpan) {
    return {
        kind: OpKind.TwoWayProperty,
        target,
        name,
        expression,
        securityContext,
        sanitizer: null,
        isStructuralTemplateAttribute,
        templateKind,
        i18nContext,
        i18nMessage,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/** Create a `StylePropOp`. */
export function createStylePropOp(xref, name, expression, unit, sourceSpan) {
    return {
        kind: OpKind.StyleProp,
        target: xref,
        name,
        expression,
        unit,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create a `ClassPropOp`.
 */
export function createClassPropOp(xref, name, expression, sourceSpan) {
    return {
        kind: OpKind.ClassProp,
        target: xref,
        name,
        expression,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/** Create a `StyleMapOp`. */
export function createStyleMapOp(xref, expression, sourceSpan) {
    return {
        kind: OpKind.StyleMap,
        target: xref,
        expression,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create a `ClassMapOp`.
 */
export function createClassMapOp(xref, expression, sourceSpan) {
    return {
        kind: OpKind.ClassMap,
        target: xref,
        expression,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create an `AttributeOp`.
 */
export function createAttributeOp(target, namespace, name, expression, securityContext, isTextAttribute, isStructuralTemplateAttribute, templateKind, i18nMessage, sourceSpan) {
    return {
        kind: OpKind.Attribute,
        target,
        namespace,
        name,
        expression,
        securityContext,
        sanitizer: null,
        isTextAttribute,
        isStructuralTemplateAttribute,
        templateKind,
        i18nContext: null,
        i18nMessage,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
/**
 * Create an `AdvanceOp`.
 */
export function createAdvanceOp(delta, sourceSpan) {
    return {
        kind: OpKind.Advance,
        delta,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Create a conditional op, which will display an embedded view according to a condtion.
 */
export function createConditionalOp(target, test, conditions, sourceSpan) {
    return {
        kind: OpKind.Conditional,
        target,
        test,
        conditions,
        processed: null,
        sourceSpan,
        contextValue: null,
        ...NEW_OP,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
    };
}
export function createRepeaterOp(repeaterCreate, targetSlot, collection, sourceSpan) {
    return {
        kind: OpKind.Repeater,
        target: repeaterCreate,
        targetSlot,
        collection,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
    };
}
export function createDeferWhenOp(target, expr, prefetch, sourceSpan) {
    return {
        kind: OpKind.DeferWhen,
        target,
        expr,
        prefetch,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
    };
}
/**
 * Create an i18n expression op.
 */
export function createI18nExpressionOp(context, target, i18nOwner, handle, expression, icuPlaceholder, i18nPlaceholder, resolutionTime, usage, name, sourceSpan) {
    return {
        kind: OpKind.I18nExpression,
        context,
        target,
        i18nOwner,
        handle,
        expression,
        icuPlaceholder,
        i18nPlaceholder,
        resolutionTime,
        usage,
        name,
        sourceSpan,
        ...NEW_OP,
        ...TRAIT_CONSUMES_VARS,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
    };
}
/**
 * Creates an op to apply i18n expression ops.
 */
export function createI18nApplyOp(owner, handle, sourceSpan) {
    return {
        kind: OpKind.I18nApply,
        owner,
        handle,
        sourceSpan,
        ...NEW_OP,
    };
}
/**
 * Creates a `StoreLetOp`.
 */
export function createStoreLetOp(target, declaredName, value, sourceSpan) {
    return {
        kind: OpKind.StoreLet,
        target,
        declaredName,
        value,
        sourceSpan,
        ...TRAIT_DEPENDS_ON_SLOT_CONTEXT,
        ...TRAIT_CONSUMES_VARS,
        ...NEW_OP,
    };
}
//# sourceMappingURL=data:application/json;base64,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