/**
 *
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../../../../src/output/output_ast';
import * as ir from '../ir';
import { CompilationJobKind as Kind, } from './compilation';
import { deleteAnyCasts } from './phases/any_cast';
import { applyI18nExpressions } from './phases/apply_i18n_expressions';
import { assignI18nSlotDependencies } from './phases/assign_i18n_slot_dependencies';
import { extractAttributes } from './phases/attribute_extraction';
import { specializeBindings } from './phases/binding_specialization';
import { chain } from './phases/chaining';
import { collapseSingletonInterpolations } from './phases/collapse_singleton_interpolations';
import { generateConditionalExpressions } from './phases/conditionals';
import { collectElementConsts } from './phases/const_collection';
import { convertI18nBindings } from './phases/convert_i18n_bindings';
import { resolveDeferDepsFns } from './phases/resolve_defer_deps_fns';
import { createI18nContexts } from './phases/create_i18n_contexts';
import { deduplicateTextBindings } from './phases/deduplicate_text_bindings';
import { configureDeferInstructions } from './phases/defer_configs';
import { resolveDeferTargetNames } from './phases/defer_resolve_targets';
import { collapseEmptyInstructions } from './phases/empty_elements';
import { expandSafeReads } from './phases/expand_safe_reads';
import { extractI18nMessages } from './phases/extract_i18n_messages';
import { generateAdvance } from './phases/generate_advance';
import { generateProjectionDefs } from './phases/generate_projection_def';
import { generateVariables } from './phases/generate_variables';
import { collectConstExpressions } from './phases/has_const_expression_collection';
import { parseHostStyleProperties } from './phases/host_style_property_parsing';
import { collectI18nConsts } from './phases/i18n_const_collection';
import { convertI18nText } from './phases/i18n_text_extraction';
import { liftLocalRefs } from './phases/local_refs';
import { emitNamespaceChanges } from './phases/namespace';
import { nameFunctionsAndVariables } from './phases/naming';
import { mergeNextContextExpressions } from './phases/next_context_merging';
import { generateNgContainerOps } from './phases/ng_container';
import { disableBindings } from './phases/nonbindable';
import { generateNullishCoalesceExpressions } from './phases/nullish_coalescing';
import { orderOps } from './phases/ordering';
import { parseExtractedStyles } from './phases/parse_extracted_styles';
import { removeContentSelectors } from './phases/phase_remove_content_selectors';
import { createPipes } from './phases/pipe_creation';
import { createVariadicPipes } from './phases/pipe_variadic';
import { propagateI18nBlocks } from './phases/propagate_i18n_blocks';
import { extractPureFunctions } from './phases/pure_function_extraction';
import { generatePureLiteralStructures } from './phases/pure_literal_structures';
import { reify } from './phases/reify';
import { removeEmptyBindings } from './phases/remove_empty_bindings';
import { removeI18nContexts } from './phases/remove_i18n_contexts';
import { removeUnusedI18nAttributesOps } from './phases/remove_unused_i18n_attrs';
import { resolveContexts } from './phases/resolve_contexts';
import { resolveDollarEvent } from './phases/resolve_dollar_event';
import { resolveI18nElementPlaceholders } from './phases/resolve_i18n_element_placeholders';
import { resolveI18nExpressionPlaceholders } from './phases/resolve_i18n_expression_placeholders';
import { resolveNames } from './phases/resolve_names';
import { resolveSanitizers } from './phases/resolve_sanitizers';
import { transformTwoWayBindingSet } from './phases/transform_two_way_binding_set';
import { saveAndRestoreView } from './phases/save_restore_view';
import { allocateSlots } from './phases/slot_allocation';
import { specializeStyleBindings } from './phases/style_binding_specialization';
import { generateTemporaryVariables } from './phases/temporary_variables';
import { generateTrackFns } from './phases/track_fn_generation';
import { optimizeTrackFns } from './phases/track_fn_optimization';
import { generateTrackVariables } from './phases/track_variables';
import { countVariables } from './phases/var_counting';
import { optimizeVariables } from './phases/variable_optimization';
import { wrapI18nIcus } from './phases/wrap_icus';
import { optimizeStoreLet } from './phases/store_let_optimization';
import { removeIllegalLetReferences } from './phases/remove_illegal_let_references';
import { generateLocalLetReferences } from './phases/generate_local_let_references';
const phases = [
    { kind: Kind.Tmpl, fn: removeContentSelectors },
    { kind: Kind.Host, fn: parseHostStyleProperties },
    { kind: Kind.Tmpl, fn: emitNamespaceChanges },
    { kind: Kind.Tmpl, fn: propagateI18nBlocks },
    { kind: Kind.Tmpl, fn: wrapI18nIcus },
    { kind: Kind.Both, fn: deduplicateTextBindings },
    { kind: Kind.Both, fn: specializeStyleBindings },
    { kind: Kind.Both, fn: specializeBindings },
    { kind: Kind.Both, fn: extractAttributes },
    { kind: Kind.Tmpl, fn: createI18nContexts },
    { kind: Kind.Both, fn: parseExtractedStyles },
    { kind: Kind.Tmpl, fn: removeEmptyBindings },
    { kind: Kind.Both, fn: collapseSingletonInterpolations },
    { kind: Kind.Both, fn: orderOps },
    { kind: Kind.Tmpl, fn: generateConditionalExpressions },
    { kind: Kind.Tmpl, fn: createPipes },
    { kind: Kind.Tmpl, fn: configureDeferInstructions },
    { kind: Kind.Tmpl, fn: convertI18nText },
    { kind: Kind.Tmpl, fn: convertI18nBindings },
    { kind: Kind.Tmpl, fn: removeUnusedI18nAttributesOps },
    { kind: Kind.Tmpl, fn: assignI18nSlotDependencies },
    { kind: Kind.Tmpl, fn: applyI18nExpressions },
    { kind: Kind.Tmpl, fn: createVariadicPipes },
    { kind: Kind.Both, fn: generatePureLiteralStructures },
    { kind: Kind.Tmpl, fn: generateProjectionDefs },
    { kind: Kind.Tmpl, fn: generateLocalLetReferences },
    { kind: Kind.Tmpl, fn: generateVariables },
    { kind: Kind.Tmpl, fn: saveAndRestoreView },
    { kind: Kind.Both, fn: deleteAnyCasts },
    { kind: Kind.Both, fn: resolveDollarEvent },
    { kind: Kind.Tmpl, fn: generateTrackVariables },
    { kind: Kind.Tmpl, fn: removeIllegalLetReferences },
    { kind: Kind.Both, fn: resolveNames },
    { kind: Kind.Tmpl, fn: resolveDeferTargetNames },
    { kind: Kind.Tmpl, fn: transformTwoWayBindingSet },
    { kind: Kind.Tmpl, fn: optimizeTrackFns },
    { kind: Kind.Both, fn: resolveContexts },
    { kind: Kind.Both, fn: resolveSanitizers },
    { kind: Kind.Tmpl, fn: liftLocalRefs },
    { kind: Kind.Both, fn: generateNullishCoalesceExpressions },
    { kind: Kind.Both, fn: expandSafeReads },
    { kind: Kind.Both, fn: generateTemporaryVariables },
    { kind: Kind.Both, fn: optimizeVariables },
    { kind: Kind.Both, fn: optimizeStoreLet },
    { kind: Kind.Tmpl, fn: allocateSlots },
    { kind: Kind.Tmpl, fn: resolveI18nElementPlaceholders },
    { kind: Kind.Tmpl, fn: resolveI18nExpressionPlaceholders },
    { kind: Kind.Tmpl, fn: extractI18nMessages },
    { kind: Kind.Tmpl, fn: generateTrackFns },
    { kind: Kind.Tmpl, fn: collectI18nConsts },
    { kind: Kind.Tmpl, fn: collectConstExpressions },
    { kind: Kind.Both, fn: collectElementConsts },
    { kind: Kind.Tmpl, fn: removeI18nContexts },
    { kind: Kind.Both, fn: countVariables },
    { kind: Kind.Tmpl, fn: generateAdvance },
    { kind: Kind.Both, fn: nameFunctionsAndVariables },
    { kind: Kind.Tmpl, fn: resolveDeferDepsFns },
    { kind: Kind.Tmpl, fn: mergeNextContextExpressions },
    { kind: Kind.Tmpl, fn: generateNgContainerOps },
    { kind: Kind.Tmpl, fn: collapseEmptyInstructions },
    { kind: Kind.Tmpl, fn: disableBindings },
    { kind: Kind.Both, fn: extractPureFunctions },
    { kind: Kind.Both, fn: reify },
    { kind: Kind.Both, fn: chain },
];
/**
 * Run all transformation phases in the correct order against a compilation job. After this
 * processing, the compilation should be in a state where it can be emitted.
 */
export function transform(job, kind) {
    for (const phase of phases) {
        if (phase.kind === kind || phase.kind === Kind.Both) {
            // The type of `Phase` above ensures it is impossible to call a phase that doesn't support the
            // job kind.
            phase.fn(job);
        }
    }
}
/**
 * Compile all views in the given `ComponentCompilation` into the final template function, which may
 * reference constants defined in a `ConstantPool`.
 */
export function emitTemplateFn(tpl, pool) {
    const rootFn = emitView(tpl.root);
    emitChildViews(tpl.root, pool);
    return rootFn;
}
function emitChildViews(parent, pool) {
    for (const unit of parent.job.units) {
        if (unit.parent !== parent.xref) {
            continue;
        }
        // Child views are emitted depth-first.
        emitChildViews(unit, pool);
        const viewFn = emitView(unit);
        pool.statements.push(viewFn.toDeclStmt(viewFn.name));
    }
}
/**
 * Emit a template function for an individual `ViewCompilation` (which may be either the root view
 * or an embedded view).
 */
function emitView(view) {
    if (view.fnName === null) {
        throw new Error(`AssertionError: view ${view.xref} is unnamed`);
    }
    const createStatements = [];
    for (const op of view.create) {
        if (op.kind !== ir.OpKind.Statement) {
            throw new Error(`AssertionError: expected all create ops to have been compiled, but got ${ir.OpKind[op.kind]}`);
        }
        createStatements.push(op.statement);
    }
    const updateStatements = [];
    for (const op of view.update) {
        if (op.kind !== ir.OpKind.Statement) {
            throw new Error(`AssertionError: expected all update ops to have been compiled, but got ${ir.OpKind[op.kind]}`);
        }
        updateStatements.push(op.statement);
    }
    const createCond = maybeGenerateRfBlock(1, createStatements);
    const updateCond = maybeGenerateRfBlock(2, updateStatements);
    return o.fn([new o.FnParam('rf'), new o.FnParam('ctx')], [...createCond, ...updateCond], 
    /* type */ undefined, 
    /* sourceSpan */ undefined, view.fnName);
}
function maybeGenerateRfBlock(flag, statements) {
    if (statements.length === 0) {
        return [];
    }
    return [
        o.ifStmt(new o.BinaryOperatorExpr(o.BinaryOperator.BitwiseAnd, o.variable('rf'), o.literal(flag)), statements),
    ];
}
export function emitHostBindingFunction(job) {
    if (job.root.fnName === null) {
        throw new Error(`AssertionError: host binding function is unnamed`);
    }
    const createStatements = [];
    for (const op of job.root.create) {
        if (op.kind !== ir.OpKind.Statement) {
            throw new Error(`AssertionError: expected all create ops to have been compiled, but got ${ir.OpKind[op.kind]}`);
        }
        createStatements.push(op.statement);
    }
    const updateStatements = [];
    for (const op of job.root.update) {
        if (op.kind !== ir.OpKind.Statement) {
            throw new Error(`AssertionError: expected all update ops to have been compiled, but got ${ir.OpKind[op.kind]}`);
        }
        updateStatements.push(op.statement);
    }
    if (createStatements.length === 0 && updateStatements.length === 0) {
        return null;
    }
    const createCond = maybeGenerateRfBlock(1, createStatements);
    const updateCond = maybeGenerateRfBlock(2, updateStatements);
    return o.fn([new o.FnParam('rf'), new o.FnParam('ctx')], [...createCond, ...updateCond], 
    /* type */ undefined, 
    /* sourceSpan */ undefined, job.root.fnName);
}
//# sourceMappingURL=data:application/json;base64,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