/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { splitNsName } from '../../../../ml_parser/tags';
import * as o from '../../../../output/output_ast';
import * as ir from '../../ir';
import { CompilationJobKind } from '../compilation';
/**
 * Looks up an element in the given map by xref ID.
 */
function lookupElement(elements, xref) {
    const el = elements.get(xref);
    if (el === undefined) {
        throw new Error('All attributes should have an element-like target.');
    }
    return el;
}
export function specializeBindings(job) {
    const elements = new Map();
    for (const unit of job.units) {
        for (const op of unit.create) {
            if (!ir.isElementOrContainerOp(op)) {
                continue;
            }
            elements.set(op.xref, op);
        }
    }
    for (const unit of job.units) {
        for (const op of unit.ops()) {
            if (op.kind !== ir.OpKind.Binding) {
                continue;
            }
            switch (op.bindingKind) {
                case ir.BindingKind.Attribute:
                    if (op.name === 'ngNonBindable') {
                        ir.OpList.remove(op);
                        const target = lookupElement(elements, op.target);
                        target.nonBindable = true;
                    }
                    else {
                        const [namespace, name] = splitNsName(op.name);
                        ir.OpList.replace(op, ir.createAttributeOp(op.target, namespace, name, op.expression, op.securityContext, op.isTextAttribute, op.isStructuralTemplateAttribute, op.templateKind, op.i18nMessage, op.sourceSpan));
                    }
                    break;
                case ir.BindingKind.Property:
                case ir.BindingKind.Animation:
                    if (job.kind === CompilationJobKind.Host) {
                        ir.OpList.replace(op, ir.createHostPropertyOp(op.name, op.expression, op.bindingKind === ir.BindingKind.Animation, op.i18nContext, op.securityContext, op.sourceSpan));
                    }
                    else {
                        ir.OpList.replace(op, ir.createPropertyOp(op.target, op.name, op.expression, op.bindingKind === ir.BindingKind.Animation, op.securityContext, op.isStructuralTemplateAttribute, op.templateKind, op.i18nContext, op.i18nMessage, op.sourceSpan));
                    }
                    break;
                case ir.BindingKind.TwoWayProperty:
                    if (!(op.expression instanceof o.Expression)) {
                        // We shouldn't be able to hit this code path since interpolations in two-way bindings
                        // result in a parser error. We assert here so that downstream we can assume that
                        // the value is always an expression.
                        throw new Error(`Expected value of two-way property binding "${op.name}" to be an expression`);
                    }
                    ir.OpList.replace(op, ir.createTwoWayPropertyOp(op.target, op.name, op.expression, op.securityContext, op.isStructuralTemplateAttribute, op.templateKind, op.i18nContext, op.i18nMessage, op.sourceSpan));
                    break;
                case ir.BindingKind.I18n:
                case ir.BindingKind.ClassName:
                case ir.BindingKind.StyleProperty:
                    throw new Error(`Unhandled binding of kind ${ir.BindingKind[op.bindingKind]}`);
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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