/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../../../../output/output_ast';
import * as ir from '../../ir';
/**
 * Safe read expressions such as `a?.b` have different semantics in Angular templates as
 * compared to JavaScript. In particular, they default to `null` instead of `undefined`. This phase
 * finds all unresolved safe read expressions, and converts them into the appropriate output AST
 * reads, guarded by null checks. We generate temporaries as needed, to avoid re-evaluating the same
 * sub-expression multiple times.
 */
export function expandSafeReads(job) {
    for (const unit of job.units) {
        for (const op of unit.ops()) {
            ir.transformExpressionsInOp(op, (e) => safeTransform(e, { job }), ir.VisitorContextFlag.None);
            ir.transformExpressionsInOp(op, ternaryTransform, ir.VisitorContextFlag.None);
        }
    }
}
// A lookup set of all the expression kinds that require a temporary variable to be generated.
const requiresTemporary = [
    o.InvokeFunctionExpr,
    o.LiteralArrayExpr,
    o.LiteralMapExpr,
    ir.SafeInvokeFunctionExpr,
    ir.PipeBindingExpr,
].map((e) => e.constructor.name);
function needsTemporaryInSafeAccess(e) {
    // TODO: We probably want to use an expression visitor to recursively visit all descendents.
    // However, that would potentially do a lot of extra work (because it cannot short circuit), so we
    // implement the logic ourselves for now.
    if (e instanceof o.UnaryOperatorExpr) {
        return needsTemporaryInSafeAccess(e.expr);
    }
    else if (e instanceof o.BinaryOperatorExpr) {
        return needsTemporaryInSafeAccess(e.lhs) || needsTemporaryInSafeAccess(e.rhs);
    }
    else if (e instanceof o.ConditionalExpr) {
        if (e.falseCase && needsTemporaryInSafeAccess(e.falseCase))
            return true;
        return needsTemporaryInSafeAccess(e.condition) || needsTemporaryInSafeAccess(e.trueCase);
    }
    else if (e instanceof o.NotExpr) {
        return needsTemporaryInSafeAccess(e.condition);
    }
    else if (e instanceof ir.AssignTemporaryExpr) {
        return needsTemporaryInSafeAccess(e.expr);
    }
    else if (e instanceof o.ReadPropExpr) {
        return needsTemporaryInSafeAccess(e.receiver);
    }
    else if (e instanceof o.ReadKeyExpr) {
        return needsTemporaryInSafeAccess(e.receiver) || needsTemporaryInSafeAccess(e.index);
    }
    // TODO: Switch to a method which is exhaustive of newly added expression subtypes.
    return (e instanceof o.InvokeFunctionExpr ||
        e instanceof o.LiteralArrayExpr ||
        e instanceof o.LiteralMapExpr ||
        e instanceof ir.SafeInvokeFunctionExpr ||
        e instanceof ir.PipeBindingExpr);
}
function temporariesIn(e) {
    const temporaries = new Set();
    // TODO: Although it's not currently supported by the transform helper, we should be able to
    // short-circuit exploring the tree to do less work. In particular, we don't have to penetrate
    // into the subexpressions of temporary assignments.
    ir.transformExpressionsInExpression(e, (e) => {
        if (e instanceof ir.AssignTemporaryExpr) {
            temporaries.add(e.xref);
        }
        return e;
    }, ir.VisitorContextFlag.None);
    return temporaries;
}
function eliminateTemporaryAssignments(e, tmps, ctx) {
    // TODO: We can be more efficient than the transform helper here. We don't need to visit any
    // descendents of temporary assignments.
    ir.transformExpressionsInExpression(e, (e) => {
        if (e instanceof ir.AssignTemporaryExpr && tmps.has(e.xref)) {
            const read = new ir.ReadTemporaryExpr(e.xref);
            // `TemplateDefinitionBuilder` has the (accidental?) behavior of generating assignments of
            // temporary variables to themselves. This happens because some subexpression that the
            // temporary refers to, possibly through nested temporaries, has a function call. We copy that
            // behavior here.
            return ctx.job.compatibility === ir.CompatibilityMode.TemplateDefinitionBuilder
                ? new ir.AssignTemporaryExpr(read, read.xref)
                : read;
        }
        return e;
    }, ir.VisitorContextFlag.None);
    return e;
}
/**
 * Creates a safe ternary guarded by the input expression, and with a body generated by the provided
 * callback on the input expression. Generates a temporary variable assignment if needed, and
 * deduplicates nested temporary assignments if needed.
 */
function safeTernaryWithTemporary(guard, body, ctx) {
    let result;
    if (needsTemporaryInSafeAccess(guard)) {
        const xref = ctx.job.allocateXrefId();
        result = [new ir.AssignTemporaryExpr(guard, xref), new ir.ReadTemporaryExpr(xref)];
    }
    else {
        result = [guard, guard.clone()];
        // Consider an expression like `a?.[b?.c()]?.d`. The `b?.c()` will be transformed first,
        // introducing a temporary assignment into the key. Then, as part of expanding the `?.d`. That
        // assignment will be duplicated into both the guard and expression sides. We de-duplicate it,
        // by transforming it from an assignment into a read on the expression side.
        eliminateTemporaryAssignments(result[1], temporariesIn(result[0]), ctx);
    }
    return new ir.SafeTernaryExpr(result[0], body(result[1]));
}
function isSafeAccessExpression(e) {
    return (e instanceof ir.SafePropertyReadExpr ||
        e instanceof ir.SafeKeyedReadExpr ||
        e instanceof ir.SafeInvokeFunctionExpr);
}
function isUnsafeAccessExpression(e) {
    return (e instanceof o.ReadPropExpr || e instanceof o.ReadKeyExpr || e instanceof o.InvokeFunctionExpr);
}
function isAccessExpression(e) {
    return isSafeAccessExpression(e) || isUnsafeAccessExpression(e);
}
function deepestSafeTernary(e) {
    if (isAccessExpression(e) && e.receiver instanceof ir.SafeTernaryExpr) {
        let st = e.receiver;
        while (st.expr instanceof ir.SafeTernaryExpr) {
            st = st.expr;
        }
        return st;
    }
    return null;
}
// TODO: When strict compatibility with TemplateDefinitionBuilder is not required, we can use `&&`
// instead to save some code size.
function safeTransform(e, ctx) {
    if (!isAccessExpression(e)) {
        return e;
    }
    const dst = deepestSafeTernary(e);
    if (dst) {
        if (e instanceof o.InvokeFunctionExpr) {
            dst.expr = dst.expr.callFn(e.args);
            return e.receiver;
        }
        if (e instanceof o.ReadPropExpr) {
            dst.expr = dst.expr.prop(e.name);
            return e.receiver;
        }
        if (e instanceof o.ReadKeyExpr) {
            dst.expr = dst.expr.key(e.index);
            return e.receiver;
        }
        if (e instanceof ir.SafeInvokeFunctionExpr) {
            dst.expr = safeTernaryWithTemporary(dst.expr, (r) => r.callFn(e.args), ctx);
            return e.receiver;
        }
        if (e instanceof ir.SafePropertyReadExpr) {
            dst.expr = safeTernaryWithTemporary(dst.expr, (r) => r.prop(e.name), ctx);
            return e.receiver;
        }
        if (e instanceof ir.SafeKeyedReadExpr) {
            dst.expr = safeTernaryWithTemporary(dst.expr, (r) => r.key(e.index), ctx);
            return e.receiver;
        }
    }
    else {
        if (e instanceof ir.SafeInvokeFunctionExpr) {
            return safeTernaryWithTemporary(e.receiver, (r) => r.callFn(e.args), ctx);
        }
        if (e instanceof ir.SafePropertyReadExpr) {
            return safeTernaryWithTemporary(e.receiver, (r) => r.prop(e.name), ctx);
        }
        if (e instanceof ir.SafeKeyedReadExpr) {
            return safeTernaryWithTemporary(e.receiver, (r) => r.key(e.index), ctx);
        }
    }
    return e;
}
function ternaryTransform(e) {
    if (!(e instanceof ir.SafeTernaryExpr)) {
        return e;
    }
    return new o.ConditionalExpr(new o.BinaryOperatorExpr(o.BinaryOperator.Equals, e.guard, o.NULL_EXPR), o.NULL_EXPR, e.expr);
}
//# sourceMappingURL=data:application/json;base64,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