/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as ir from '../../ir';
import { CompilationJobKind } from '../compilation';
function kindTest(kind) {
    return (op) => op.kind === kind;
}
function kindWithInterpolationTest(kind, interpolation) {
    return (op) => {
        return op.kind === kind && interpolation === op.expression instanceof ir.Interpolation;
    };
}
function basicListenerKindTest(op) {
    return ((op.kind === ir.OpKind.Listener && !(op.hostListener && op.isAnimationListener)) ||
        op.kind === ir.OpKind.TwoWayListener);
}
function nonInterpolationPropertyKindTest(op) {
    return ((op.kind === ir.OpKind.Property || op.kind === ir.OpKind.TwoWayProperty) &&
        !(op.expression instanceof ir.Interpolation));
}
/**
 * Defines the groups based on `OpKind` that ops will be divided into, for the various create
 * op kinds. Ops will be collected into groups, then optionally transformed, before recombining
 * the groups in the order defined here.
 */
const CREATE_ORDERING = [
    { test: (op) => op.kind === ir.OpKind.Listener && op.hostListener && op.isAnimationListener },
    { test: basicListenerKindTest },
];
/**
 * Defines the groups based on `OpKind` that ops will be divided into, for the various update
 * op kinds.
 */
const UPDATE_ORDERING = [
    { test: kindTest(ir.OpKind.StyleMap), transform: keepLast },
    { test: kindTest(ir.OpKind.ClassMap), transform: keepLast },
    { test: kindTest(ir.OpKind.StyleProp) },
    { test: kindTest(ir.OpKind.ClassProp) },
    { test: kindWithInterpolationTest(ir.OpKind.Attribute, true) },
    { test: kindWithInterpolationTest(ir.OpKind.Property, true) },
    { test: nonInterpolationPropertyKindTest },
    { test: kindWithInterpolationTest(ir.OpKind.Attribute, false) },
];
/**
 * Host bindings have their own update ordering.
 */
const UPDATE_HOST_ORDERING = [
    { test: kindWithInterpolationTest(ir.OpKind.HostProperty, true) },
    { test: kindWithInterpolationTest(ir.OpKind.HostProperty, false) },
    { test: kindTest(ir.OpKind.Attribute) },
    { test: kindTest(ir.OpKind.StyleMap), transform: keepLast },
    { test: kindTest(ir.OpKind.ClassMap), transform: keepLast },
    { test: kindTest(ir.OpKind.StyleProp) },
    { test: kindTest(ir.OpKind.ClassProp) },
];
/**
 * The set of all op kinds we handle in the reordering phase.
 */
const handledOpKinds = new Set([
    ir.OpKind.Listener,
    ir.OpKind.TwoWayListener,
    ir.OpKind.StyleMap,
    ir.OpKind.ClassMap,
    ir.OpKind.StyleProp,
    ir.OpKind.ClassProp,
    ir.OpKind.Property,
    ir.OpKind.TwoWayProperty,
    ir.OpKind.HostProperty,
    ir.OpKind.Attribute,
]);
/**
 * Many type of operations have ordering constraints that must be respected. For example, a
 * `ClassMap` instruction must be ordered after a `StyleMap` instruction, in order to have
 * predictable semantics that match TemplateDefinitionBuilder and don't break applications.
 */
export function orderOps(job) {
    for (const unit of job.units) {
        // First, we pull out ops that need to be ordered. Then, when we encounter an op that shouldn't
        // be reordered, put the ones we've pulled so far back in the correct order. Finally, if we
        // still have ops pulled at the end, put them back in the correct order.
        // Create mode:
        orderWithin(unit.create, CREATE_ORDERING);
        // Update mode:
        const ordering = unit.job.kind === CompilationJobKind.Host ? UPDATE_HOST_ORDERING : UPDATE_ORDERING;
        orderWithin(unit.update, ordering);
    }
}
/**
 * Order all the ops within the specified group.
 */
function orderWithin(opList, ordering) {
    let opsToOrder = [];
    // Only reorder ops that target the same xref; do not mix ops that target different xrefs.
    let firstTargetInGroup = null;
    for (const op of opList) {
        const currentTarget = ir.hasDependsOnSlotContextTrait(op) ? op.target : null;
        if (!handledOpKinds.has(op.kind) ||
            (currentTarget !== firstTargetInGroup &&
                firstTargetInGroup !== null &&
                currentTarget !== null)) {
            ir.OpList.insertBefore(reorder(opsToOrder, ordering), op);
            opsToOrder = [];
            firstTargetInGroup = null;
        }
        if (handledOpKinds.has(op.kind)) {
            opsToOrder.push(op);
            ir.OpList.remove(op);
            firstTargetInGroup = currentTarget ?? firstTargetInGroup;
        }
    }
    opList.push(reorder(opsToOrder, ordering));
}
/**
 * Reorders the given list of ops according to the ordering defined by `ORDERING`.
 */
function reorder(ops, ordering) {
    // Break the ops list into groups based on OpKind.
    const groups = Array.from(ordering, () => new Array());
    for (const op of ops) {
        const groupIndex = ordering.findIndex((o) => o.test(op));
        groups[groupIndex].push(op);
    }
    // Reassemble the groups into a single list, in the correct order.
    return groups.flatMap((group, i) => {
        const transform = ordering[i].transform;
        return transform ? transform(group) : group;
    });
}
/**
 * Keeps only the last op in a list of ops.
 */
function keepLast(ops) {
    return ops.slice(ops.length - 1);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3JkZXJpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi8uLi8uLi8uLi8uLi9wYWNrYWdlcy9jb21waWxlci9zcmMvdGVtcGxhdGUvcGlwZWxpbmUvc3JjL3BoYXNlcy9vcmRlcmluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7O0dBTUc7QUFFSCxPQUFPLEtBQUssRUFBRSxNQUFNLFVBQVUsQ0FBQztBQUMvQixPQUFPLEVBQUMsa0JBQWtCLEVBQXNCLE1BQU0sZ0JBQWdCLENBQUM7QUFFdkUsU0FBUyxRQUFRLENBQUMsSUFBZTtJQUMvQixPQUFPLENBQUMsRUFBZSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUMsSUFBSSxLQUFLLElBQUksQ0FBQztBQUMvQyxDQUFDO0FBRUQsU0FBUyx5QkFBeUIsQ0FDaEMsSUFBdUUsRUFDdkUsYUFBc0I7SUFFdEIsT0FBTyxDQUFDLEVBQWUsRUFBRSxFQUFFO1FBQ3pCLE9BQU8sRUFBRSxDQUFDLElBQUksS0FBSyxJQUFJLElBQUksYUFBYSxLQUFLLEVBQUUsQ0FBQyxVQUFVLFlBQVksRUFBRSxDQUFDLGFBQWEsQ0FBQztJQUN6RixDQUFDLENBQUM7QUFDSixDQUFDO0FBRUQsU0FBUyxxQkFBcUIsQ0FBQyxFQUFlO0lBQzVDLE9BQU8sQ0FDTCxDQUFDLEVBQUUsQ0FBQyxJQUFJLEtBQUssRUFBRSxDQUFDLE1BQU0sQ0FBQyxRQUFRLElBQUksQ0FBQyxDQUFDLEVBQUUsQ0FBQyxZQUFZLElBQUksRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDaEYsRUFBRSxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FDckMsQ0FBQztBQUNKLENBQUM7QUFFRCxTQUFTLGdDQUFnQyxDQUFDLEVBQWU7SUFDdkQsT0FBTyxDQUNMLENBQUMsRUFBRSxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsTUFBTSxDQUFDLFFBQVEsSUFBSSxFQUFFLENBQUMsSUFBSSxLQUFLLEVBQUUsQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDO1FBQ3hFLENBQUMsQ0FBQyxFQUFFLENBQUMsVUFBVSxZQUFZLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FDN0MsQ0FBQztBQUNKLENBQUM7QUFPRDs7OztHQUlHO0FBQ0gsTUFBTSxlQUFlLEdBQTZCO0lBQ2hELEVBQUMsSUFBSSxFQUFFLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUMsSUFBSSxLQUFLLEVBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUSxJQUFJLEVBQUUsQ0FBQyxZQUFZLElBQUksRUFBRSxDQUFDLG1CQUFtQixFQUFDO0lBQzNGLEVBQUMsSUFBSSxFQUFFLHFCQUFxQixFQUFDO0NBQzlCLENBQUM7QUFFRjs7O0dBR0c7QUFDSCxNQUFNLGVBQWUsR0FBNkI7SUFDaEQsRUFBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLEVBQUUsU0FBUyxFQUFFLFFBQVEsRUFBQztJQUN6RCxFQUFDLElBQUksRUFBRSxRQUFRLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsRUFBRSxTQUFTLEVBQUUsUUFBUSxFQUFDO0lBQ3pELEVBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxFQUFDO0lBQ3JDLEVBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxFQUFDO0lBQ3JDLEVBQUMsSUFBSSxFQUFFLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxFQUFDO0lBQzVELEVBQUMsSUFBSSxFQUFFLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxFQUFDO0lBQzNELEVBQUMsSUFBSSxFQUFFLGdDQUFnQyxFQUFDO0lBQ3hDLEVBQUMsSUFBSSxFQUFFLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxFQUFDO0NBQzlELENBQUM7QUFFRjs7R0FFRztBQUNILE1BQU0sb0JBQW9CLEdBQTZCO0lBQ3JELEVBQUMsSUFBSSxFQUFFLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsWUFBWSxFQUFFLElBQUksQ0FBQyxFQUFDO0lBQy9ELEVBQUMsSUFBSSxFQUFFLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsWUFBWSxFQUFFLEtBQUssQ0FBQyxFQUFDO0lBQ2hFLEVBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxFQUFDO0lBQ3JDLEVBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxFQUFFLFNBQVMsRUFBRSxRQUFRLEVBQUM7SUFDekQsRUFBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLEVBQUUsU0FBUyxFQUFFLFFBQVEsRUFBQztJQUN6RCxFQUFDLElBQUksRUFBRSxRQUFRLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsRUFBQztJQUNyQyxFQUFDLElBQUksRUFBRSxRQUFRLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsRUFBQztDQUN0QyxDQUFDO0FBRUY7O0dBRUc7QUFDSCxNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQztJQUM3QixFQUFFLENBQUMsTUFBTSxDQUFDLFFBQVE7SUFDbEIsRUFBRSxDQUFDLE1BQU0sQ0FBQyxjQUFjO0lBQ3hCLEVBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUTtJQUNsQixFQUFFLENBQUMsTUFBTSxDQUFDLFFBQVE7SUFDbEIsRUFBRSxDQUFDLE1BQU0sQ0FBQyxTQUFTO0lBQ25CLEVBQUUsQ0FBQyxNQUFNLENBQUMsU0FBUztJQUNuQixFQUFFLENBQUMsTUFBTSxDQUFDLFFBQVE7SUFDbEIsRUFBRSxDQUFDLE1BQU0sQ0FBQyxjQUFjO0lBQ3hCLEVBQUUsQ0FBQyxNQUFNLENBQUMsWUFBWTtJQUN0QixFQUFFLENBQUMsTUFBTSxDQUFDLFNBQVM7Q0FDcEIsQ0FBQyxDQUFDO0FBRUg7Ozs7R0FJRztBQUNILE1BQU0sVUFBVSxRQUFRLENBQUMsR0FBbUI7SUFDMUMsS0FBSyxNQUFNLElBQUksSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDN0IsK0ZBQStGO1FBQy9GLDJGQUEyRjtRQUMzRix3RUFBd0U7UUFFeEUsZUFBZTtRQUNmLFdBQVcsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLGVBQXlELENBQUMsQ0FBQztRQUVwRixlQUFlO1FBQ2YsTUFBTSxRQUFRLEdBQ1osSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLEtBQUssa0JBQWtCLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDO1FBQ3JGLFdBQVcsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLFFBQWtELENBQUMsQ0FBQztJQUMvRSxDQUFDO0FBQ0gsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBUyxXQUFXLENBQ2xCLE1BQTRDLEVBQzVDLFFBQWdEO0lBRWhELElBQUksVUFBVSxHQUFxQyxFQUFFLENBQUM7SUFDdEQsMEZBQTBGO0lBQzFGLElBQUksa0JBQWtCLEdBQXFCLElBQUksQ0FBQztJQUNoRCxLQUFLLE1BQU0sRUFBRSxJQUFJLE1BQU0sRUFBRSxDQUFDO1FBQ3hCLE1BQU0sYUFBYSxHQUFHLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO1FBQzdFLElBQ0UsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUM7WUFDNUIsQ0FBQyxhQUFhLEtBQUssa0JBQWtCO2dCQUNuQyxrQkFBa0IsS0FBSyxJQUFJO2dCQUMzQixhQUFhLEtBQUssSUFBSSxDQUFDLEVBQ3pCLENBQUM7WUFDRCxFQUFFLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsVUFBVSxFQUFFLFFBQVEsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQzFELFVBQVUsR0FBRyxFQUFFLENBQUM7WUFDaEIsa0JBQWtCLEdBQUcsSUFBSSxDQUFDO1FBQzVCLENBQUM7UUFDRCxJQUFJLGNBQWMsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7WUFDaEMsVUFBVSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUNwQixFQUFFLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUNyQixrQkFBa0IsR0FBRyxhQUFhLElBQUksa0JBQWtCLENBQUM7UUFDM0QsQ0FBQztJQUNILENBQUM7SUFDRCxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxVQUFVLEVBQUUsUUFBUSxDQUFDLENBQUMsQ0FBQztBQUM3QyxDQUFDO0FBRUQ7O0dBRUc7QUFDSCxTQUFTLE9BQU8sQ0FDZCxHQUFhLEVBQ2IsUUFBd0I7SUFFeEIsa0RBQWtEO0lBQ2xELE1BQU0sTUFBTSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksS0FBSyxFQUFLLENBQUMsQ0FBQztJQUMxRCxLQUFLLE1BQU0sRUFBRSxJQUFJLEdBQUcsRUFBRSxDQUFDO1FBQ3JCLE1BQU0sVUFBVSxHQUFHLFFBQVEsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUN6RCxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFDRCxrRUFBa0U7SUFDbEUsT0FBTyxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsS0FBSyxFQUFFLENBQUMsRUFBRSxFQUFFO1FBQ2pDLE1BQU0sU0FBUyxHQUFHLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDeEMsT0FBTyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO0lBQzlDLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBUyxRQUFRLENBQUksR0FBYTtJQUNoQyxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQztBQUNuQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IEdvb2dsZSBMTEMgQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiBVc2Ugb2YgdGhpcyBzb3VyY2UgY29kZSBpcyBnb3Zlcm5lZCBieSBhbiBNSVQtc3R5bGUgbGljZW5zZSB0aGF0IGNhbiBiZVxuICogZm91bmQgaW4gdGhlIExJQ0VOU0UgZmlsZSBhdCBodHRwczovL2FuZ3VsYXIuaW8vbGljZW5zZVxuICovXG5cbmltcG9ydCAqIGFzIGlyIGZyb20gJy4uLy4uL2lyJztcbmltcG9ydCB7Q29tcGlsYXRpb25Kb2JLaW5kLCB0eXBlIENvbXBpbGF0aW9uSm9ifSBmcm9tICcuLi9jb21waWxhdGlvbic7XG5cbmZ1bmN0aW9uIGtpbmRUZXN0KGtpbmQ6IGlyLk9wS2luZCk6IChvcDogaXIuVXBkYXRlT3ApID0+IGJvb2xlYW4ge1xuICByZXR1cm4gKG9wOiBpci5VcGRhdGVPcCkgPT4gb3Aua2luZCA9PT0ga2luZDtcbn1cblxuZnVuY3Rpb24ga2luZFdpdGhJbnRlcnBvbGF0aW9uVGVzdChcbiAga2luZDogaXIuT3BLaW5kLkF0dHJpYnV0ZSB8IGlyLk9wS2luZC5Qcm9wZXJ0eSB8IGlyLk9wS2luZC5Ib3N0UHJvcGVydHksXG4gIGludGVycG9sYXRpb246IGJvb2xlYW4sXG4pOiAob3A6IGlyLlVwZGF0ZU9wKSA9PiBib29sZWFuIHtcbiAgcmV0dXJuIChvcDogaXIuVXBkYXRlT3ApID0+IHtcbiAgICByZXR1cm4gb3Aua2luZCA9PT0ga2luZCAmJiBpbnRlcnBvbGF0aW9uID09PSBvcC5leHByZXNzaW9uIGluc3RhbmNlb2YgaXIuSW50ZXJwb2xhdGlvbjtcbiAgfTtcbn1cblxuZnVuY3Rpb24gYmFzaWNMaXN0ZW5lcktpbmRUZXN0KG9wOiBpci5DcmVhdGVPcCk6IGJvb2xlYW4ge1xuICByZXR1cm4gKFxuICAgIChvcC5raW5kID09PSBpci5PcEtpbmQuTGlzdGVuZXIgJiYgIShvcC5ob3N0TGlzdGVuZXIgJiYgb3AuaXNBbmltYXRpb25MaXN0ZW5lcikpIHx8XG4gICAgb3Aua2luZCA9PT0gaXIuT3BLaW5kLlR3b1dheUxpc3RlbmVyXG4gICk7XG59XG5cbmZ1bmN0aW9uIG5vbkludGVycG9sYXRpb25Qcm9wZXJ0eUtpbmRUZXN0KG9wOiBpci5VcGRhdGVPcCk6IGJvb2xlYW4ge1xuICByZXR1cm4gKFxuICAgIChvcC5raW5kID09PSBpci5PcEtpbmQuUHJvcGVydHkgfHwgb3Aua2luZCA9PT0gaXIuT3BLaW5kLlR3b1dheVByb3BlcnR5KSAmJlxuICAgICEob3AuZXhwcmVzc2lvbiBpbnN0YW5jZW9mIGlyLkludGVycG9sYXRpb24pXG4gICk7XG59XG5cbmludGVyZmFjZSBSdWxlPFQgZXh0ZW5kcyBpci5DcmVhdGVPcCB8IGlyLlVwZGF0ZU9wPiB7XG4gIHRlc3Q6IChvcDogVCkgPT4gYm9vbGVhbjtcbiAgdHJhbnNmb3JtPzogKG9wczogQXJyYXk8VD4pID0+IEFycmF5PFQ+O1xufVxuXG4vKipcbiAqIERlZmluZXMgdGhlIGdyb3VwcyBiYXNlZCBvbiBgT3BLaW5kYCB0aGF0IG9wcyB3aWxsIGJlIGRpdmlkZWQgaW50bywgZm9yIHRoZSB2YXJpb3VzIGNyZWF0ZVxuICogb3Aga2luZHMuIE9wcyB3aWxsIGJlIGNvbGxlY3RlZCBpbnRvIGdyb3VwcywgdGhlbiBvcHRpb25hbGx5IHRyYW5zZm9ybWVkLCBiZWZvcmUgcmVjb21iaW5pbmdcbiAqIHRoZSBncm91cHMgaW4gdGhlIG9yZGVyIGRlZmluZWQgaGVyZS5cbiAqL1xuY29uc3QgQ1JFQVRFX09SREVSSU5HOiBBcnJheTxSdWxlPGlyLkNyZWF0ZU9wPj4gPSBbXG4gIHt0ZXN0OiAob3ApID0+IG9wLmtpbmQgPT09IGlyLk9wS2luZC5MaXN0ZW5lciAmJiBvcC5ob3N0TGlzdGVuZXIgJiYgb3AuaXNBbmltYXRpb25MaXN0ZW5lcn0sXG4gIHt0ZXN0OiBiYXNpY0xpc3RlbmVyS2luZFRlc3R9LFxuXTtcblxuLyoqXG4gKiBEZWZpbmVzIHRoZSBncm91cHMgYmFzZWQgb24gYE9wS2luZGAgdGhhdCBvcHMgd2lsbCBiZSBkaXZpZGVkIGludG8sIGZvciB0aGUgdmFyaW91cyB1cGRhdGVcbiAqIG9wIGtpbmRzLlxuICovXG5jb25zdCBVUERBVEVfT1JERVJJTkc6IEFycmF5PFJ1bGU8aXIuVXBkYXRlT3A+PiA9IFtcbiAge3Rlc3Q6IGtpbmRUZXN0KGlyLk9wS2luZC5TdHlsZU1hcCksIHRyYW5zZm9ybToga2VlcExhc3R9LFxuICB7dGVzdDoga2luZFRlc3QoaXIuT3BLaW5kLkNsYXNzTWFwKSwgdHJhbnNmb3JtOiBrZWVwTGFzdH0sXG4gIHt0ZXN0OiBraW5kVGVzdChpci5PcEtpbmQuU3R5bGVQcm9wKX0sXG4gIHt0ZXN0OiBraW5kVGVzdChpci5PcEtpbmQuQ2xhc3NQcm9wKX0sXG4gIHt0ZXN0OiBraW5kV2l0aEludGVycG9sYXRpb25UZXN0KGlyLk9wS2luZC5BdHRyaWJ1dGUsIHRydWUpfSxcbiAge3Rlc3Q6IGtpbmRXaXRoSW50ZXJwb2xhdGlvblRlc3QoaXIuT3BLaW5kLlByb3BlcnR5LCB0cnVlKX0sXG4gIHt0ZXN0OiBub25JbnRlcnBvbGF0aW9uUHJvcGVydHlLaW5kVGVzdH0sXG4gIHt0ZXN0OiBraW5kV2l0aEludGVycG9sYXRpb25UZXN0KGlyLk9wS2luZC5BdHRyaWJ1dGUsIGZhbHNlKX0sXG5dO1xuXG4vKipcbiAqIEhvc3QgYmluZGluZ3MgaGF2ZSB0aGVpciBvd24gdXBkYXRlIG9yZGVyaW5nLlxuICovXG5jb25zdCBVUERBVEVfSE9TVF9PUkRFUklORzogQXJyYXk8UnVsZTxpci5VcGRhdGVPcD4+ID0gW1xuICB7dGVzdDoga2luZFdpdGhJbnRlcnBvbGF0aW9uVGVzdChpci5PcEtpbmQuSG9zdFByb3BlcnR5LCB0cnVlKX0sXG4gIHt0ZXN0OiBraW5kV2l0aEludGVycG9sYXRpb25UZXN0KGlyLk9wS2luZC5Ib3N0UHJvcGVydHksIGZhbHNlKX0sXG4gIHt0ZXN0OiBraW5kVGVzdChpci5PcEtpbmQuQXR0cmlidXRlKX0sXG4gIHt0ZXN0OiBraW5kVGVzdChpci5PcEtpbmQuU3R5bGVNYXApLCB0cmFuc2Zvcm06IGtlZXBMYXN0fSxcbiAge3Rlc3Q6IGtpbmRUZXN0KGlyLk9wS2luZC5DbGFzc01hcCksIHRyYW5zZm9ybToga2VlcExhc3R9LFxuICB7dGVzdDoga2luZFRlc3QoaXIuT3BLaW5kLlN0eWxlUHJvcCl9LFxuICB7dGVzdDoga2luZFRlc3QoaXIuT3BLaW5kLkNsYXNzUHJvcCl9LFxuXTtcblxuLyoqXG4gKiBUaGUgc2V0IG9mIGFsbCBvcCBraW5kcyB3ZSBoYW5kbGUgaW4gdGhlIHJlb3JkZXJpbmcgcGhhc2UuXG4gKi9cbmNvbnN0IGhhbmRsZWRPcEtpbmRzID0gbmV3IFNldChbXG4gIGlyLk9wS2luZC5MaXN0ZW5lcixcbiAgaXIuT3BLaW5kLlR3b1dheUxpc3RlbmVyLFxuICBpci5PcEtpbmQuU3R5bGVNYXAsXG4gIGlyLk9wS2luZC5DbGFzc01hcCxcbiAgaXIuT3BLaW5kLlN0eWxlUHJvcCxcbiAgaXIuT3BLaW5kLkNsYXNzUHJvcCxcbiAgaXIuT3BLaW5kLlByb3BlcnR5LFxuICBpci5PcEtpbmQuVHdvV2F5UHJvcGVydHksXG4gIGlyLk9wS2luZC5Ib3N0UHJvcGVydHksXG4gIGlyLk9wS2luZC5BdHRyaWJ1dGUsXG5dKTtcblxuLyoqXG4gKiBNYW55IHR5cGUgb2Ygb3BlcmF0aW9ucyBoYXZlIG9yZGVyaW5nIGNvbnN0cmFpbnRzIHRoYXQgbXVzdCBiZSByZXNwZWN0ZWQuIEZvciBleGFtcGxlLCBhXG4gKiBgQ2xhc3NNYXBgIGluc3RydWN0aW9uIG11c3QgYmUgb3JkZXJlZCBhZnRlciBhIGBTdHlsZU1hcGAgaW5zdHJ1Y3Rpb24sIGluIG9yZGVyIHRvIGhhdmVcbiAqIHByZWRpY3RhYmxlIHNlbWFudGljcyB0aGF0IG1hdGNoIFRlbXBsYXRlRGVmaW5pdGlvbkJ1aWxkZXIgYW5kIGRvbid0IGJyZWFrIGFwcGxpY2F0aW9ucy5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIG9yZGVyT3BzKGpvYjogQ29tcGlsYXRpb25Kb2IpIHtcbiAgZm9yIChjb25zdCB1bml0IG9mIGpvYi51bml0cykge1xuICAgIC8vIEZpcnN0LCB3ZSBwdWxsIG91dCBvcHMgdGhhdCBuZWVkIHRvIGJlIG9yZGVyZWQuIFRoZW4sIHdoZW4gd2UgZW5jb3VudGVyIGFuIG9wIHRoYXQgc2hvdWxkbid0XG4gICAgLy8gYmUgcmVvcmRlcmVkLCBwdXQgdGhlIG9uZXMgd2UndmUgcHVsbGVkIHNvIGZhciBiYWNrIGluIHRoZSBjb3JyZWN0IG9yZGVyLiBGaW5hbGx5LCBpZiB3ZVxuICAgIC8vIHN0aWxsIGhhdmUgb3BzIHB1bGxlZCBhdCB0aGUgZW5kLCBwdXQgdGhlbSBiYWNrIGluIHRoZSBjb3JyZWN0IG9yZGVyLlxuXG4gICAgLy8gQ3JlYXRlIG1vZGU6XG4gICAgb3JkZXJXaXRoaW4odW5pdC5jcmVhdGUsIENSRUFURV9PUkRFUklORyBhcyBBcnJheTxSdWxlPGlyLkNyZWF0ZU9wIHwgaXIuVXBkYXRlT3A+Pik7XG5cbiAgICAvLyBVcGRhdGUgbW9kZTpcbiAgICBjb25zdCBvcmRlcmluZyA9XG4gICAgICB1bml0LmpvYi5raW5kID09PSBDb21waWxhdGlvbkpvYktpbmQuSG9zdCA/IFVQREFURV9IT1NUX09SREVSSU5HIDogVVBEQVRFX09SREVSSU5HO1xuICAgIG9yZGVyV2l0aGluKHVuaXQudXBkYXRlLCBvcmRlcmluZyBhcyBBcnJheTxSdWxlPGlyLkNyZWF0ZU9wIHwgaXIuVXBkYXRlT3A+Pik7XG4gIH1cbn1cblxuLyoqXG4gKiBPcmRlciBhbGwgdGhlIG9wcyB3aXRoaW4gdGhlIHNwZWNpZmllZCBncm91cC5cbiAqL1xuZnVuY3Rpb24gb3JkZXJXaXRoaW4oXG4gIG9wTGlzdDogaXIuT3BMaXN0PGlyLkNyZWF0ZU9wIHwgaXIuVXBkYXRlT3A+LFxuICBvcmRlcmluZzogQXJyYXk8UnVsZTxpci5DcmVhdGVPcCB8IGlyLlVwZGF0ZU9wPj4sXG4pIHtcbiAgbGV0IG9wc1RvT3JkZXI6IEFycmF5PGlyLkNyZWF0ZU9wIHwgaXIuVXBkYXRlT3A+ID0gW107XG4gIC8vIE9ubHkgcmVvcmRlciBvcHMgdGhhdCB0YXJnZXQgdGhlIHNhbWUgeHJlZjsgZG8gbm90IG1peCBvcHMgdGhhdCB0YXJnZXQgZGlmZmVyZW50IHhyZWZzLlxuICBsZXQgZmlyc3RUYXJnZXRJbkdyb3VwOiBpci5YcmVmSWQgfCBudWxsID0gbnVsbDtcbiAgZm9yIChjb25zdCBvcCBvZiBvcExpc3QpIHtcbiAgICBjb25zdCBjdXJyZW50VGFyZ2V0ID0gaXIuaGFzRGVwZW5kc09uU2xvdENvbnRleHRUcmFpdChvcCkgPyBvcC50YXJnZXQgOiBudWxsO1xuICAgIGlmIChcbiAgICAgICFoYW5kbGVkT3BLaW5kcy5oYXMob3Aua2luZCkgfHxcbiAgICAgIChjdXJyZW50VGFyZ2V0ICE9PSBmaXJzdFRhcmdldEluR3JvdXAgJiZcbiAgICAgICAgZmlyc3RUYXJnZXRJbkdyb3VwICE9PSBudWxsICYmXG4gICAgICAgIGN1cnJlbnRUYXJnZXQgIT09IG51bGwpXG4gICAgKSB7XG4gICAgICBpci5PcExpc3QuaW5zZXJ0QmVmb3JlKHJlb3JkZXIob3BzVG9PcmRlciwgb3JkZXJpbmcpLCBvcCk7XG4gICAgICBvcHNUb09yZGVyID0gW107XG4gICAgICBmaXJzdFRhcmdldEluR3JvdXAgPSBudWxsO1xuICAgIH1cbiAgICBpZiAoaGFuZGxlZE9wS2luZHMuaGFzKG9wLmtpbmQpKSB7XG4gICAgICBvcHNUb09yZGVyLnB1c2gob3ApO1xuICAgICAgaXIuT3BMaXN0LnJlbW92ZShvcCk7XG4gICAgICBmaXJzdFRhcmdldEluR3JvdXAgPSBjdXJyZW50VGFyZ2V0ID8/IGZpcnN0VGFyZ2V0SW5Hcm91cDtcbiAgICB9XG4gIH1cbiAgb3BMaXN0LnB1c2gocmVvcmRlcihvcHNUb09yZGVyLCBvcmRlcmluZykpO1xufVxuXG4vKipcbiAqIFJlb3JkZXJzIHRoZSBnaXZlbiBsaXN0IG9mIG9wcyBhY2NvcmRpbmcgdG8gdGhlIG9yZGVyaW5nIGRlZmluZWQgYnkgYE9SREVSSU5HYC5cbiAqL1xuZnVuY3Rpb24gcmVvcmRlcjxUIGV4dGVuZHMgaXIuQ3JlYXRlT3AgfCBpci5VcGRhdGVPcD4oXG4gIG9wczogQXJyYXk8VD4sXG4gIG9yZGVyaW5nOiBBcnJheTxSdWxlPFQ+Pixcbik6IEFycmF5PFQ+IHtcbiAgLy8gQnJlYWsgdGhlIG9wcyBsaXN0IGludG8gZ3JvdXBzIGJhc2VkIG9uIE9wS2luZC5cbiAgY29uc3QgZ3JvdXBzID0gQXJyYXkuZnJvbShvcmRlcmluZywgKCkgPT4gbmV3IEFycmF5PFQ+KCkpO1xuICBmb3IgKGNvbnN0IG9wIG9mIG9wcykge1xuICAgIGNvbnN0IGdyb3VwSW5kZXggPSBvcmRlcmluZy5maW5kSW5kZXgoKG8pID0+IG8udGVzdChvcCkpO1xuICAgIGdyb3Vwc1tncm91cEluZGV4XS5wdXNoKG9wKTtcbiAgfVxuICAvLyBSZWFzc2VtYmxlIHRoZSBncm91cHMgaW50byBhIHNpbmdsZSBsaXN0LCBpbiB0aGUgY29ycmVjdCBvcmRlci5cbiAgcmV0dXJuIGdyb3Vwcy5mbGF0TWFwKChncm91cCwgaSkgPT4ge1xuICAgIGNvbnN0IHRyYW5zZm9ybSA9IG9yZGVyaW5nW2ldLnRyYW5zZm9ybTtcbiAgICByZXR1cm4gdHJhbnNmb3JtID8gdHJhbnNmb3JtKGdyb3VwKSA6IGdyb3VwO1xuICB9KTtcbn1cblxuLyoqXG4gKiBLZWVwcyBvbmx5IHRoZSBsYXN0IG9wIGluIGEgbGlzdCBvZiBvcHMuXG4gKi9cbmZ1bmN0aW9uIGtlZXBMYXN0PFQ+KG9wczogQXJyYXk8VD4pIHtcbiAgcmV0dXJuIG9wcy5zbGljZShvcHMubGVuZ3RoIC0gMSk7XG59XG4iXX0=