/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as o from '../../../../output/output_ast';
import * as ir from '../../ir';
/**
 * Optimize variables declared and used in the IR.
 *
 * Variables are eagerly generated by pipeline stages for all possible values that could be
 * referenced. This stage processes the list of declared variables and all variable usages,
 * and optimizes where possible. It performs 3 main optimizations:
 *
 *   * It transforms variable declarations to side effectful expressions when the
 *     variable is not used, but its initializer has global effects which other
 *     operations rely upon.
 *   * It removes variable declarations if those variables are not referenced and
 *     either they do not have global effects, or nothing relies on them.
 *   * It inlines variable declarations when those variables are only used once
 *     and the inlining is semantically safe.
 *
 * To guarantee correctness, analysis of "fences" in the instruction lists is used to determine
 * which optimizations are safe to perform.
 */
export function optimizeVariables(job) {
    for (const unit of job.units) {
        inlineAlwaysInlineVariables(unit.create);
        inlineAlwaysInlineVariables(unit.update);
        for (const op of unit.create) {
            if (op.kind === ir.OpKind.Listener || op.kind === ir.OpKind.TwoWayListener) {
                inlineAlwaysInlineVariables(op.handlerOps);
            }
        }
        optimizeVariablesInOpList(unit.create, job.compatibility);
        optimizeVariablesInOpList(unit.update, job.compatibility);
        for (const op of unit.create) {
            if (op.kind === ir.OpKind.Listener || op.kind === ir.OpKind.TwoWayListener) {
                optimizeVariablesInOpList(op.handlerOps, job.compatibility);
            }
        }
    }
}
/**
 * A [fence](https://en.wikipedia.org/wiki/Memory_barrier) flag for an expression which indicates
 * how that expression can be optimized in relation to other expressions or instructions.
 *
 * `Fence`s are a bitfield, so multiple flags may be set on a single expression.
 */
var Fence;
(function (Fence) {
    /**
     * Empty flag (no fence exists).
     */
    Fence[Fence["None"] = 0] = "None";
    /**
     * A context read fence, meaning that the expression in question reads from the "current view"
     * context of the runtime.
     */
    Fence[Fence["ViewContextRead"] = 1] = "ViewContextRead";
    /**
     * A context write fence, meaning that the expression in question writes to the "current view"
     * context of the runtime.
     *
     * Note that all `ContextWrite` fences are implicitly `ContextRead` fences as operations which
     * change the view context do so based on the current one.
     */
    Fence[Fence["ViewContextWrite"] = 2] = "ViewContextWrite";
    /**
     * Indicates that a call is required for its side-effects, even if nothing reads its result.
     *
     * This is also true of `ViewContextWrite` operations **if** they are followed by a
     * `ViewContextRead`.
     */
    Fence[Fence["SideEffectful"] = 4] = "SideEffectful";
})(Fence || (Fence = {}));
function inlineAlwaysInlineVariables(ops) {
    const vars = new Map();
    for (const op of ops) {
        if (op.kind === ir.OpKind.Variable && op.flags & ir.VariableFlags.AlwaysInline) {
            ir.visitExpressionsInOp(op, (expr) => {
                if (ir.isIrExpression(expr) && fencesForIrExpression(expr) !== Fence.None) {
                    throw new Error(`AssertionError: A context-sensitive variable was marked AlwaysInline`);
                }
            });
            vars.set(op.xref, op);
        }
        ir.transformExpressionsInOp(op, (expr) => {
            if (expr instanceof ir.ReadVariableExpr && vars.has(expr.xref)) {
                const varOp = vars.get(expr.xref);
                // Inline by cloning, because we might inline into multiple places.
                return varOp.initializer.clone();
            }
            return expr;
        }, ir.VisitorContextFlag.None);
    }
    for (const op of vars.values()) {
        ir.OpList.remove(op);
    }
}
/**
 * Process a list of operations and optimize variables within that list.
 */
function optimizeVariablesInOpList(ops, compatibility) {
    const varDecls = new Map();
    const varUsages = new Map();
    // Track variables that are used outside of the immediate operation list. For example, within
    // `ListenerOp` handler operations of listeners in the current operation list.
    const varRemoteUsages = new Set();
    const opMap = new Map();
    // First, extract information about variables declared or used within the whole list.
    for (const op of ops) {
        if (op.kind === ir.OpKind.Variable) {
            if (varDecls.has(op.xref) || varUsages.has(op.xref)) {
                throw new Error(`Should not see two declarations of the same variable: ${op.xref}`);
            }
            varDecls.set(op.xref, op);
            varUsages.set(op.xref, 0);
        }
        opMap.set(op, collectOpInfo(op));
        countVariableUsages(op, varUsages, varRemoteUsages);
    }
    // The next step is to remove any variable declarations for variables that aren't used. The
    // variable initializer expressions may be side-effectful, so they may need to be retained as
    // expression statements.
    // Track whether we've seen an operation which reads from the view context yet. This is used to
    // determine whether a write to the view context in a variable initializer can be observed.
    let contextIsUsed = false;
    // Note that iteration through the list happens in reverse, which guarantees that we'll process
    // all reads of a variable prior to processing its declaration.
    for (const op of ops.reversed()) {
        const opInfo = opMap.get(op);
        if (op.kind === ir.OpKind.Variable && varUsages.get(op.xref) === 0) {
            // This variable is unused and can be removed. We might need to keep the initializer around,
            // though, if something depends on it running.
            if ((contextIsUsed && opInfo.fences & Fence.ViewContextWrite) ||
                opInfo.fences & Fence.SideEffectful) {
                // This variable initializer has a side effect which must be retained. Either:
                //  * it writes to the view context, and we know there is a future operation which depends
                //    on that write, or
                //  * it's an operation which is inherently side-effectful.
                // We can't remove the initializer, but we can remove the variable declaration itself and
                // replace it with a side-effectful statement.
                const stmtOp = ir.createStatementOp(op.initializer.toStmt());
                opMap.set(stmtOp, opInfo);
                ir.OpList.replace(op, stmtOp);
            }
            else {
                // It's safe to delete this entire variable declaration as nothing depends on it, even
                // side-effectfully. Note that doing this might make other variables unused. Since we're
                // iterating in reverse order, we should always be processing usages before declarations
                // and therefore by the time we get to a declaration, all removable usages will have been
                // removed.
                uncountVariableUsages(op, varUsages);
                ir.OpList.remove(op);
            }
            opMap.delete(op);
            varDecls.delete(op.xref);
            varUsages.delete(op.xref);
            continue;
        }
        // Does this operation depend on the view context?
        if (opInfo.fences & Fence.ViewContextRead) {
            contextIsUsed = true;
        }
    }
    // Next, inline any remaining variables with exactly one usage.
    const toInline = [];
    for (const [id, count] of varUsages) {
        const decl = varDecls.get(id);
        // We can inline variables that:
        //  - are used exactly once, and
        //  - are not used remotely
        // OR
        //  - are marked for always inlining
        const isAlwaysInline = !!(decl.flags & ir.VariableFlags.AlwaysInline);
        if (count !== 1 || isAlwaysInline) {
            // We can't inline this variable as it's used more than once.
            continue;
        }
        if (varRemoteUsages.has(id)) {
            // This variable is used once, but across an operation boundary, so it can't be inlined.
            continue;
        }
        toInline.push(id);
    }
    let candidate;
    while ((candidate = toInline.pop())) {
        // We will attempt to inline this variable. If inlining fails (due to fences for example),
        // no future operation will make inlining legal.
        const decl = varDecls.get(candidate);
        const varInfo = opMap.get(decl);
        const isAlwaysInline = !!(decl.flags & ir.VariableFlags.AlwaysInline);
        if (isAlwaysInline) {
            throw new Error(`AssertionError: Found an 'AlwaysInline' variable after the always inlining pass.`);
        }
        // Scan operations following the variable declaration and look for the point where that variable
        // is used. There should only be one usage given the precondition above.
        for (let targetOp = decl.next; targetOp.kind !== ir.OpKind.ListEnd; targetOp = targetOp.next) {
            const opInfo = opMap.get(targetOp);
            // Is the variable used in this operation?
            if (opInfo.variablesUsed.has(candidate)) {
                if (compatibility === ir.CompatibilityMode.TemplateDefinitionBuilder &&
                    !allowConservativeInlining(decl, targetOp)) {
                    // We're in conservative mode, and this variable is not eligible for inlining into the
                    // target operation in this mode.
                    break;
                }
                // Yes, try to inline it. Inlining may not be successful if fences in this operation before
                // the variable's usage cannot be safely crossed.
                if (tryInlineVariableInitializer(candidate, decl.initializer, targetOp, varInfo.fences)) {
                    // Inlining was successful! Update the tracking structures to reflect the inlined
                    // variable.
                    opInfo.variablesUsed.delete(candidate);
                    // Add all variables used in the variable's initializer to its new usage site.
                    for (const id of varInfo.variablesUsed) {
                        opInfo.variablesUsed.add(id);
                    }
                    // Merge fences in the variable's initializer into its new usage site.
                    opInfo.fences |= varInfo.fences;
                    // Delete tracking info related to the declaration.
                    varDecls.delete(candidate);
                    varUsages.delete(candidate);
                    opMap.delete(decl);
                    // And finally, delete the original declaration from the operation list.
                    ir.OpList.remove(decl);
                }
                // Whether inlining succeeded or failed, we're done processing this variable.
                break;
            }
            // If the variable is not used in this operation, then we'd need to inline across it. Check if
            // that's safe to do.
            if (!safeToInlinePastFences(opInfo.fences, varInfo.fences)) {
                // We can't safely inline this variable beyond this operation, so don't proceed with
                // inlining this variable.
                break;
            }
        }
    }
}
/**
 * Given an `ir.Expression`, returns the `Fence` flags for that expression type.
 */
function fencesForIrExpression(expr) {
    switch (expr.kind) {
        case ir.ExpressionKind.NextContext:
            return Fence.ViewContextRead | Fence.ViewContextWrite;
        case ir.ExpressionKind.RestoreView:
            return Fence.ViewContextRead | Fence.ViewContextWrite | Fence.SideEffectful;
        case ir.ExpressionKind.StoreLet:
            return Fence.SideEffectful;
        case ir.ExpressionKind.Reference:
        case ir.ExpressionKind.ContextLetReference:
            return Fence.ViewContextRead;
        default:
            return Fence.None;
    }
}
/**
 * Build the `OpInfo` structure for the given `op`. This performs two operations:
 *
 *  * It tracks which variables are used in the operation's expressions.
 *  * It rolls up fence flags for expressions within the operation.
 */
function collectOpInfo(op) {
    let fences = Fence.None;
    const variablesUsed = new Set();
    ir.visitExpressionsInOp(op, (expr) => {
        if (!ir.isIrExpression(expr)) {
            return;
        }
        switch (expr.kind) {
            case ir.ExpressionKind.ReadVariable:
                variablesUsed.add(expr.xref);
                break;
            default:
                fences |= fencesForIrExpression(expr);
        }
    });
    return { fences, variablesUsed };
}
/**
 * Count the number of usages of each variable, being careful to track whether those usages are
 * local or remote.
 */
function countVariableUsages(op, varUsages, varRemoteUsage) {
    ir.visitExpressionsInOp(op, (expr, flags) => {
        if (!ir.isIrExpression(expr)) {
            return;
        }
        if (expr.kind !== ir.ExpressionKind.ReadVariable) {
            return;
        }
        const count = varUsages.get(expr.xref);
        if (count === undefined) {
            // This variable is declared outside the current scope of optimization.
            return;
        }
        varUsages.set(expr.xref, count + 1);
        if (flags & ir.VisitorContextFlag.InChildOperation) {
            varRemoteUsage.add(expr.xref);
        }
    });
}
/**
 * Remove usages of a variable in `op` from the `varUsages` tracking.
 */
function uncountVariableUsages(op, varUsages) {
    ir.visitExpressionsInOp(op, (expr) => {
        if (!ir.isIrExpression(expr)) {
            return;
        }
        if (expr.kind !== ir.ExpressionKind.ReadVariable) {
            return;
        }
        const count = varUsages.get(expr.xref);
        if (count === undefined) {
            // This variable is declared outside the current scope of optimization.
            return;
        }
        else if (count === 0) {
            throw new Error(`Inaccurate variable count: ${expr.xref} - found another read but count is already 0`);
        }
        varUsages.set(expr.xref, count - 1);
    });
}
/**
 * Checks whether it's safe to inline a variable across a particular operation.
 *
 * @param fences the fences of the operation which the inlining will cross
 * @param declFences the fences of the variable being inlined.
 */
function safeToInlinePastFences(fences, declFences) {
    if (fences & Fence.ViewContextWrite) {
        // It's not safe to inline context reads across context writes.
        if (declFences & Fence.ViewContextRead) {
            return false;
        }
    }
    else if (fences & Fence.ViewContextRead) {
        // It's not safe to inline context writes across context reads.
        if (declFences & Fence.ViewContextWrite) {
            return false;
        }
    }
    return true;
}
/**
 * Attempt to inline the initializer of a variable into a target operation's expressions.
 *
 * This may or may not be safe to do. For example, the variable could be read following the
 * execution of an expression with fences that don't permit the variable to be inlined across them.
 */
function tryInlineVariableInitializer(id, initializer, target, declFences) {
    // We use `ir.transformExpressionsInOp` to walk the expressions and inline the variable if
    // possible. Since this operation is callback-based, once inlining succeeds or fails we can't
    // "stop" the expression processing, and have to keep track of whether inlining has succeeded or
    // is no longer allowed.
    let inlined = false;
    let inliningAllowed = true;
    ir.transformExpressionsInOp(target, (expr, flags) => {
        if (!ir.isIrExpression(expr)) {
            return expr;
        }
        if (inlined || !inliningAllowed) {
            // Either the inlining has already succeeded, or we've passed a fence that disallows inlining
            // at this point, so don't try.
            return expr;
        }
        else if (flags & ir.VisitorContextFlag.InChildOperation &&
            declFences & Fence.ViewContextRead) {
            // We cannot inline variables that are sensitive to the current context across operation
            // boundaries.
            return expr;
        }
        switch (expr.kind) {
            case ir.ExpressionKind.ReadVariable:
                if (expr.xref === id) {
                    // This is the usage site of the variable. Since nothing has disallowed inlining, it's
                    // safe to inline the initializer here.
                    inlined = true;
                    return initializer;
                }
                break;
            default:
                // For other types of `ir.Expression`s, whether inlining is allowed depends on their fences.
                const exprFences = fencesForIrExpression(expr);
                inliningAllowed = inliningAllowed && safeToInlinePastFences(exprFences, declFences);
                break;
        }
        return expr;
    }, ir.VisitorContextFlag.None);
    return inlined;
}
/**
 * Determines whether inlining of `decl` should be allowed in "conservative" mode.
 *
 * In conservative mode, inlining behavior is limited to those operations which the
 * `TemplateDefinitionBuilder` supported, with the goal of producing equivalent output.
 */
function allowConservativeInlining(decl, target) {
    // TODO(alxhub): understand exactly how TemplateDefinitionBuilder approaches inlining, and record
    // that behavior here.
    switch (decl.variable.kind) {
        case ir.SemanticVariableKind.Identifier:
            if (decl.initializer instanceof o.ReadVarExpr && decl.initializer.name === 'ctx') {
                // Although TemplateDefinitionBuilder is cautious about inlining, we still want to do so
                // when the variable is the context, to imitate its behavior with aliases in control flow
                // blocks. This quirky behavior will become dead code once compatibility mode is no longer
                // supported.
                return true;
            }
            return false;
        case ir.SemanticVariableKind.Context:
            // Context can only be inlined into other variables.
            return target.kind === ir.OpKind.Variable;
        default:
            return true;
    }
}
//# sourceMappingURL=data:application/json;base64,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