/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as eventLib from './event';
import * as eventInfoLib from './event_info';
import { EventType } from './event_type';
/**
 * Update `EventInfo` to be `eventType = 'click'` and sets `a11yClickKey` if it
 * is a a11y click.
 */
export function updateEventInfoForA11yClick(eventInfo) {
    if (!eventLib.isActionKeyEvent(eventInfoLib.getEvent(eventInfo))) {
        return;
    }
    eventInfoLib.setA11yClickKey(eventInfo, true);
    // A 'click' triggered by a DOM keypress should be mapped to the 'click'
    // jsaction.
    eventInfoLib.setEventType(eventInfo, EventType.CLICK);
}
/**
 * Call `preventDefault` on an a11y click if it is space key or to prevent the
 * browser's default action for native HTML controls.
 */
export function preventDefaultForA11yClick(eventInfo) {
    if (!eventInfoLib.getA11yClickKey(eventInfo) ||
        (!eventLib.isSpaceKeyEvent(eventInfoLib.getEvent(eventInfo)) &&
            !eventLib.shouldCallPreventDefaultOnNativeHtmlControl(eventInfoLib.getEvent(eventInfo)))) {
        return;
    }
    eventLib.preventDefault(eventInfoLib.getEvent(eventInfo));
}
/**
 * Sets the `action` to `clickonly` for a click event that is not an a11y click
 * and if there is not already a click action.
 */
export function populateClickOnlyAction(actionElement, eventInfo, actionMap) {
    if (
    // If there's already an action, don't attempt to set a CLICKONLY
    eventInfoLib.getAction(eventInfo) ||
        // Only attempt CLICKONLY if the type is CLICK
        eventInfoLib.getEventType(eventInfo) !== EventType.CLICK ||
        // a11y clicks are never CLICKONLY
        eventInfoLib.getA11yClickKey(eventInfo) ||
        actionMap[EventType.CLICKONLY] === undefined) {
        return;
    }
    eventInfoLib.setEventType(eventInfo, EventType.CLICKONLY);
    eventInfoLib.setAction(eventInfo, actionMap[EventType.CLICKONLY], actionElement);
}
//# sourceMappingURL=data:application/json;base64,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