/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { createEventInfoFromParameters } from './event_info';
/**
 * EarlyEventContract intercepts events in the bubbling phase at the
 * boundary of the document body. This mapping will be passed to the
 * late-loaded EventContract.
 */
export class EarlyEventContract {
    constructor(dataContainer = window, container = window.document.documentElement) {
        this.dataContainer = dataContainer;
        dataContainer._ejsa = createEarlyJsactionData(container);
    }
    /**
     * Installs a list of event types for container .
     */
    addEvents(types, capture) {
        addEvents(this.dataContainer._ejsa, types, capture);
    }
}
/** Creates an `EarlyJsactionData` object. */
export function createEarlyJsactionData(container) {
    const q = [];
    const d = (eventInfo) => {
        q.push(eventInfo);
    };
    const h = (event) => {
        d(createEventInfoFromParameters(event.type, event, event.target, container, Date.now()));
    };
    return {
        c: container,
        q,
        et: [],
        etc: [],
        d,
        h,
    };
}
/** Add all the events to the container stored in the `EarlyJsactionData`. */
export function addEvents(earlyJsactionData, types, capture) {
    for (let i = 0; i < types.length; i++) {
        const eventType = types[i];
        const eventTypes = capture ? earlyJsactionData.etc : earlyJsactionData.et;
        eventTypes.push(eventType);
        earlyJsactionData.c.addEventListener(eventType, earlyJsactionData.h, capture);
    }
}
/** Get the queued `EventInfo` objects that were dispatched before a dispatcher was registered. */
export function getQueuedEventInfos(earlyJsactionData) {
    return earlyJsactionData?.q ?? [];
}
/** Register a different dispatcher function on the `EarlyJsactionData`. */
export function registerDispatcher(earlyJsactionData, dispatcher) {
    if (!earlyJsactionData) {
        return;
    }
    earlyJsactionData.d = dispatcher;
}
/** Removes all event listener handlers. */
export function removeAllEventListeners(earlyJsactionData) {
    if (!earlyJsactionData) {
        return;
    }
    removeEventListeners(earlyJsactionData.c, earlyJsactionData.et, earlyJsactionData.h);
    removeEventListeners(earlyJsactionData.c, earlyJsactionData.etc, earlyJsactionData.h, true);
}
function removeEventListeners(container, eventTypes, earlyEventHandler, capture) {
    for (let i = 0; i < eventTypes.length; i++) {
        container.removeEventListener(eventTypes[i], earlyEventHandler, /* useCapture */ capture);
    }
}
//# sourceMappingURL=data:application/json;base64,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