/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ActionResolver } from './action_resolver';
import { Dispatcher } from './dispatcher';
import { Restriction } from './restriction';
/** An internal symbol used to indicate whether propagation should be stopped or not. */
export const PROPAGATION_STOPPED_SYMBOL = Symbol.for('propagationStopped');
/** Extra event phases beyond what the browser provides. */
export const EventPhase = {
    REPLAY: 101,
};
const PREVENT_DEFAULT_ERROR_MESSAGE_DETAILS = ' Because event replay occurs after browser dispatch, `preventDefault` would have no ' +
    'effect. You can check whether an event is being replayed by accessing the event phase: ' +
    '`event.eventPhase === EventPhase.REPLAY`.';
const PREVENT_DEFAULT_ERROR_MESSAGE = `\`preventDefault\` called during event replay.`;
const COMPOSED_PATH_ERROR_MESSAGE_DETAILS = ' Because event replay occurs after browser ' +
    'dispatch, `composedPath()` will be empty. Iterate parent nodes from `event.target` or ' +
    '`event.currentTarget` if you need to check elements in the event path.';
const COMPOSED_PATH_ERROR_MESSAGE = `\`composedPath\` called during event replay.`;
/**
 * A dispatcher that uses browser-based `Event` semantics, for example bubbling, `stopPropagation`,
 * `currentTarget`, etc.
 */
export class EventDispatcher {
    constructor(dispatchDelegate) {
        this.dispatchDelegate = dispatchDelegate;
        this.actionResolver = new ActionResolver();
        this.dispatcher = new Dispatcher((eventInfoWrapper) => {
            this.dispatchToDelegate(eventInfoWrapper);
        }, {
            actionResolver: this.actionResolver,
        });
    }
    /**
     * The entrypoint for the `EventContract` dispatch.
     */
    dispatch(eventInfo) {
        this.dispatcher.dispatch(eventInfo);
    }
    /** Internal method that does basic disaptching. */
    dispatchToDelegate(eventInfoWrapper) {
        if (eventInfoWrapper.getIsReplay()) {
            prepareEventForReplay(eventInfoWrapper);
        }
        prepareEventForBubbling(eventInfoWrapper);
        while (eventInfoWrapper.getAction()) {
            prepareEventForDispatch(eventInfoWrapper);
            this.dispatchDelegate(eventInfoWrapper.getEvent(), eventInfoWrapper.getAction().name);
            if (propagationStopped(eventInfoWrapper)) {
                return;
            }
            this.actionResolver.resolveParentAction(eventInfoWrapper.eventInfo);
        }
    }
}
function prepareEventForBubbling(eventInfoWrapper) {
    const event = eventInfoWrapper.getEvent();
    const stopPropagation = () => {
        event[PROPAGATION_STOPPED_SYMBOL] = true;
    };
    patchEventInstance(event, 'stopPropagation', stopPropagation);
    patchEventInstance(event, 'stopImmediatePropagation', stopPropagation);
}
function propagationStopped(eventInfoWrapper) {
    const event = eventInfoWrapper.getEvent();
    return !!event[PROPAGATION_STOPPED_SYMBOL];
}
function prepareEventForReplay(eventInfoWrapper) {
    const event = eventInfoWrapper.getEvent();
    const target = eventInfoWrapper.getTargetElement();
    patchEventInstance(event, 'target', target);
    patchEventInstance(event, 'eventPhase', EventPhase.REPLAY);
    patchEventInstance(event, 'preventDefault', () => {
        throw new Error(PREVENT_DEFAULT_ERROR_MESSAGE + (ngDevMode ? PREVENT_DEFAULT_ERROR_MESSAGE_DETAILS : ''));
    });
    patchEventInstance(event, 'composedPath', () => {
        throw new Error(COMPOSED_PATH_ERROR_MESSAGE + (ngDevMode ? COMPOSED_PATH_ERROR_MESSAGE_DETAILS : ''));
    });
}
function prepareEventForDispatch(eventInfoWrapper) {
    const event = eventInfoWrapper.getEvent();
    const currentTarget = eventInfoWrapper.getAction()?.element;
    if (currentTarget) {
        patchEventInstance(event, 'currentTarget', currentTarget, {
            // `currentTarget` is going to get reassigned every dispatch.
            configurable: true,
        });
    }
}
/**
 * Patch `Event` instance during non-standard `Event` dispatch. This patches just the `Event`
 * instance that the browser created, it does not patch global properties or methods.
 *
 * This is necessary because dispatching an `Event` outside of browser dispatch results in
 * incorrect properties and methods that need to be polyfilled or do not work.
 *
 * JSAction dispatch adds two extra "phases" to event dispatch:
 * 1. Event delegation - the event is being dispatched by a delegating event handler on a container
 *    (typically `window.document.documentElement`), to a delegated event handler on some child
 *    element. Certain `Event` properties will be unintuitive, such as `currentTarget`, which would
 *    be the container rather than the child element. Bubbling would also not work. In order to
 *    emulate the browser, these properties and methods on the `Event` are patched.
 * 2. Event replay - the event is being dispatched by the framework once the handlers have been
 *    loaded (during hydration, or late-loaded). Certain `Event` properties can be unset by the
 *    browser because the `Event` is no longer actively being dispatched, such as `target`. Other
 *    methods have no effect because the `Event` has already been dispatched, such as
 *    `preventDefault`. Bubbling would also not work. These properties and methods are patched,
 *    either to fill in information that the browser may have removed, or to throw errors in methods
 *    that no longer behave as expected.
 */
function patchEventInstance(event, property, value, { configurable = false } = {}) {
    Object.defineProperty(event, property, { value, configurable });
}
/**
 * Registers deferred functionality for an EventContract and a Jsaction
 * Dispatcher.
 */
export function registerDispatcher(eventContract, dispatcher) {
    eventContract.ecrd((eventInfo) => {
        dispatcher.dispatch(eventInfo);
    }, Restriction.I_AM_THE_JSACTION_FRAMEWORK);
}
//# sourceMappingURL=data:application/json;base64,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