/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
/** Added for readability when accessing stable property names. */
export function getEventType(eventInfo) {
    return eventInfo.eventType;
}
/** Added for readability when accessing stable property names. */
export function setEventType(eventInfo, eventType) {
    eventInfo.eventType = eventType;
}
/** Added for readability when accessing stable property names. */
export function getEvent(eventInfo) {
    return eventInfo.event;
}
/** Added for readability when accessing stable property names. */
export function setEvent(eventInfo, event) {
    eventInfo.event = event;
}
/** Added for readability when accessing stable property names. */
export function getTargetElement(eventInfo) {
    return eventInfo.targetElement;
}
/** Added for readability when accessing stable property names. */
export function setTargetElement(eventInfo, targetElement) {
    eventInfo.targetElement = targetElement;
}
/** Added for readability when accessing stable property names. */
export function getContainer(eventInfo) {
    return eventInfo.eic;
}
/** Added for readability when accessing stable property names. */
export function setContainer(eventInfo, container) {
    eventInfo.eic = container;
}
/** Added for readability when accessing stable property names. */
export function getTimestamp(eventInfo) {
    return eventInfo.timeStamp;
}
/** Added for readability when accessing stable property names. */
export function setTimestamp(eventInfo, timestamp) {
    eventInfo.timeStamp = timestamp;
}
/** Added for readability when accessing stable property names. */
export function getAction(eventInfo) {
    return eventInfo.eia;
}
/** Added for readability when accessing stable property names. */
export function setAction(eventInfo, actionName, actionElement) {
    eventInfo.eia = [actionName, actionElement];
}
/** Added for readability when accessing stable property names. */
export function unsetAction(eventInfo) {
    eventInfo.eia = undefined;
}
/** Added for readability when accessing stable property names. */
export function getActionName(actionInfo) {
    return actionInfo[0];
}
/** Added for readability when accessing stable property names. */
export function getActionElement(actionInfo) {
    return actionInfo[1];
}
/** Added for readability when accessing stable property names. */
export function getIsReplay(eventInfo) {
    return eventInfo.eirp;
}
/** Added for readability when accessing stable property names. */
export function setIsReplay(eventInfo, replay) {
    eventInfo.eirp = replay;
}
/** Added for readability when accessing stable property names. */
export function getA11yClickKey(eventInfo) {
    return eventInfo.eiack;
}
/** Added for readability when accessing stable property names. */
export function setA11yClickKey(eventInfo, a11yClickKey) {
    eventInfo.eiack = a11yClickKey;
}
/** Added for readability when accessing stable property names. */
export function getResolved(eventInfo) {
    return eventInfo.eir;
}
/** Added for readability when accessing stable property names. */
export function setResolved(eventInfo, resolved) {
    eventInfo.eir = resolved;
}
/** Clones an `EventInfo` */
export function cloneEventInfo(eventInfo) {
    return {
        eventType: eventInfo.eventType,
        event: eventInfo.event,
        targetElement: eventInfo.targetElement,
        eic: eventInfo.eic,
        eia: eventInfo.eia,
        timeStamp: eventInfo.timeStamp,
        eirp: eventInfo.eirp,
        eiack: eventInfo.eiack,
        eir: eventInfo.eir,
    };
}
/**
 * Utility function for creating an `EventInfo`.
 *
 * This can be used from code-size sensitive compilation units, as taking
 * parameters vs. an `Object` literal reduces code size.
 */
export function createEventInfoFromParameters(eventType, event, targetElement, container, timestamp, action, isReplay, a11yClickKey) {
    return {
        eventType,
        event,
        targetElement,
        eic: container,
        timeStamp: timestamp,
        eia: action,
        eirp: isReplay,
        eiack: a11yClickKey,
    };
}
/**
 * Utility function for creating an `EventInfo`.
 *
 * This should be used in compilation units that are less sensitive to code
 * size.
 */
export function createEventInfo({ eventType, event, targetElement, container, timestamp, action, isReplay, a11yClickKey, }) {
    return {
        eventType,
        event,
        targetElement,
        eic: container,
        timeStamp: timestamp,
        eia: action ? [action.name, action.element] : undefined,
        eirp: isReplay,
        eiack: a11yClickKey,
    };
}
/**
 * Utility class around an `EventInfo`.
 *
 * This should be used in compilation units that are less sensitive to code
 * size.
 */
export class EventInfoWrapper {
    constructor(eventInfo) {
        this.eventInfo = eventInfo;
    }
    getEventType() {
        return getEventType(this.eventInfo);
    }
    setEventType(eventType) {
        setEventType(this.eventInfo, eventType);
    }
    getEvent() {
        return getEvent(this.eventInfo);
    }
    setEvent(event) {
        setEvent(this.eventInfo, event);
    }
    getTargetElement() {
        return getTargetElement(this.eventInfo);
    }
    setTargetElement(targetElement) {
        setTargetElement(this.eventInfo, targetElement);
    }
    getContainer() {
        return getContainer(this.eventInfo);
    }
    setContainer(container) {
        setContainer(this.eventInfo, container);
    }
    getTimestamp() {
        return getTimestamp(this.eventInfo);
    }
    setTimestamp(timestamp) {
        setTimestamp(this.eventInfo, timestamp);
    }
    getAction() {
        const action = getAction(this.eventInfo);
        if (!action)
            return undefined;
        return {
            name: action[0],
            element: action[1],
        };
    }
    setAction(action) {
        if (!action) {
            unsetAction(this.eventInfo);
            return;
        }
        setAction(this.eventInfo, action.name, action.element);
    }
    getIsReplay() {
        return getIsReplay(this.eventInfo);
    }
    setIsReplay(replay) {
        setIsReplay(this.eventInfo, replay);
    }
    getResolved() {
        return getResolved(this.eventInfo);
    }
    setResolved(resolved) {
        setResolved(this.eventInfo, resolved);
    }
    clone() {
        return new EventInfoWrapper(cloneEventInfo(this.eventInfo));
    }
}
//# sourceMappingURL=data:application/json;base64,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