/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { consumerAfterComputation, consumerBeforeComputation, consumerDestroy, consumerMarkDirty, consumerPollProducersForChange, isInNotificationPhase, REACTIVE_NODE, SIGNAL, } from './graph';
export function createWatch(fn, schedule, allowSignalWrites) {
    const node = Object.create(WATCH_NODE);
    if (allowSignalWrites) {
        node.consumerAllowSignalWrites = true;
    }
    node.fn = fn;
    node.schedule = schedule;
    const registerOnCleanup = (cleanupFn) => {
        node.cleanupFn = cleanupFn;
    };
    function isWatchNodeDestroyed(node) {
        return node.fn === null && node.schedule === null;
    }
    function destroyWatchNode(node) {
        if (!isWatchNodeDestroyed(node)) {
            consumerDestroy(node); // disconnect watcher from the reactive graph
            node.cleanupFn();
            // nullify references to the integration functions to mark node as destroyed
            node.fn = null;
            node.schedule = null;
            node.cleanupFn = NOOP_CLEANUP_FN;
        }
    }
    const run = () => {
        if (node.fn === null) {
            // trying to run a destroyed watch is noop
            return;
        }
        if (isInNotificationPhase()) {
            throw new Error(`Schedulers cannot synchronously execute watches while scheduling.`);
        }
        node.dirty = false;
        if (node.hasRun && !consumerPollProducersForChange(node)) {
            return;
        }
        node.hasRun = true;
        const prevConsumer = consumerBeforeComputation(node);
        try {
            node.cleanupFn();
            node.cleanupFn = NOOP_CLEANUP_FN;
            node.fn(registerOnCleanup);
        }
        finally {
            consumerAfterComputation(node, prevConsumer);
        }
    };
    node.ref = {
        notify: () => consumerMarkDirty(node),
        run,
        cleanup: () => node.cleanupFn(),
        destroy: () => destroyWatchNode(node),
        [SIGNAL]: node,
    };
    return node.ref;
}
const NOOP_CLEANUP_FN = () => { };
// Note: Using an IIFE here to ensure that the spread assignment is not considered
// a side-effect, ending up preserving `COMPUTED_NODE` and `REACTIVE_NODE`.
// TODO: remove when https://github.com/evanw/esbuild/issues/3392 is resolved.
const WATCH_NODE = /* @__PURE__ */ (() => {
    return {
        ...REACTIVE_NODE,
        consumerIsAlwaysLive: true,
        consumerAllowSignalWrites: false,
        consumerMarkedDirty: (node) => {
            if (node.schedule !== null) {
                node.schedule(node.ref);
            }
        },
        hasRun: false,
        cleanupFn: NOOP_CLEANUP_FN,
    };
})();
//# sourceMappingURL=data:application/json;base64,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