/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { assertInInjectionContext, DestroyRef, inject } from '@angular/core';
import { Observable } from 'rxjs';
import { takeUntil } from 'rxjs/operators';
/**
 * Operator which completes the Observable when the calling context (component, directive, service,
 * etc) is destroyed.
 *
 * @param destroyRef optionally, the `DestroyRef` representing the current context. This can be
 *     passed explicitly to use `takeUntilDestroyed` outside of an [injection
 * context](guide/di/dependency-injection-context). Otherwise, the current `DestroyRef` is injected.
 *
 * @developerPreview
 */
export function takeUntilDestroyed(destroyRef) {
    if (!destroyRef) {
        assertInInjectionContext(takeUntilDestroyed);
        destroyRef = inject(DestroyRef);
    }
    const destroyed$ = new Observable((observer) => {
        const unregisterFn = destroyRef.onDestroy(observer.next.bind(observer));
        return unregisterFn;
    });
    return (source) => {
        return source.pipe(takeUntil(destroyed$));
    };
}
//# sourceMappingURL=data:application/json;base64,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