/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { producerAccessed, SIGNAL } from '@angular/core/primitives/signals';
import { RuntimeError } from '../../errors';
import { INPUT_SIGNAL_NODE, REQUIRED_UNSET_VALUE } from './input_signal_node';
export const ɵINPUT_SIGNAL_BRAND_READ_TYPE = /* @__PURE__ */ Symbol();
export const ɵINPUT_SIGNAL_BRAND_WRITE_TYPE = /* @__PURE__ */ Symbol();
/**
 * Creates an input signal.
 *
 * @param initialValue The initial value.
 *   Can be set to {@link REQUIRED_UNSET_VALUE} for required inputs.
 * @param options Additional options for the input. e.g. a transform, or an alias.
 */
export function createInputSignal(initialValue, options) {
    const node = Object.create(INPUT_SIGNAL_NODE);
    node.value = initialValue;
    // Perf note: Always set `transformFn` here to ensure that `node` always
    // has the same v8 class shape, allowing monomorphic reads on input signals.
    node.transformFn = options?.transform;
    function inputValueFn() {
        // Record that someone looked at this signal.
        producerAccessed(node);
        if (node.value === REQUIRED_UNSET_VALUE) {
            throw new RuntimeError(-950 /* RuntimeErrorCode.REQUIRED_INPUT_NO_VALUE */, ngDevMode && 'Input is required but no value is available yet.');
        }
        return node.value;
    }
    inputValueFn[SIGNAL] = node;
    if (ngDevMode) {
        inputValueFn.toString = () => `[Input Signal: ${inputValueFn()}]`;
    }
    return inputValueFn;
}
//# sourceMappingURL=data:application/json;base64,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