/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { assertInInjectionContext } from '../di';
import { createMultiResultQuerySignalFn, createSingleResultOptionalQuerySignalFn, createSingleResultRequiredQuerySignalFn, } from '../render3/query_reactive';
function viewChildFn(locator, opts) {
    ngDevMode && assertInInjectionContext(viewChild);
    return createSingleResultOptionalQuerySignalFn();
}
function viewChildRequiredFn(locator, opts) {
    ngDevMode && assertInInjectionContext(viewChild);
    return createSingleResultRequiredQuerySignalFn();
}
/**
 * Initializes a view child query.
 *
 * Consider using `viewChild.required` for queries that should always match.
 *
 * @usageNotes
 * Create a child query in your component by declaring a
 * class field and initializing it with the `viewChild()` function.
 *
 * ```ts
 * @Component({template: '<div #el></div><my-component #cmp />'})
 * export class TestComponent {
 *   divEl = viewChild<ElementRef>('el');                   // Signal<ElementRef|undefined>
 *   divElRequired = viewChild.required<ElementRef>('el');  // Signal<ElementRef>
 *   cmp = viewChild(MyComponent);                          // Signal<MyComponent|undefined>
 *   cmpRequired = viewChild.required(MyComponent);         // Signal<MyComponent>
 * }
 * ```
 *
 * @developerPreview
 * @initializerApiFunction
 */
export const viewChild = (() => {
    // Note: This may be considered a side-effect, but nothing will depend on
    // this assignment, unless this `viewChild` constant export is accessed. It's a
    // self-contained side effect that is local to the user facing `viewChild` export.
    viewChildFn.required = viewChildRequiredFn;
    return viewChildFn;
})();
/**
 * Initializes a view children query.
 *
 * Query results are represented as a signal of a read-only collection containing all matched
 * elements.
 *
 * @usageNotes
 * Create a children query in your component by declaring a
 * class field and initializing it with the `viewChildren()` function.
 *
 * ```ts
 * @Component({...})
 * export class TestComponent {
 *   divEls = viewChildren<ElementRef>('el');   // Signal<ReadonlyArray<ElementRef>>
 * }
 * ```
 *
 * @initializerApiFunction
 * @developerPreview
 */
export function viewChildren(locator, opts) {
    ngDevMode && assertInInjectionContext(viewChildren);
    return createMultiResultQuerySignalFn();
}
export function contentChildFn(locator, opts) {
    ngDevMode && assertInInjectionContext(contentChild);
    return createSingleResultOptionalQuerySignalFn();
}
function contentChildRequiredFn(locator, opts) {
    ngDevMode && assertInInjectionContext(contentChildren);
    return createSingleResultRequiredQuerySignalFn();
}
/**
 * Initializes a content child query. Consider using `contentChild.required` for queries that should
 * always match.
 *
 * @usageNotes
 * Create a child query in your component by declaring a
 * class field and initializing it with the `contentChild()` function.
 *
 * ```ts
 * @Component({...})
 * export class TestComponent {
 *   headerEl = contentChild<ElementRef>('h');                    // Signal<ElementRef|undefined>
 *   headerElElRequired = contentChild.required<ElementRef>('h'); // Signal<ElementRef>
 *   header = contentChild(MyHeader);                             // Signal<MyHeader|undefined>
 *   headerRequired = contentChild.required(MyHeader);            // Signal<MyHeader>
 * }
 * ```
 *
 * @initializerApiFunction
 * @developerPreview
 */
export const contentChild = (() => {
    // Note: This may be considered a side-effect, but nothing will depend on
    // this assignment, unless this `viewChild` constant export is accessed. It's a
    // self-contained side effect that is local to the user facing `viewChild` export.
    contentChildFn.required = contentChildRequiredFn;
    return contentChildFn;
})();
/**
 * Initializes a content children query.
 *
 * Query results are represented as a signal of a read-only collection containing all matched
 * elements.
 *
 * @usageNotes
 * Create a children query in your component by declaring a
 * class field and initializing it with the `contentChildren()` function.
 *
 * ```ts
 * @Component({...})
 * export class TestComponent {
 *   headerEl = contentChildren<ElementRef>('h');   // Signal<ReadonlyArray<ElementRef>>
 * }
 * ```
 *
 * @initializerApiFunction
 * @developerPreview
 */
export function contentChildren(locator, opts) {
    return createMultiResultQuerySignalFn();
}
//# sourceMappingURL=data:application/json;base64,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