/*!
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { PREFETCH_TRIGGER_CLEANUP_FNS, TRIGGER_CLEANUP_FNS, } from './interfaces';
/**
 * Registers a cleanup function associated with a prefetching trigger
 * or a regular trigger of a defer block.
 */
export function storeTriggerCleanupFn(type, lDetails, cleanupFn) {
    const key = type === 1 /* TriggerType.Prefetch */ ? PREFETCH_TRIGGER_CLEANUP_FNS : TRIGGER_CLEANUP_FNS;
    if (lDetails[key] === null) {
        lDetails[key] = [];
    }
    lDetails[key].push(cleanupFn);
}
/**
 * Invokes registered cleanup functions either for prefetch or for regular triggers.
 */
export function invokeTriggerCleanupFns(type, lDetails) {
    const key = type === 1 /* TriggerType.Prefetch */ ? PREFETCH_TRIGGER_CLEANUP_FNS : TRIGGER_CLEANUP_FNS;
    const cleanupFns = lDetails[key];
    if (cleanupFns !== null) {
        for (const cleanupFn of cleanupFns) {
            cleanupFn();
        }
        lDetails[key] = null;
    }
}
/**
 * Invokes registered cleanup functions for both prefetch and regular triggers.
 */
export function invokeAllTriggerCleanupFns(lDetails) {
    invokeTriggerCleanupFns(1 /* TriggerType.Prefetch */, lDetails);
    invokeTriggerCleanupFns(0 /* TriggerType.Regular */, lDetails);
}
//# sourceMappingURL=data:application/json;base64,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