/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import '../util/ng_dev_mode';
import { RuntimeError } from '../errors';
import { emitInjectEvent } from '../render3/debug/injector_profiler';
import { stringify } from '../util/stringify';
import { resolveForwardRef } from './forward_ref';
import { getInjectImplementation, injectRootLimpMode } from './inject_switch';
import { InjectFlags, } from './interface/injector';
const _THROW_IF_NOT_FOUND = {};
export const THROW_IF_NOT_FOUND = _THROW_IF_NOT_FOUND;
/*
 * Name of a property (that we patch onto DI decorator), which is used as an annotation of which
 * InjectFlag this decorator represents. This allows to avoid direct references to the DI decorators
 * in the code, thus making them tree-shakable.
 */
const DI_DECORATOR_FLAG = '__NG_DI_FLAG__';
export const NG_TEMP_TOKEN_PATH = 'ngTempTokenPath';
const NG_TOKEN_PATH = 'ngTokenPath';
const NEW_LINE = /\n/gm;
const NO_NEW_LINE = 'ɵ';
export const SOURCE = '__source';
/**
 * Current injector value used by `inject`.
 * - `undefined`: it is an error to call `inject`
 * - `null`: `inject` can be called but there is no injector (limp-mode).
 * - Injector instance: Use the injector for resolution.
 */
let _currentInjector = undefined;
export function getCurrentInjector() {
    return _currentInjector;
}
export function setCurrentInjector(injector) {
    const former = _currentInjector;
    _currentInjector = injector;
    return former;
}
export function injectInjectorOnly(token, flags = InjectFlags.Default) {
    if (_currentInjector === undefined) {
        throw new RuntimeError(-203 /* RuntimeErrorCode.MISSING_INJECTION_CONTEXT */, ngDevMode &&
            `inject() must be called from an injection context such as a constructor, a factory function, a field initializer, or a function used with \`runInInjectionContext\`.`);
    }
    else if (_currentInjector === null) {
        return injectRootLimpMode(token, undefined, flags);
    }
    else {
        const value = _currentInjector.get(token, flags & InjectFlags.Optional ? null : undefined, flags);
        ngDevMode && emitInjectEvent(token, value, flags);
        return value;
    }
}
export function ɵɵinject(token, flags = InjectFlags.Default) {
    return (getInjectImplementation() || injectInjectorOnly)(resolveForwardRef(token), flags);
}
/**
 * Throws an error indicating that a factory function could not be generated by the compiler for a
 * particular class.
 *
 * The name of the class is not mentioned here, but will be in the generated factory function name
 * and thus in the stack trace.
 *
 * @codeGenApi
 */
export function ɵɵinvalidFactoryDep(index) {
    throw new RuntimeError(202 /* RuntimeErrorCode.INVALID_FACTORY_DEPENDENCY */, ngDevMode &&
        `This constructor is not compatible with Angular Dependency Injection because its dependency at index ${index} of the parameter list is invalid.
This can happen if the dependency type is a primitive like a string or if an ancestor of this class is missing an Angular decorator.

Please check that 1) the type for the parameter at index ${index} is correct and 2) the correct Angular decorators are defined for this class and its ancestors.`);
}
/**
 * Injects a token from the currently active injector.
 * `inject` is only supported in an [injection context](guide/di/dependency-injection-context). It
 * can be used during:
 * - Construction (via the `constructor`) of a class being instantiated by the DI system, such
 * as an `@Injectable` or `@Component`.
 * - In the initializer for fields of such classes.
 * - In the factory function specified for `useFactory` of a `Provider` or an `@Injectable`.
 * - In the `factory` function specified for an `InjectionToken`.
 * - In a stackframe of a function call in a DI context
 *
 * @param token A token that represents a dependency that should be injected.
 * @param flags Optional flags that control how injection is executed.
 * The flags correspond to injection strategies that can be specified with
 * parameter decorators `@Host`, `@Self`, `@SkipSelf`, and `@Optional`.
 * @returns the injected value if operation is successful, `null` otherwise.
 * @throws if called outside of a supported context.
 *
 * @usageNotes
 * In practice the `inject()` calls are allowed in a constructor, a constructor parameter and a
 * field initializer:
 *
 * ```typescript
 * @Injectable({providedIn: 'root'})
 * export class Car {
 *   radio: Radio|undefined;
 *   // OK: field initializer
 *   spareTyre = inject(Tyre);
 *
 *   constructor() {
 *     // OK: constructor body
 *     this.radio = inject(Radio);
 *   }
 * }
 * ```
 *
 * It is also legal to call `inject` from a provider's factory:
 *
 * ```typescript
 * providers: [
 *   {provide: Car, useFactory: () => {
 *     // OK: a class factory
 *     const engine = inject(Engine);
 *     return new Car(engine);
 *   }}
 * ]
 * ```
 *
 * Calls to the `inject()` function outside of the class creation context will result in error. Most
 * notably, calls to `inject()` are disallowed after a class instance was created, in methods
 * (including lifecycle hooks):
 *
 * ```typescript
 * @Component({ ... })
 * export class CarComponent {
 *   ngOnInit() {
 *     // ERROR: too late, the component instance was already created
 *     const engine = inject(Engine);
 *     engine.start();
 *   }
 * }
 * ```
 *
 * @publicApi
 */
export function inject(token, flags = InjectFlags.Default) {
    // The `as any` here _shouldn't_ be necessary, but without it JSCompiler
    // throws a disambiguation  error due to the multiple signatures.
    return ɵɵinject(token, convertToBitFlags(flags));
}
// Converts object-based DI flags (`InjectOptions`) to bit flags (`InjectFlags`).
export function convertToBitFlags(flags) {
    if (typeof flags === 'undefined' || typeof flags === 'number') {
        return flags;
    }
    // While TypeScript doesn't accept it without a cast, bitwise OR with false-y values in
    // JavaScript is a no-op. We can use that for a very codesize-efficient conversion from
    // `InjectOptions` to `InjectFlags`.
    return (0 /* InternalInjectFlags.Default */ | // comment to force a line break in the formatter
        (flags.optional && 8 /* InternalInjectFlags.Optional */) |
        (flags.host && 1 /* InternalInjectFlags.Host */) |
        (flags.self && 2 /* InternalInjectFlags.Self */) |
        (flags.skipSelf && 4 /* InternalInjectFlags.SkipSelf */));
}
export function injectArgs(types) {
    const args = [];
    for (let i = 0; i < types.length; i++) {
        const arg = resolveForwardRef(types[i]);
        if (Array.isArray(arg)) {
            if (arg.length === 0) {
                throw new RuntimeError(900 /* RuntimeErrorCode.INVALID_DIFFER_INPUT */, ngDevMode && 'Arguments array must have arguments.');
            }
            let type = undefined;
            let flags = InjectFlags.Default;
            for (let j = 0; j < arg.length; j++) {
                const meta = arg[j];
                const flag = getInjectFlag(meta);
                if (typeof flag === 'number') {
                    // Special case when we handle @Inject decorator.
                    if (flag === -1 /* DecoratorFlags.Inject */) {
                        type = meta.token;
                    }
                    else {
                        flags |= flag;
                    }
                }
                else {
                    type = meta;
                }
            }
            args.push(ɵɵinject(type, flags));
        }
        else {
            args.push(ɵɵinject(arg));
        }
    }
    return args;
}
/**
 * Attaches a given InjectFlag to a given decorator using monkey-patching.
 * Since DI decorators can be used in providers `deps` array (when provider is configured using
 * `useFactory`) without initialization (e.g. `Host`) and as an instance (e.g. `new Host()`), we
 * attach the flag to make it available both as a static property and as a field on decorator
 * instance.
 *
 * @param decorator Provided DI decorator.
 * @param flag InjectFlag that should be applied.
 */
export function attachInjectFlag(decorator, flag) {
    decorator[DI_DECORATOR_FLAG] = flag;
    decorator.prototype[DI_DECORATOR_FLAG] = flag;
    return decorator;
}
/**
 * Reads monkey-patched property that contains InjectFlag attached to a decorator.
 *
 * @param token Token that may contain monkey-patched DI flags property.
 */
export function getInjectFlag(token) {
    return token[DI_DECORATOR_FLAG];
}
export function catchInjectorError(e, token, injectorErrorName, source) {
    const tokenPath = e[NG_TEMP_TOKEN_PATH];
    if (token[SOURCE]) {
        tokenPath.unshift(token[SOURCE]);
    }
    e.message = formatError('\n' + e.message, tokenPath, injectorErrorName, source);
    e[NG_TOKEN_PATH] = tokenPath;
    e[NG_TEMP_TOKEN_PATH] = null;
    throw e;
}
export function formatError(text, obj, injectorErrorName, source = null) {
    text = text && text.charAt(0) === '\n' && text.charAt(1) == NO_NEW_LINE ? text.slice(2) : text;
    let context = stringify(obj);
    if (Array.isArray(obj)) {
        context = obj.map(stringify).join(' -> ');
    }
    else if (typeof obj === 'object') {
        let parts = [];
        for (let key in obj) {
            if (obj.hasOwnProperty(key)) {
                let value = obj[key];
                parts.push(key + ':' + (typeof value === 'string' ? JSON.stringify(value) : stringify(value)));
            }
        }
        context = `{${parts.join(', ')}}`;
    }
    return `${injectorErrorName}${source ? '(' + source + ')' : ''}[${context}]: ${text.replace(NEW_LINE, '\n  ')}`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5qZWN0b3JfY29tcGF0aWJpbGl0eS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uLy4uL3BhY2thZ2VzL2NvcmUvc3JjL2RpL2luamVjdG9yX2NvbXBhdGliaWxpdHkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7OztHQU1HO0FBRUgsT0FBTyxxQkFBcUIsQ0FBQztBQUU3QixPQUFPLEVBQUMsWUFBWSxFQUFtQixNQUFNLFdBQVcsQ0FBQztBQUV6RCxPQUFPLEVBQUMsZUFBZSxFQUFDLE1BQU0sb0NBQW9DLENBQUM7QUFDbkUsT0FBTyxFQUFDLFNBQVMsRUFBQyxNQUFNLG1CQUFtQixDQUFDO0FBRTVDLE9BQU8sRUFBQyxpQkFBaUIsRUFBQyxNQUFNLGVBQWUsQ0FBQztBQUNoRCxPQUFPLEVBQUMsdUJBQXVCLEVBQUUsa0JBQWtCLEVBQUMsTUFBTSxpQkFBaUIsQ0FBQztBQUU1RSxPQUFPLEVBRUwsV0FBVyxHQUdaLE1BQU0sc0JBQXNCLENBQUM7QUFJOUIsTUFBTSxtQkFBbUIsR0FBRyxFQUFFLENBQUM7QUFDL0IsTUFBTSxDQUFDLE1BQU0sa0JBQWtCLEdBQUcsbUJBQW1CLENBQUM7QUFFdEQ7Ozs7R0FJRztBQUNILE1BQU0saUJBQWlCLEdBQUcsZ0JBQWdCLENBQUM7QUFFM0MsTUFBTSxDQUFDLE1BQU0sa0JBQWtCLEdBQUcsaUJBQWlCLENBQUM7QUFDcEQsTUFBTSxhQUFhLEdBQUcsYUFBYSxDQUFDO0FBQ3BDLE1BQU0sUUFBUSxHQUFHLE1BQU0sQ0FBQztBQUN4QixNQUFNLFdBQVcsR0FBRyxHQUFHLENBQUM7QUFDeEIsTUFBTSxDQUFDLE1BQU0sTUFBTSxHQUFHLFVBQVUsQ0FBQztBQUVqQzs7Ozs7R0FLRztBQUNILElBQUksZ0JBQWdCLEdBQWdDLFNBQVMsQ0FBQztBQUU5RCxNQUFNLFVBQVUsa0JBQWtCO0lBQ2hDLE9BQU8sZ0JBQWdCLENBQUM7QUFDMUIsQ0FBQztBQUVELE1BQU0sVUFBVSxrQkFBa0IsQ0FDaEMsUUFBcUM7SUFFckMsTUFBTSxNQUFNLEdBQUcsZ0JBQWdCLENBQUM7SUFDaEMsZ0JBQWdCLEdBQUcsUUFBUSxDQUFDO0lBQzVCLE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUM7QUFJRCxNQUFNLFVBQVUsa0JBQWtCLENBQ2hDLEtBQXVCLEVBQ3ZCLEtBQUssR0FBRyxXQUFXLENBQUMsT0FBTztJQUUzQixJQUFJLGdCQUFnQixLQUFLLFNBQVMsRUFBRSxDQUFDO1FBQ25DLE1BQU0sSUFBSSxZQUFZLHdEQUVwQixTQUFTO1lBQ1Asc0tBQXNLLENBQ3pLLENBQUM7SUFDSixDQUFDO1NBQU0sSUFBSSxnQkFBZ0IsS0FBSyxJQUFJLEVBQUUsQ0FBQztRQUNyQyxPQUFPLGtCQUFrQixDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDckQsQ0FBQztTQUFNLENBQUM7UUFDTixNQUFNLEtBQUssR0FBRyxnQkFBZ0IsQ0FBQyxHQUFHLENBQ2hDLEtBQUssRUFDTCxLQUFLLEdBQUcsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxTQUFTLEVBQy9DLEtBQUssQ0FDTixDQUFDO1FBQ0YsU0FBUyxJQUFJLGVBQWUsQ0FBQyxLQUFzQixFQUFFLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNuRSxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7QUFDSCxDQUFDO0FBb0JELE1BQU0sVUFBVSxRQUFRLENBQ3RCLEtBQTRDLEVBQzVDLEtBQUssR0FBRyxXQUFXLENBQUMsT0FBTztJQUUzQixPQUFPLENBQUMsdUJBQXVCLEVBQUUsSUFBSSxrQkFBa0IsQ0FBQyxDQUN0RCxpQkFBaUIsQ0FBQyxLQUFnQixDQUFDLEVBQ25DLEtBQUssQ0FDTixDQUFDO0FBQ0osQ0FBQztBQUVEOzs7Ozs7OztHQVFHO0FBQ0gsTUFBTSxVQUFVLG1CQUFtQixDQUFDLEtBQWE7SUFDL0MsTUFBTSxJQUFJLFlBQVksd0RBRXBCLFNBQVM7UUFDUCx3R0FBd0csS0FBSzs7OzJEQUd4RCxLQUFLLGlHQUFpRyxDQUM5SixDQUFDO0FBQ0osQ0FBQztBQXFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQWdFRztBQUNILE1BQU0sVUFBVSxNQUFNLENBQ3BCLEtBQTRDLEVBQzVDLFFBQXFDLFdBQVcsQ0FBQyxPQUFPO0lBRXhELHdFQUF3RTtJQUN4RSxpRUFBaUU7SUFDakUsT0FBTyxRQUFRLENBQUMsS0FBWSxFQUFFLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7QUFDMUQsQ0FBQztBQUVELGlGQUFpRjtBQUNqRixNQUFNLFVBQVUsaUJBQWlCLENBQy9CLEtBQThDO0lBRTlDLElBQUksT0FBTyxLQUFLLEtBQUssV0FBVyxJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsRUFBRSxDQUFDO1FBQzlELE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUVELHVGQUF1RjtJQUN2Rix1RkFBdUY7SUFDdkYsb0NBQW9DO0lBQ3BDLE9BQU8sQ0FBQyxzQ0FBOEIsaURBQWlEO1FBQ3BGLENBQUMsS0FBSyxDQUFDLFFBQVEsd0NBQWdDLENBQVk7UUFDM0QsQ0FBQyxLQUFLLENBQUMsSUFBSSxvQ0FBNEIsQ0FBWTtRQUNuRCxDQUFDLEtBQUssQ0FBQyxJQUFJLG9DQUE0QixDQUFZO1FBQ25ELENBQUMsS0FBSyxDQUFDLFFBQVEsd0NBQWdDLENBQVksQ0FBZ0IsQ0FBQztBQUNqRixDQUFDO0FBRUQsTUFBTSxVQUFVLFVBQVUsQ0FBQyxLQUFxQztJQUM5RCxNQUFNLElBQUksR0FBVSxFQUFFLENBQUM7SUFDdkIsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQztRQUN0QyxNQUFNLEdBQUcsR0FBRyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN4QyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQztZQUN2QixJQUFJLEdBQUcsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFLENBQUM7Z0JBQ3JCLE1BQU0sSUFBSSxZQUFZLGtEQUVwQixTQUFTLElBQUksc0NBQXNDLENBQ3BELENBQUM7WUFDSixDQUFDO1lBQ0QsSUFBSSxJQUFJLEdBQTBCLFNBQVMsQ0FBQztZQUM1QyxJQUFJLEtBQUssR0FBZ0IsV0FBVyxDQUFDLE9BQU8sQ0FBQztZQUU3QyxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsR0FBRyxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDO2dCQUNwQyxNQUFNLElBQUksR0FBRyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3BCLE1BQU0sSUFBSSxHQUFHLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFDakMsSUFBSSxPQUFPLElBQUksS0FBSyxRQUFRLEVBQUUsQ0FBQztvQkFDN0IsaURBQWlEO29CQUNqRCxJQUFJLElBQUksbUNBQTBCLEVBQUUsQ0FBQzt3QkFDbkMsSUFBSSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUM7b0JBQ3BCLENBQUM7eUJBQU0sQ0FBQzt3QkFDTixLQUFLLElBQUksSUFBSSxDQUFDO29CQUNoQixDQUFDO2dCQUNILENBQUM7cUJBQU0sQ0FBQztvQkFDTixJQUFJLEdBQUcsSUFBSSxDQUFDO2dCQUNkLENBQUM7WUFDSCxDQUFDO1lBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSyxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDcEMsQ0FBQzthQUFNLENBQUM7WUFDTixJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQzNCLENBQUM7SUFDSCxDQUFDO0lBQ0QsT0FBTyxJQUFJLENBQUM7QUFDZCxDQUFDO0FBRUQ7Ozs7Ozs7OztHQVNHO0FBQ0gsTUFBTSxVQUFVLGdCQUFnQixDQUFDLFNBQWMsRUFBRSxJQUEwQztJQUN6RixTQUFTLENBQUMsaUJBQWlCLENBQUMsR0FBRyxJQUFJLENBQUM7SUFDcEMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLElBQUksQ0FBQztJQUM5QyxPQUFPLFNBQVMsQ0FBQztBQUNuQixDQUFDO0FBRUQ7Ozs7R0FJRztBQUNILE1BQU0sVUFBVSxhQUFhLENBQUMsS0FBVTtJQUN0QyxPQUFPLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0FBQ2xDLENBQUM7QUFFRCxNQUFNLFVBQVUsa0JBQWtCLENBQ2hDLENBQU0sRUFDTixLQUFVLEVBQ1YsaUJBQXlCLEVBQ3pCLE1BQXFCO0lBRXJCLE1BQU0sU0FBUyxHQUFVLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQy9DLElBQUksS0FBSyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUM7UUFDbEIsU0FBUyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBQ0QsQ0FBQyxDQUFDLE9BQU8sR0FBRyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxPQUFPLEVBQUUsU0FBUyxFQUFFLGlCQUFpQixFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQ2hGLENBQUMsQ0FBQyxhQUFhLENBQUMsR0FBRyxTQUFTLENBQUM7SUFDN0IsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLEdBQUcsSUFBSSxDQUFDO0lBQzdCLE1BQU0sQ0FBQyxDQUFDO0FBQ1YsQ0FBQztBQUVELE1BQU0sVUFBVSxXQUFXLENBQ3pCLElBQVksRUFDWixHQUFRLEVBQ1IsaUJBQXlCLEVBQ3pCLFNBQXdCLElBQUk7SUFFNUIsSUFBSSxHQUFHLElBQUksSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxLQUFLLElBQUksSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxJQUFJLFdBQVcsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO0lBQy9GLElBQUksT0FBTyxHQUFHLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUM3QixJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQztRQUN2QixPQUFPLEdBQUcsR0FBRyxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDNUMsQ0FBQztTQUFNLElBQUksT0FBTyxHQUFHLEtBQUssUUFBUSxFQUFFLENBQUM7UUFDbkMsSUFBSSxLQUFLLEdBQWEsRUFBRSxDQUFDO1FBQ3pCLEtBQUssSUFBSSxHQUFHLElBQUksR0FBRyxFQUFFLENBQUM7WUFDcEIsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUM7Z0JBQzVCLElBQUksS0FBSyxHQUFHLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDckIsS0FBSyxDQUFDLElBQUksQ0FDUixHQUFHLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxLQUFLLEtBQUssUUFBUSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FDbkYsQ0FBQztZQUNKLENBQUM7UUFDSCxDQUFDO1FBQ0QsT0FBTyxHQUFHLElBQUksS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDO0lBQ3BDLENBQUM7SUFDRCxPQUFPLEdBQUcsaUJBQWlCLEdBQUcsTUFBTSxDQUFDLENBQUMsQ0FBQyxHQUFHLEdBQUcsTUFBTSxHQUFHLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxJQUFJLE9BQU8sTUFBTSxJQUFJLENBQUMsT0FBTyxDQUN6RixRQUFRLEVBQ1IsTUFBTSxDQUNQLEVBQUUsQ0FBQztBQUNOLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIEBsaWNlbnNlXG4gKiBDb3B5cmlnaHQgR29vZ2xlIExMQyBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqIFVzZSBvZiB0aGlzIHNvdXJjZSBjb2RlIGlzIGdvdmVybmVkIGJ5IGFuIE1JVC1zdHlsZSBsaWNlbnNlIHRoYXQgY2FuIGJlXG4gKiBmb3VuZCBpbiB0aGUgTElDRU5TRSBmaWxlIGF0IGh0dHBzOi8vYW5ndWxhci5pby9saWNlbnNlXG4gKi9cblxuaW1wb3J0ICcuLi91dGlsL25nX2Rldl9tb2RlJztcblxuaW1wb3J0IHtSdW50aW1lRXJyb3IsIFJ1bnRpbWVFcnJvckNvZGV9IGZyb20gJy4uL2Vycm9ycyc7XG5pbXBvcnQge1R5cGV9IGZyb20gJy4uL2ludGVyZmFjZS90eXBlJztcbmltcG9ydCB7ZW1pdEluamVjdEV2ZW50fSBmcm9tICcuLi9yZW5kZXIzL2RlYnVnL2luamVjdG9yX3Byb2ZpbGVyJztcbmltcG9ydCB7c3RyaW5naWZ5fSBmcm9tICcuLi91dGlsL3N0cmluZ2lmeSc7XG5cbmltcG9ydCB7cmVzb2x2ZUZvcndhcmRSZWZ9IGZyb20gJy4vZm9yd2FyZF9yZWYnO1xuaW1wb3J0IHtnZXRJbmplY3RJbXBsZW1lbnRhdGlvbiwgaW5qZWN0Um9vdExpbXBNb2RlfSBmcm9tICcuL2luamVjdF9zd2l0Y2gnO1xuaW1wb3J0IHR5cGUge0luamVjdG9yfSBmcm9tICcuL2luamVjdG9yJztcbmltcG9ydCB7XG4gIERlY29yYXRvckZsYWdzLFxuICBJbmplY3RGbGFncyxcbiAgSW5qZWN0T3B0aW9ucyxcbiAgSW50ZXJuYWxJbmplY3RGbGFncyxcbn0gZnJvbSAnLi9pbnRlcmZhY2UvaW5qZWN0b3InO1xuaW1wb3J0IHtQcm92aWRlclRva2VufSBmcm9tICcuL3Byb3ZpZGVyX3Rva2VuJztcbmltcG9ydCB0eXBlIHtIb3N0QXR0cmlidXRlVG9rZW59IGZyb20gJy4vaG9zdF9hdHRyaWJ1dGVfdG9rZW4nO1xuXG5jb25zdCBfVEhST1dfSUZfTk9UX0ZPVU5EID0ge307XG5leHBvcnQgY29uc3QgVEhST1dfSUZfTk9UX0ZPVU5EID0gX1RIUk9XX0lGX05PVF9GT1VORDtcblxuLypcbiAqIE5hbWUgb2YgYSBwcm9wZXJ0eSAodGhhdCB3ZSBwYXRjaCBvbnRvIERJIGRlY29yYXRvciksIHdoaWNoIGlzIHVzZWQgYXMgYW4gYW5ub3RhdGlvbiBvZiB3aGljaFxuICogSW5qZWN0RmxhZyB0aGlzIGRlY29yYXRvciByZXByZXNlbnRzLiBUaGlzIGFsbG93cyB0byBhdm9pZCBkaXJlY3QgcmVmZXJlbmNlcyB0byB0aGUgREkgZGVjb3JhdG9yc1xuICogaW4gdGhlIGNvZGUsIHRodXMgbWFraW5nIHRoZW0gdHJlZS1zaGFrYWJsZS5cbiAqL1xuY29uc3QgRElfREVDT1JBVE9SX0ZMQUcgPSAnX19OR19ESV9GTEFHX18nO1xuXG5leHBvcnQgY29uc3QgTkdfVEVNUF9UT0tFTl9QQVRIID0gJ25nVGVtcFRva2VuUGF0aCc7XG5jb25zdCBOR19UT0tFTl9QQVRIID0gJ25nVG9rZW5QYXRoJztcbmNvbnN0IE5FV19MSU5FID0gL1xcbi9nbTtcbmNvbnN0IE5PX05FV19MSU5FID0gJ8m1JztcbmV4cG9ydCBjb25zdCBTT1VSQ0UgPSAnX19zb3VyY2UnO1xuXG4vKipcbiAqIEN1cnJlbnQgaW5qZWN0b3IgdmFsdWUgdXNlZCBieSBgaW5qZWN0YC5cbiAqIC0gYHVuZGVmaW5lZGA6IGl0IGlzIGFuIGVycm9yIHRvIGNhbGwgYGluamVjdGBcbiAqIC0gYG51bGxgOiBgaW5qZWN0YCBjYW4gYmUgY2FsbGVkIGJ1dCB0aGVyZSBpcyBubyBpbmplY3RvciAobGltcC1tb2RlKS5cbiAqIC0gSW5qZWN0b3IgaW5zdGFuY2U6IFVzZSB0aGUgaW5qZWN0b3IgZm9yIHJlc29sdXRpb24uXG4gKi9cbmxldCBfY3VycmVudEluamVjdG9yOiBJbmplY3RvciB8IHVuZGVmaW5lZCB8IG51bGwgPSB1bmRlZmluZWQ7XG5cbmV4cG9ydCBmdW5jdGlvbiBnZXRDdXJyZW50SW5qZWN0b3IoKTogSW5qZWN0b3IgfCB1bmRlZmluZWQgfCBudWxsIHtcbiAgcmV0dXJuIF9jdXJyZW50SW5qZWN0b3I7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBzZXRDdXJyZW50SW5qZWN0b3IoXG4gIGluamVjdG9yOiBJbmplY3RvciB8IG51bGwgfCB1bmRlZmluZWQsXG4pOiBJbmplY3RvciB8IHVuZGVmaW5lZCB8IG51bGwge1xuICBjb25zdCBmb3JtZXIgPSBfY3VycmVudEluamVjdG9yO1xuICBfY3VycmVudEluamVjdG9yID0gaW5qZWN0b3I7XG4gIHJldHVybiBmb3JtZXI7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpbmplY3RJbmplY3Rvck9ubHk8VD4odG9rZW46IFByb3ZpZGVyVG9rZW48VD4pOiBUO1xuZXhwb3J0IGZ1bmN0aW9uIGluamVjdEluamVjdG9yT25seTxUPih0b2tlbjogUHJvdmlkZXJUb2tlbjxUPiwgZmxhZ3M/OiBJbmplY3RGbGFncyk6IFQgfCBudWxsO1xuZXhwb3J0IGZ1bmN0aW9uIGluamVjdEluamVjdG9yT25seTxUPihcbiAgdG9rZW46IFByb3ZpZGVyVG9rZW48VD4sXG4gIGZsYWdzID0gSW5qZWN0RmxhZ3MuRGVmYXVsdCxcbik6IFQgfCBudWxsIHtcbiAgaWYgKF9jdXJyZW50SW5qZWN0b3IgPT09IHVuZGVmaW5lZCkge1xuICAgIHRocm93IG5ldyBSdW50aW1lRXJyb3IoXG4gICAgICBSdW50aW1lRXJyb3JDb2RlLk1JU1NJTkdfSU5KRUNUSU9OX0NPTlRFWFQsXG4gICAgICBuZ0Rldk1vZGUgJiZcbiAgICAgICAgYGluamVjdCgpIG11c3QgYmUgY2FsbGVkIGZyb20gYW4gaW5qZWN0aW9uIGNvbnRleHQgc3VjaCBhcyBhIGNvbnN0cnVjdG9yLCBhIGZhY3RvcnkgZnVuY3Rpb24sIGEgZmllbGQgaW5pdGlhbGl6ZXIsIG9yIGEgZnVuY3Rpb24gdXNlZCB3aXRoIFxcYHJ1bkluSW5qZWN0aW9uQ29udGV4dFxcYC5gLFxuICAgICk7XG4gIH0gZWxzZSBpZiAoX2N1cnJlbnRJbmplY3RvciA9PT0gbnVsbCkge1xuICAgIHJldHVybiBpbmplY3RSb290TGltcE1vZGUodG9rZW4sIHVuZGVmaW5lZCwgZmxhZ3MpO1xuICB9IGVsc2Uge1xuICAgIGNvbnN0IHZhbHVlID0gX2N1cnJlbnRJbmplY3Rvci5nZXQoXG4gICAgICB0b2tlbixcbiAgICAgIGZsYWdzICYgSW5qZWN0RmxhZ3MuT3B0aW9uYWwgPyBudWxsIDogdW5kZWZpbmVkLFxuICAgICAgZmxhZ3MsXG4gICAgKTtcbiAgICBuZ0Rldk1vZGUgJiYgZW1pdEluamVjdEV2ZW50KHRva2VuIGFzIFR5cGU8dW5rbm93bj4sIHZhbHVlLCBmbGFncyk7XG4gICAgcmV0dXJuIHZhbHVlO1xuICB9XG59XG5cbi8qKlxuICogR2VuZXJhdGVkIGluc3RydWN0aW9uOiBpbmplY3RzIGEgdG9rZW4gZnJvbSB0aGUgY3VycmVudGx5IGFjdGl2ZSBpbmplY3Rvci5cbiAqXG4gKiAoQWRkaXRpb25hbCBkb2N1bWVudGF0aW9uIG1vdmVkIHRvIGBpbmplY3RgLCBhcyBpdCBpcyB0aGUgcHVibGljIEFQSSwgYW5kIGFuIGFsaWFzIGZvciB0aGlzXG4gKiBpbnN0cnVjdGlvbilcbiAqXG4gKiBAc2VlIGluamVjdFxuICogQGNvZGVHZW5BcGlcbiAqIEBwdWJsaWNBcGkgVGhpcyBpbnN0cnVjdGlvbiBoYXMgYmVlbiBlbWl0dGVkIGJ5IFZpZXdFbmdpbmUgZm9yIHNvbWUgdGltZSBhbmQgaXMgZGVwbG95ZWQgdG8gbnBtLlxuICovXG5leHBvcnQgZnVuY3Rpb24gybXJtWluamVjdDxUPih0b2tlbjogUHJvdmlkZXJUb2tlbjxUPik6IFQ7XG5leHBvcnQgZnVuY3Rpb24gybXJtWluamVjdDxUPih0b2tlbjogUHJvdmlkZXJUb2tlbjxUPiwgZmxhZ3M/OiBJbmplY3RGbGFncyk6IFQgfCBudWxsO1xuZXhwb3J0IGZ1bmN0aW9uIMm1ybVpbmplY3QodG9rZW46IEhvc3RBdHRyaWJ1dGVUb2tlbik6IHN0cmluZztcbmV4cG9ydCBmdW5jdGlvbiDJtcm1aW5qZWN0KHRva2VuOiBIb3N0QXR0cmlidXRlVG9rZW4sIGZsYWdzPzogSW5qZWN0RmxhZ3MpOiBzdHJpbmcgfCBudWxsO1xuZXhwb3J0IGZ1bmN0aW9uIMm1ybVpbmplY3Q8VD4oXG4gIHRva2VuOiBQcm92aWRlclRva2VuPFQ+IHwgSG9zdEF0dHJpYnV0ZVRva2VuLFxuICBmbGFncz86IEluamVjdEZsYWdzLFxuKTogc3RyaW5nIHwgbnVsbDtcbmV4cG9ydCBmdW5jdGlvbiDJtcm1aW5qZWN0PFQ+KFxuICB0b2tlbjogUHJvdmlkZXJUb2tlbjxUPiB8IEhvc3RBdHRyaWJ1dGVUb2tlbixcbiAgZmxhZ3MgPSBJbmplY3RGbGFncy5EZWZhdWx0LFxuKTogVCB8IG51bGwge1xuICByZXR1cm4gKGdldEluamVjdEltcGxlbWVudGF0aW9uKCkgfHwgaW5qZWN0SW5qZWN0b3JPbmx5KShcbiAgICByZXNvbHZlRm9yd2FyZFJlZih0b2tlbiBhcyBUeXBlPFQ+KSxcbiAgICBmbGFncyxcbiAgKTtcbn1cblxuLyoqXG4gKiBUaHJvd3MgYW4gZXJyb3IgaW5kaWNhdGluZyB0aGF0IGEgZmFjdG9yeSBmdW5jdGlvbiBjb3VsZCBub3QgYmUgZ2VuZXJhdGVkIGJ5IHRoZSBjb21waWxlciBmb3IgYVxuICogcGFydGljdWxhciBjbGFzcy5cbiAqXG4gKiBUaGUgbmFtZSBvZiB0aGUgY2xhc3MgaXMgbm90IG1lbnRpb25lZCBoZXJlLCBidXQgd2lsbCBiZSBpbiB0aGUgZ2VuZXJhdGVkIGZhY3RvcnkgZnVuY3Rpb24gbmFtZVxuICogYW5kIHRodXMgaW4gdGhlIHN0YWNrIHRyYWNlLlxuICpcbiAqIEBjb2RlR2VuQXBpXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiDJtcm1aW52YWxpZEZhY3RvcnlEZXAoaW5kZXg6IG51bWJlcik6IG5ldmVyIHtcbiAgdGhyb3cgbmV3IFJ1bnRpbWVFcnJvcihcbiAgICBSdW50aW1lRXJyb3JDb2RlLklOVkFMSURfRkFDVE9SWV9ERVBFTkRFTkNZLFxuICAgIG5nRGV2TW9kZSAmJlxuICAgICAgYFRoaXMgY29uc3RydWN0b3IgaXMgbm90IGNvbXBhdGlibGUgd2l0aCBBbmd1bGFyIERlcGVuZGVuY3kgSW5qZWN0aW9uIGJlY2F1c2UgaXRzIGRlcGVuZGVuY3kgYXQgaW5kZXggJHtpbmRleH0gb2YgdGhlIHBhcmFtZXRlciBsaXN0IGlzIGludmFsaWQuXG5UaGlzIGNhbiBoYXBwZW4gaWYgdGhlIGRlcGVuZGVuY3kgdHlwZSBpcyBhIHByaW1pdGl2ZSBsaWtlIGEgc3RyaW5nIG9yIGlmIGFuIGFuY2VzdG9yIG9mIHRoaXMgY2xhc3MgaXMgbWlzc2luZyBhbiBBbmd1bGFyIGRlY29yYXRvci5cblxuUGxlYXNlIGNoZWNrIHRoYXQgMSkgdGhlIHR5cGUgZm9yIHRoZSBwYXJhbWV0ZXIgYXQgaW5kZXggJHtpbmRleH0gaXMgY29ycmVjdCBhbmQgMikgdGhlIGNvcnJlY3QgQW5ndWxhciBkZWNvcmF0b3JzIGFyZSBkZWZpbmVkIGZvciB0aGlzIGNsYXNzIGFuZCBpdHMgYW5jZXN0b3JzLmAsXG4gICk7XG59XG5cbi8qKlxuICogQHBhcmFtIHRva2VuIEEgdG9rZW4gdGhhdCByZXByZXNlbnRzIGEgZGVwZW5kZW5jeSB0aGF0IHNob3VsZCBiZSBpbmplY3RlZC5cbiAqIEByZXR1cm5zIHRoZSBpbmplY3RlZCB2YWx1ZSBpZiBvcGVyYXRpb24gaXMgc3VjY2Vzc2Z1bCwgYG51bGxgIG90aGVyd2lzZS5cbiAqIEB0aHJvd3MgaWYgY2FsbGVkIG91dHNpZGUgb2YgYSBzdXBwb3J0ZWQgY29udGV4dC5cbiAqXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBpbmplY3Q8VD4odG9rZW46IFByb3ZpZGVyVG9rZW48VD4pOiBUO1xuLyoqXG4gKiBAcGFyYW0gdG9rZW4gQSB0b2tlbiB0aGF0IHJlcHJlc2VudHMgYSBkZXBlbmRlbmN5IHRoYXQgc2hvdWxkIGJlIGluamVjdGVkLlxuICogQHBhcmFtIGZsYWdzIENvbnRyb2wgaG93IGluamVjdGlvbiBpcyBleGVjdXRlZC4gVGhlIGZsYWdzIGNvcnJlc3BvbmQgdG8gaW5qZWN0aW9uIHN0cmF0ZWdpZXMgdGhhdFxuICogICAgIGNhbiBiZSBzcGVjaWZpZWQgd2l0aCBwYXJhbWV0ZXIgZGVjb3JhdG9ycyBgQEhvc3RgLCBgQFNlbGZgLCBgQFNraXBTZWxmYCwgYW5kIGBAT3B0aW9uYWxgLlxuICogQHJldHVybnMgdGhlIGluamVjdGVkIHZhbHVlIGlmIG9wZXJhdGlvbiBpcyBzdWNjZXNzZnVsLCBgbnVsbGAgb3RoZXJ3aXNlLlxuICogQHRocm93cyBpZiBjYWxsZWQgb3V0c2lkZSBvZiBhIHN1cHBvcnRlZCBjb250ZXh0LlxuICpcbiAqIEBwdWJsaWNBcGlcbiAqIEBkZXByZWNhdGVkIHByZWZlciBhbiBvcHRpb25zIG9iamVjdCBpbnN0ZWFkIG9mIGBJbmplY3RGbGFnc2BcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGluamVjdDxUPih0b2tlbjogUHJvdmlkZXJUb2tlbjxUPiwgZmxhZ3M/OiBJbmplY3RGbGFncyk6IFQgfCBudWxsO1xuLyoqXG4gKiBAcGFyYW0gdG9rZW4gQSB0b2tlbiB0aGF0IHJlcHJlc2VudHMgYSBkZXBlbmRlbmN5IHRoYXQgc2hvdWxkIGJlIGluamVjdGVkLlxuICogQHBhcmFtIG9wdGlvbnMgQ29udHJvbCBob3cgaW5qZWN0aW9uIGlzIGV4ZWN1dGVkLiBPcHRpb25zIGNvcnJlc3BvbmQgdG8gaW5qZWN0aW9uIHN0cmF0ZWdpZXNcbiAqICAgICB0aGF0IGNhbiBiZSBzcGVjaWZpZWQgd2l0aCBwYXJhbWV0ZXIgZGVjb3JhdG9ycyBgQEhvc3RgLCBgQFNlbGZgLCBgQFNraXBTZWxmYCwgYW5kXG4gKiAgICAgYEBPcHRpb25hbGAuXG4gKiBAcmV0dXJucyB0aGUgaW5qZWN0ZWQgdmFsdWUgaWYgb3BlcmF0aW9uIGlzIHN1Y2Nlc3NmdWwuXG4gKiBAdGhyb3dzIGlmIGNhbGxlZCBvdXRzaWRlIG9mIGEgc3VwcG9ydGVkIGNvbnRleHQsIG9yIGlmIHRoZSB0b2tlbiBpcyBub3QgZm91bmQuXG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgZnVuY3Rpb24gaW5qZWN0PFQ+KHRva2VuOiBQcm92aWRlclRva2VuPFQ+LCBvcHRpb25zOiBJbmplY3RPcHRpb25zICYge29wdGlvbmFsPzogZmFsc2V9KTogVDtcbi8qKlxuICogQHBhcmFtIHRva2VuIEEgdG9rZW4gdGhhdCByZXByZXNlbnRzIGEgZGVwZW5kZW5jeSB0aGF0IHNob3VsZCBiZSBpbmplY3RlZC5cbiAqIEBwYXJhbSBvcHRpb25zIENvbnRyb2wgaG93IGluamVjdGlvbiBpcyBleGVjdXRlZC4gT3B0aW9ucyBjb3JyZXNwb25kIHRvIGluamVjdGlvbiBzdHJhdGVnaWVzXG4gKiAgICAgdGhhdCBjYW4gYmUgc3BlY2lmaWVkIHdpdGggcGFyYW1ldGVyIGRlY29yYXRvcnMgYEBIb3N0YCwgYEBTZWxmYCwgYEBTa2lwU2VsZmAsIGFuZFxuICogICAgIGBAT3B0aW9uYWxgLlxuICogQHJldHVybnMgdGhlIGluamVjdGVkIHZhbHVlIGlmIG9wZXJhdGlvbiBpcyBzdWNjZXNzZnVsLCAgYG51bGxgIGlmIHRoZSB0b2tlbiBpcyBub3RcbiAqICAgICBmb3VuZCBhbmQgb3B0aW9uYWwgaW5qZWN0aW9uIGhhcyBiZWVuIHJlcXVlc3RlZC5cbiAqIEB0aHJvd3MgaWYgY2FsbGVkIG91dHNpZGUgb2YgYSBzdXBwb3J0ZWQgY29udGV4dCwgb3IgaWYgdGhlIHRva2VuIGlzIG5vdCBmb3VuZCBhbmQgb3B0aW9uYWxcbiAqICAgICBpbmplY3Rpb24gd2FzIG5vdCByZXF1ZXN0ZWQuXG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgZnVuY3Rpb24gaW5qZWN0PFQ+KHRva2VuOiBQcm92aWRlclRva2VuPFQ+LCBvcHRpb25zOiBJbmplY3RPcHRpb25zKTogVCB8IG51bGw7XG4vKipcbiAqIEBwYXJhbSB0b2tlbiBBIHRva2VuIHRoYXQgcmVwcmVzZW50cyBhIHN0YXRpYyBhdHRyaWJ1dGUgb24gdGhlIGhvc3Qgbm9kZSB0aGF0IHNob3VsZCBiZSBpbmplY3RlZC5cbiAqIEByZXR1cm5zIFZhbHVlIG9mIHRoZSBhdHRyaWJ1dGUgaWYgaXQgZXhpc3RzLlxuICogQHRocm93cyBJZiBjYWxsZWQgb3V0c2lkZSBvZiBhIHN1cHBvcnRlZCBjb250ZXh0IG9yIHRoZSBhdHRyaWJ1dGUgZG9lcyBub3QgZXhpc3QuXG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgZnVuY3Rpb24gaW5qZWN0KHRva2VuOiBIb3N0QXR0cmlidXRlVG9rZW4pOiBzdHJpbmc7XG4vKipcbiAqIEBwYXJhbSB0b2tlbiBBIHRva2VuIHRoYXQgcmVwcmVzZW50cyBhIHN0YXRpYyBhdHRyaWJ1dGUgb24gdGhlIGhvc3Qgbm9kZSB0aGF0IHNob3VsZCBiZSBpbmplY3RlZC5cbiAqIEByZXR1cm5zIFZhbHVlIG9mIHRoZSBhdHRyaWJ1dGUgaWYgaXQgZXhpc3RzLCBvdGhlcndpc2UgYG51bGxgLlxuICogQHRocm93cyBJZiBjYWxsZWQgb3V0c2lkZSBvZiBhIHN1cHBvcnRlZCBjb250ZXh0LlxuICpcbiAqIEBwdWJsaWNBcGlcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGluamVjdCh0b2tlbjogSG9zdEF0dHJpYnV0ZVRva2VuLCBvcHRpb25zOiB7b3B0aW9uYWw6IHRydWV9KTogc3RyaW5nIHwgbnVsbDtcbi8qKlxuICogQHBhcmFtIHRva2VuIEEgdG9rZW4gdGhhdCByZXByZXNlbnRzIGEgc3RhdGljIGF0dHJpYnV0ZSBvbiB0aGUgaG9zdCBub2RlIHRoYXQgc2hvdWxkIGJlIGluamVjdGVkLlxuICogQHJldHVybnMgVmFsdWUgb2YgdGhlIGF0dHJpYnV0ZSBpZiBpdCBleGlzdHMuXG4gKiBAdGhyb3dzIElmIGNhbGxlZCBvdXRzaWRlIG9mIGEgc3VwcG9ydGVkIGNvbnRleHQgb3IgdGhlIGF0dHJpYnV0ZSBkb2VzIG5vdCBleGlzdC5cbiAqXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBpbmplY3QodG9rZW46IEhvc3RBdHRyaWJ1dGVUb2tlbiwgb3B0aW9uczoge29wdGlvbmFsOiBmYWxzZX0pOiBzdHJpbmc7XG4vKipcbiAqIEluamVjdHMgYSB0b2tlbiBmcm9tIHRoZSBjdXJyZW50bHkgYWN0aXZlIGluamVjdG9yLlxuICogYGluamVjdGAgaXMgb25seSBzdXBwb3J0ZWQgaW4gYW4gW2luamVjdGlvbiBjb250ZXh0XShndWlkZS9kaS9kZXBlbmRlbmN5LWluamVjdGlvbi1jb250ZXh0KS4gSXRcbiAqIGNhbiBiZSB1c2VkIGR1cmluZzpcbiAqIC0gQ29uc3RydWN0aW9uICh2aWEgdGhlIGBjb25zdHJ1Y3RvcmApIG9mIGEgY2xhc3MgYmVpbmcgaW5zdGFudGlhdGVkIGJ5IHRoZSBESSBzeXN0ZW0sIHN1Y2hcbiAqIGFzIGFuIGBASW5qZWN0YWJsZWAgb3IgYEBDb21wb25lbnRgLlxuICogLSBJbiB0aGUgaW5pdGlhbGl6ZXIgZm9yIGZpZWxkcyBvZiBzdWNoIGNsYXNzZXMuXG4gKiAtIEluIHRoZSBmYWN0b3J5IGZ1bmN0aW9uIHNwZWNpZmllZCBmb3IgYHVzZUZhY3RvcnlgIG9mIGEgYFByb3ZpZGVyYCBvciBhbiBgQEluamVjdGFibGVgLlxuICogLSBJbiB0aGUgYGZhY3RvcnlgIGZ1bmN0aW9uIHNwZWNpZmllZCBmb3IgYW4gYEluamVjdGlvblRva2VuYC5cbiAqIC0gSW4gYSBzdGFja2ZyYW1lIG9mIGEgZnVuY3Rpb24gY2FsbCBpbiBhIERJIGNvbnRleHRcbiAqXG4gKiBAcGFyYW0gdG9rZW4gQSB0b2tlbiB0aGF0IHJlcHJlc2VudHMgYSBkZXBlbmRlbmN5IHRoYXQgc2hvdWxkIGJlIGluamVjdGVkLlxuICogQHBhcmFtIGZsYWdzIE9wdGlvbmFsIGZsYWdzIHRoYXQgY29udHJvbCBob3cgaW5qZWN0aW9uIGlzIGV4ZWN1dGVkLlxuICogVGhlIGZsYWdzIGNvcnJlc3BvbmQgdG8gaW5qZWN0aW9uIHN0cmF0ZWdpZXMgdGhhdCBjYW4gYmUgc3BlY2lmaWVkIHdpdGhcbiAqIHBhcmFtZXRlciBkZWNvcmF0b3JzIGBASG9zdGAsIGBAU2VsZmAsIGBAU2tpcFNlbGZgLCBhbmQgYEBPcHRpb25hbGAuXG4gKiBAcmV0dXJucyB0aGUgaW5qZWN0ZWQgdmFsdWUgaWYgb3BlcmF0aW9uIGlzIHN1Y2Nlc3NmdWwsIGBudWxsYCBvdGhlcndpc2UuXG4gKiBAdGhyb3dzIGlmIGNhbGxlZCBvdXRzaWRlIG9mIGEgc3VwcG9ydGVkIGNvbnRleHQuXG4gKlxuICogQHVzYWdlTm90ZXNcbiAqIEluIHByYWN0aWNlIHRoZSBgaW5qZWN0KClgIGNhbGxzIGFyZSBhbGxvd2VkIGluIGEgY29uc3RydWN0b3IsIGEgY29uc3RydWN0b3IgcGFyYW1ldGVyIGFuZCBhXG4gKiBmaWVsZCBpbml0aWFsaXplcjpcbiAqXG4gKiBgYGB0eXBlc2NyaXB0XG4gKiBASW5qZWN0YWJsZSh7cHJvdmlkZWRJbjogJ3Jvb3QnfSlcbiAqIGV4cG9ydCBjbGFzcyBDYXIge1xuICogICByYWRpbzogUmFkaW98dW5kZWZpbmVkO1xuICogICAvLyBPSzogZmllbGQgaW5pdGlhbGl6ZXJcbiAqICAgc3BhcmVUeXJlID0gaW5qZWN0KFR5cmUpO1xuICpcbiAqICAgY29uc3RydWN0b3IoKSB7XG4gKiAgICAgLy8gT0s6IGNvbnN0cnVjdG9yIGJvZHlcbiAqICAgICB0aGlzLnJhZGlvID0gaW5qZWN0KFJhZGlvKTtcbiAqICAgfVxuICogfVxuICogYGBgXG4gKlxuICogSXQgaXMgYWxzbyBsZWdhbCB0byBjYWxsIGBpbmplY3RgIGZyb20gYSBwcm92aWRlcidzIGZhY3Rvcnk6XG4gKlxuICogYGBgdHlwZXNjcmlwdFxuICogcHJvdmlkZXJzOiBbXG4gKiAgIHtwcm92aWRlOiBDYXIsIHVzZUZhY3Rvcnk6ICgpID0+IHtcbiAqICAgICAvLyBPSzogYSBjbGFzcyBmYWN0b3J5XG4gKiAgICAgY29uc3QgZW5naW5lID0gaW5qZWN0KEVuZ2luZSk7XG4gKiAgICAgcmV0dXJuIG5ldyBDYXIoZW5naW5lKTtcbiAqICAgfX1cbiAqIF1cbiAqIGBgYFxuICpcbiAqIENhbGxzIHRvIHRoZSBgaW5qZWN0KClgIGZ1bmN0aW9uIG91dHNpZGUgb2YgdGhlIGNsYXNzIGNyZWF0aW9uIGNvbnRleHQgd2lsbCByZXN1bHQgaW4gZXJyb3IuIE1vc3RcbiAqIG5vdGFibHksIGNhbGxzIHRvIGBpbmplY3QoKWAgYXJlIGRpc2FsbG93ZWQgYWZ0ZXIgYSBjbGFzcyBpbnN0YW5jZSB3YXMgY3JlYXRlZCwgaW4gbWV0aG9kc1xuICogKGluY2x1ZGluZyBsaWZlY3ljbGUgaG9va3MpOlxuICpcbiAqIGBgYHR5cGVzY3JpcHRcbiAqIEBDb21wb25lbnQoeyAuLi4gfSlcbiAqIGV4cG9ydCBjbGFzcyBDYXJDb21wb25lbnQge1xuICogICBuZ09uSW5pdCgpIHtcbiAqICAgICAvLyBFUlJPUjogdG9vIGxhdGUsIHRoZSBjb21wb25lbnQgaW5zdGFuY2Ugd2FzIGFscmVhZHkgY3JlYXRlZFxuICogICAgIGNvbnN0IGVuZ2luZSA9IGluamVjdChFbmdpbmUpO1xuICogICAgIGVuZ2luZS5zdGFydCgpO1xuICogICB9XG4gKiB9XG4gKiBgYGBcbiAqXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBpbmplY3Q8VD4oXG4gIHRva2VuOiBQcm92aWRlclRva2VuPFQ+IHwgSG9zdEF0dHJpYnV0ZVRva2VuLFxuICBmbGFnczogSW5qZWN0RmxhZ3MgfCBJbmplY3RPcHRpb25zID0gSW5qZWN0RmxhZ3MuRGVmYXVsdCxcbikge1xuICAvLyBUaGUgYGFzIGFueWAgaGVyZSBfc2hvdWxkbid0XyBiZSBuZWNlc3NhcnksIGJ1dCB3aXRob3V0IGl0IEpTQ29tcGlsZXJcbiAgLy8gdGhyb3dzIGEgZGlzYW1iaWd1YXRpb24gIGVycm9yIGR1ZSB0byB0aGUgbXVsdGlwbGUgc2lnbmF0dXJlcy5cbiAgcmV0dXJuIMm1ybVpbmplY3QodG9rZW4gYXMgYW55LCBjb252ZXJ0VG9CaXRGbGFncyhmbGFncykpO1xufVxuXG4vLyBDb252ZXJ0cyBvYmplY3QtYmFzZWQgREkgZmxhZ3MgKGBJbmplY3RPcHRpb25zYCkgdG8gYml0IGZsYWdzIChgSW5qZWN0RmxhZ3NgKS5cbmV4cG9ydCBmdW5jdGlvbiBjb252ZXJ0VG9CaXRGbGFncyhcbiAgZmxhZ3M6IEluamVjdE9wdGlvbnMgfCBJbmplY3RGbGFncyB8IHVuZGVmaW5lZCxcbik6IEluamVjdEZsYWdzIHwgdW5kZWZpbmVkIHtcbiAgaWYgKHR5cGVvZiBmbGFncyA9PT0gJ3VuZGVmaW5lZCcgfHwgdHlwZW9mIGZsYWdzID09PSAnbnVtYmVyJykge1xuICAgIHJldHVybiBmbGFncztcbiAgfVxuXG4gIC8vIFdoaWxlIFR5cGVTY3JpcHQgZG9lc24ndCBhY2NlcHQgaXQgd2l0aG91dCBhIGNhc3QsIGJpdHdpc2UgT1Igd2l0aCBmYWxzZS15IHZhbHVlcyBpblxuICAvLyBKYXZhU2NyaXB0IGlzIGEgbm8tb3AuIFdlIGNhbiB1c2UgdGhhdCBmb3IgYSB2ZXJ5IGNvZGVzaXplLWVmZmljaWVudCBjb252ZXJzaW9uIGZyb21cbiAgLy8gYEluamVjdE9wdGlvbnNgIHRvIGBJbmplY3RGbGFnc2AuXG4gIHJldHVybiAoSW50ZXJuYWxJbmplY3RGbGFncy5EZWZhdWx0IHwgLy8gY29tbWVudCB0byBmb3JjZSBhIGxpbmUgYnJlYWsgaW4gdGhlIGZvcm1hdHRlclxuICAgICgoZmxhZ3Mub3B0aW9uYWwgJiYgSW50ZXJuYWxJbmplY3RGbGFncy5PcHRpb25hbCkgYXMgbnVtYmVyKSB8XG4gICAgKChmbGFncy5ob3N0ICYmIEludGVybmFsSW5qZWN0RmxhZ3MuSG9zdCkgYXMgbnVtYmVyKSB8XG4gICAgKChmbGFncy5zZWxmICYmIEludGVybmFsSW5qZWN0RmxhZ3MuU2VsZikgYXMgbnVtYmVyKSB8XG4gICAgKChmbGFncy5za2lwU2VsZiAmJiBJbnRlcm5hbEluamVjdEZsYWdzLlNraXBTZWxmKSBhcyBudW1iZXIpKSBhcyBJbmplY3RGbGFncztcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGluamVjdEFyZ3ModHlwZXM6IChQcm92aWRlclRva2VuPGFueT4gfCBhbnlbXSlbXSk6IGFueVtdIHtcbiAgY29uc3QgYXJnczogYW55W10gPSBbXTtcbiAgZm9yIChsZXQgaSA9IDA7IGkgPCB0eXBlcy5sZW5ndGg7IGkrKykge1xuICAgIGNvbnN0IGFyZyA9IHJlc29sdmVGb3J3YXJkUmVmKHR5cGVzW2ldKTtcbiAgICBpZiAoQXJyYXkuaXNBcnJheShhcmcpKSB7XG4gICAgICBpZiAoYXJnLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICB0aHJvdyBuZXcgUnVudGltZUVycm9yKFxuICAgICAgICAgIFJ1bnRpbWVFcnJvckNvZGUuSU5WQUxJRF9ESUZGRVJfSU5QVVQsXG4gICAgICAgICAgbmdEZXZNb2RlICYmICdBcmd1bWVudHMgYXJyYXkgbXVzdCBoYXZlIGFyZ3VtZW50cy4nLFxuICAgICAgICApO1xuICAgICAgfVxuICAgICAgbGV0IHR5cGU6IFR5cGU8YW55PiB8IHVuZGVmaW5lZCA9IHVuZGVmaW5lZDtcbiAgICAgIGxldCBmbGFnczogSW5qZWN0RmxhZ3MgPSBJbmplY3RGbGFncy5EZWZhdWx0O1xuXG4gICAgICBmb3IgKGxldCBqID0gMDsgaiA8IGFyZy5sZW5ndGg7IGorKykge1xuICAgICAgICBjb25zdCBtZXRhID0gYXJnW2pdO1xuICAgICAgICBjb25zdCBmbGFnID0gZ2V0SW5qZWN0RmxhZyhtZXRhKTtcbiAgICAgICAgaWYgKHR5cGVvZiBmbGFnID09PSAnbnVtYmVyJykge1xuICAgICAgICAgIC8vIFNwZWNpYWwgY2FzZSB3aGVuIHdlIGhhbmRsZSBASW5qZWN0IGRlY29yYXRvci5cbiAgICAgICAgICBpZiAoZmxhZyA9PT0gRGVjb3JhdG9yRmxhZ3MuSW5qZWN0KSB7XG4gICAgICAgICAgICB0eXBlID0gbWV0YS50b2tlbjtcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgZmxhZ3MgfD0gZmxhZztcbiAgICAgICAgICB9XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgdHlwZSA9IG1ldGE7XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgYXJncy5wdXNoKMm1ybVpbmplY3QodHlwZSEsIGZsYWdzKSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGFyZ3MucHVzaCjJtcm1aW5qZWN0KGFyZykpO1xuICAgIH1cbiAgfVxuICByZXR1cm4gYXJncztcbn1cblxuLyoqXG4gKiBBdHRhY2hlcyBhIGdpdmVuIEluamVjdEZsYWcgdG8gYSBnaXZlbiBkZWNvcmF0b3IgdXNpbmcgbW9ua2V5LXBhdGNoaW5nLlxuICogU2luY2UgREkgZGVjb3JhdG9ycyBjYW4gYmUgdXNlZCBpbiBwcm92aWRlcnMgYGRlcHNgIGFycmF5ICh3aGVuIHByb3ZpZGVyIGlzIGNvbmZpZ3VyZWQgdXNpbmdcbiAqIGB1c2VGYWN0b3J5YCkgd2l0aG91dCBpbml0aWFsaXphdGlvbiAoZS5nLiBgSG9zdGApIGFuZCBhcyBhbiBpbnN0YW5jZSAoZS5nLiBgbmV3IEhvc3QoKWApLCB3ZVxuICogYXR0YWNoIHRoZSBmbGFnIHRvIG1ha2UgaXQgYXZhaWxhYmxlIGJvdGggYXMgYSBzdGF0aWMgcHJvcGVydHkgYW5kIGFzIGEgZmllbGQgb24gZGVjb3JhdG9yXG4gKiBpbnN0YW5jZS5cbiAqXG4gKiBAcGFyYW0gZGVjb3JhdG9yIFByb3ZpZGVkIERJIGRlY29yYXRvci5cbiAqIEBwYXJhbSBmbGFnIEluamVjdEZsYWcgdGhhdCBzaG91bGQgYmUgYXBwbGllZC5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGF0dGFjaEluamVjdEZsYWcoZGVjb3JhdG9yOiBhbnksIGZsYWc6IEludGVybmFsSW5qZWN0RmxhZ3MgfCBEZWNvcmF0b3JGbGFncyk6IGFueSB7XG4gIGRlY29yYXRvcltESV9ERUNPUkFUT1JfRkxBR10gPSBmbGFnO1xuICBkZWNvcmF0b3IucHJvdG90eXBlW0RJX0RFQ09SQVRPUl9GTEFHXSA9IGZsYWc7XG4gIHJldHVybiBkZWNvcmF0b3I7XG59XG5cbi8qKlxuICogUmVhZHMgbW9ua2V5LXBhdGNoZWQgcHJvcGVydHkgdGhhdCBjb250YWlucyBJbmplY3RGbGFnIGF0dGFjaGVkIHRvIGEgZGVjb3JhdG9yLlxuICpcbiAqIEBwYXJhbSB0b2tlbiBUb2tlbiB0aGF0IG1heSBjb250YWluIG1vbmtleS1wYXRjaGVkIERJIGZsYWdzIHByb3BlcnR5LlxuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0SW5qZWN0RmxhZyh0b2tlbjogYW55KTogbnVtYmVyIHwgdW5kZWZpbmVkIHtcbiAgcmV0dXJuIHRva2VuW0RJX0RFQ09SQVRPUl9GTEFHXTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGNhdGNoSW5qZWN0b3JFcnJvcihcbiAgZTogYW55LFxuICB0b2tlbjogYW55LFxuICBpbmplY3RvckVycm9yTmFtZTogc3RyaW5nLFxuICBzb3VyY2U6IHN0cmluZyB8IG51bGwsXG4pOiBuZXZlciB7XG4gIGNvbnN0IHRva2VuUGF0aDogYW55W10gPSBlW05HX1RFTVBfVE9LRU5fUEFUSF07XG4gIGlmICh0b2tlbltTT1VSQ0VdKSB7XG4gICAgdG9rZW5QYXRoLnVuc2hpZnQodG9rZW5bU09VUkNFXSk7XG4gIH1cbiAgZS5tZXNzYWdlID0gZm9ybWF0RXJyb3IoJ1xcbicgKyBlLm1lc3NhZ2UsIHRva2VuUGF0aCwgaW5qZWN0b3JFcnJvck5hbWUsIHNvdXJjZSk7XG4gIGVbTkdfVE9LRU5fUEFUSF0gPSB0b2tlblBhdGg7XG4gIGVbTkdfVEVNUF9UT0tFTl9QQVRIXSA9IG51bGw7XG4gIHRocm93IGU7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBmb3JtYXRFcnJvcihcbiAgdGV4dDogc3RyaW5nLFxuICBvYmo6IGFueSxcbiAgaW5qZWN0b3JFcnJvck5hbWU6IHN0cmluZyxcbiAgc291cmNlOiBzdHJpbmcgfCBudWxsID0gbnVsbCxcbik6IHN0cmluZyB7XG4gIHRleHQgPSB0ZXh0ICYmIHRleHQuY2hhckF0KDApID09PSAnXFxuJyAmJiB0ZXh0LmNoYXJBdCgxKSA9PSBOT19ORVdfTElORSA/IHRleHQuc2xpY2UoMikgOiB0ZXh0O1xuICBsZXQgY29udGV4dCA9IHN0cmluZ2lmeShvYmopO1xuICBpZiAoQXJyYXkuaXNBcnJheShvYmopKSB7XG4gICAgY29udGV4dCA9IG9iai5tYXAoc3RyaW5naWZ5KS5qb2luKCcgLT4gJyk7XG4gIH0gZWxzZSBpZiAodHlwZW9mIG9iaiA9PT0gJ29iamVjdCcpIHtcbiAgICBsZXQgcGFydHMgPSA8c3RyaW5nW10+W107XG4gICAgZm9yIChsZXQga2V5IGluIG9iaikge1xuICAgICAgaWYgKG9iai5oYXNPd25Qcm9wZXJ0eShrZXkpKSB7XG4gICAgICAgIGxldCB2YWx1ZSA9IG9ialtrZXldO1xuICAgICAgICBwYXJ0cy5wdXNoKFxuICAgICAgICAgIGtleSArICc6JyArICh0eXBlb2YgdmFsdWUgPT09ICdzdHJpbmcnID8gSlNPTi5zdHJpbmdpZnkodmFsdWUpIDogc3RyaW5naWZ5KHZhbHVlKSksXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfVxuICAgIGNvbnRleHQgPSBgeyR7cGFydHMuam9pbignLCAnKX19YDtcbiAgfVxuICByZXR1cm4gYCR7aW5qZWN0b3JFcnJvck5hbWV9JHtzb3VyY2UgPyAnKCcgKyBzb3VyY2UgKyAnKScgOiAnJ31bJHtjb250ZXh0fV06ICR7dGV4dC5yZXBsYWNlKFxuICAgIE5FV19MSU5FLFxuICAgICdcXG4gICcsXG4gICl9YDtcbn1cbiJdfQ==