/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { getClosureSafeProperty } from '../../util/property';
/**
 * Construct an injectable definition which defines how a token will be constructed by the DI
 * system, and in which injectors (if any) it will be available.
 *
 * This should be assigned to a static `ɵprov` field on a type, which will then be an
 * `InjectableType`.
 *
 * Options:
 * * `providedIn` determines which injectors will include the injectable, by either associating it
 *   with an `@NgModule` or other `InjectorType`, or by specifying that this injectable should be
 *   provided in the `'root'` injector, which will be the application-level injector in most apps.
 * * `factory` gives the zero argument function which will create an instance of the injectable.
 *   The factory can call [`inject`](api/core/inject) to access the `Injector` and request injection
 * of dependencies.
 *
 * @codeGenApi
 * @publicApi This instruction has been emitted by ViewEngine for some time and is deployed to npm.
 */
export function ɵɵdefineInjectable(opts) {
    return {
        token: opts.token,
        providedIn: opts.providedIn || null,
        factory: opts.factory,
        value: undefined,
    };
}
/**
 * @deprecated in v8, delete after v10. This API should be used only by generated code, and that
 * code should now use ɵɵdefineInjectable instead.
 * @publicApi
 */
export const defineInjectable = ɵɵdefineInjectable;
/**
 * Construct an `InjectorDef` which configures an injector.
 *
 * This should be assigned to a static injector def (`ɵinj`) field on a type, which will then be an
 * `InjectorType`.
 *
 * Options:
 *
 * * `providers`: an optional array of providers to add to the injector. Each provider must
 *   either have a factory or point to a type which has a `ɵprov` static property (the
 *   type must be an `InjectableType`).
 * * `imports`: an optional array of imports of other `InjectorType`s or `InjectorTypeWithModule`s
 *   whose providers will also be added to the injector. Locally provided types will override
 *   providers from imports.
 *
 * @codeGenApi
 */
export function ɵɵdefineInjector(options) {
    return { providers: options.providers || [], imports: options.imports || [] };
}
/**
 * Read the injectable def (`ɵprov`) for `type` in a way which is immune to accidentally reading
 * inherited value.
 *
 * @param type A type which may have its own (non-inherited) `ɵprov`.
 */
export function getInjectableDef(type) {
    return getOwnDefinition(type, NG_PROV_DEF) || getOwnDefinition(type, NG_INJECTABLE_DEF);
}
export function isInjectable(type) {
    return getInjectableDef(type) !== null;
}
/**
 * Return definition only if it is defined directly on `type` and is not inherited from a base
 * class of `type`.
 */
function getOwnDefinition(type, field) {
    return type.hasOwnProperty(field) ? type[field] : null;
}
/**
 * Read the injectable def (`ɵprov`) for `type` or read the `ɵprov` from one of its ancestors.
 *
 * @param type A type which may have `ɵprov`, via inheritance.
 *
 * @deprecated Will be removed in a future version of Angular, where an error will occur in the
 *     scenario if we find the `ɵprov` on an ancestor only.
 */
export function getInheritedInjectableDef(type) {
    const def = type && (type[NG_PROV_DEF] || type[NG_INJECTABLE_DEF]);
    if (def) {
        ngDevMode &&
            console.warn(`DEPRECATED: DI is instantiating a token "${type.name}" that inherits its @Injectable decorator but does not provide one itself.\n` +
                `This will become an error in a future version of Angular. Please add @Injectable() to the "${type.name}" class.`);
        return def;
    }
    else {
        return null;
    }
}
/**
 * Read the injector def type in a way which is immune to accidentally reading inherited value.
 *
 * @param type type which may have an injector def (`ɵinj`)
 */
export function getInjectorDef(type) {
    return type && (type.hasOwnProperty(NG_INJ_DEF) || type.hasOwnProperty(NG_INJECTOR_DEF))
        ? type[NG_INJ_DEF]
        : null;
}
export const NG_PROV_DEF = getClosureSafeProperty({ ɵprov: getClosureSafeProperty });
export const NG_INJ_DEF = getClosureSafeProperty({ ɵinj: getClosureSafeProperty });
// We need to keep these around so we can read off old defs if new defs are unavailable
export const NG_INJECTABLE_DEF = getClosureSafeProperty({ ngInjectableDef: getClosureSafeProperty });
export const NG_INJECTOR_DEF = getClosureSafeProperty({ ngInjectorDef: getClosureSafeProperty });
//# sourceMappingURL=data:application/json;base64,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