/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ERROR_DETAILS_PAGE_BASE_URL } from './error_details_base_url';
/**
 * Class that represents a runtime error.
 * Formats and outputs the error message in a consistent way.
 *
 * Example:
 * ```
 *  throw new RuntimeError(
 *    RuntimeErrorCode.INJECTOR_ALREADY_DESTROYED,
 *    ngDevMode && 'Injector has already been destroyed.');
 * ```
 *
 * Note: the `message` argument contains a descriptive error message as a string in development
 * mode (when the `ngDevMode` is defined). In production mode (after tree-shaking pass), the
 * `message` argument becomes `false`, thus we account for it in the typings and the runtime
 * logic.
 */
export class RuntimeError extends Error {
    constructor(code, message) {
        super(formatRuntimeError(code, message));
        this.code = code;
    }
}
/**
 * Called to format a runtime error.
 * See additional info on the `message` argument type in the `RuntimeError` class description.
 */
export function formatRuntimeError(code, message) {
    // Error code might be a negative number, which is a special marker that instructs the logic to
    // generate a link to the error details page on angular.io.
    // We also prepend `0` to non-compile-time errors.
    const fullCode = `NG0${Math.abs(code)}`;
    let errorMessage = `${fullCode}${message ? ': ' + message : ''}`;
    if (ngDevMode && code < 0) {
        const addPeriodSeparator = !errorMessage.match(/[.,;!?\n]$/);
        const separator = addPeriodSeparator ? '.' : '';
        errorMessage = `${errorMessage}${separator} Find more at ${ERROR_DETAILS_PAGE_BASE_URL}/${fullCode}`;
    }
    return errorMessage;
}
//# sourceMappingURL=data:application/json;base64,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