/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { setActiveConsumer } from '@angular/core/primitives/signals';
import { Subject, Subscription } from 'rxjs';
import { isInInjectionContext } from './di/contextual';
import { inject } from './di/injector_compatibility';
import { DestroyRef } from './linker/destroy_ref';
import { PendingTasks } from './pending_tasks';
class EventEmitter_ extends Subject {
    constructor(isAsync = false) {
        super();
        this.destroyRef = undefined;
        this.pendingTasks = undefined;
        this.__isAsync = isAsync;
        // Attempt to retrieve a `DestroyRef` and `PendingTasks` optionally.
        // For backwards compatibility reasons, this cannot be required.
        if (isInInjectionContext()) {
            this.destroyRef = inject(DestroyRef, { optional: true }) ?? undefined;
            this.pendingTasks = inject(PendingTasks, { optional: true }) ?? undefined;
        }
    }
    emit(value) {
        const prevConsumer = setActiveConsumer(null);
        try {
            super.next(value);
        }
        finally {
            setActiveConsumer(prevConsumer);
        }
    }
    subscribe(observerOrNext, error, complete) {
        let nextFn = observerOrNext;
        let errorFn = error || (() => null);
        let completeFn = complete;
        if (observerOrNext && typeof observerOrNext === 'object') {
            const observer = observerOrNext;
            nextFn = observer.next?.bind(observer);
            errorFn = observer.error?.bind(observer);
            completeFn = observer.complete?.bind(observer);
        }
        if (this.__isAsync) {
            errorFn = this.wrapInTimeout(errorFn);
            if (nextFn) {
                nextFn = this.wrapInTimeout(nextFn);
            }
            if (completeFn) {
                completeFn = this.wrapInTimeout(completeFn);
            }
        }
        const sink = super.subscribe({ next: nextFn, error: errorFn, complete: completeFn });
        if (observerOrNext instanceof Subscription) {
            observerOrNext.add(sink);
        }
        return sink;
    }
    wrapInTimeout(fn) {
        return (value) => {
            const taskId = this.pendingTasks?.add();
            setTimeout(() => {
                fn(value);
                if (taskId !== undefined) {
                    this.pendingTasks?.remove(taskId);
                }
            });
        };
    }
}
/**
 * @publicApi
 */
export const EventEmitter = EventEmitter_;
//# sourceMappingURL=data:application/json;base64,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