/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { isEarlyEventType, isCaptureEventType, EventContractContainer, EventContract, EventDispatcher, registerDispatcher, getAppScopedQueuedEventInfos, clearAppScopedEarlyEventContract, } from '@angular/core/primitives/event-dispatch';
import { APP_BOOTSTRAP_LISTENER, ApplicationRef, whenStable } from '../application/application_ref';
import { ENVIRONMENT_INITIALIZER, Injector } from '../di';
import { inject } from '../di/injector_compatibility';
import { setStashFn } from '../render3/instructions/listener';
import { CLEANUP } from '../render3/interfaces/view';
import { isPlatformBrowser } from '../render3/util/misc_utils';
import { unwrapRNode } from '../render3/util/view_utils';
import { EVENT_REPLAY_ENABLED_DEFAULT, IS_EVENT_REPLAY_ENABLED, IS_GLOBAL_EVENT_DELEGATION_ENABLED, } from './tokens';
import { sharedStashFunction, removeListeners, invokeRegisteredListeners, JSACTION_EVENT_CONTRACT, } from '../event_delegation_utils';
import { APP_ID } from '../application/application_tokens';
import { performanceMarkFeature } from '../util/performance';
/**
 * A set of DOM elements with `jsaction` attributes.
 */
const jsactionSet = new Set();
function isGlobalEventDelegationEnabled(injector) {
    return injector.get(IS_GLOBAL_EVENT_DELEGATION_ENABLED, false);
}
/**
 * Determines whether Event Replay feature should be activated on the client.
 */
function shouldEnableEventReplay(injector) {
    return (injector.get(IS_EVENT_REPLAY_ENABLED, EVENT_REPLAY_ENABLED_DEFAULT) &&
        !isGlobalEventDelegationEnabled(injector));
}
/**
 * Returns a set of providers required to setup support for event replay.
 * Requires hydration to be enabled separately.
 */
export function withEventReplay() {
    return [
        {
            provide: IS_EVENT_REPLAY_ENABLED,
            useFactory: () => {
                let isEnabled = true;
                if (isPlatformBrowser()) {
                    // Note: globalThis[CONTRACT_PROPERTY] may be undefined in case Event Replay feature
                    // is enabled, but there are no events configured in this application, in which case
                    // we don't activate this feature, since there are no events to replay.
                    const appId = inject(APP_ID);
                    isEnabled = !!window._ejsas?.[appId];
                }
                if (isEnabled) {
                    performanceMarkFeature('NgEventReplay');
                }
                return isEnabled;
            },
        },
        {
            provide: ENVIRONMENT_INITIALIZER,
            useValue: () => {
                const injector = inject(Injector);
                if (isPlatformBrowser(injector) && shouldEnableEventReplay(injector)) {
                    setStashFn((rEl, eventName, listenerFn) => {
                        sharedStashFunction(rEl, eventName, listenerFn);
                        jsactionSet.add(rEl);
                    });
                }
            },
            multi: true,
        },
        {
            provide: APP_BOOTSTRAP_LISTENER,
            useFactory: () => {
                if (isPlatformBrowser()) {
                    const injector = inject(Injector);
                    const appRef = inject(ApplicationRef);
                    return () => {
                        if (!shouldEnableEventReplay(injector)) {
                            return;
                        }
                        // Kick off event replay logic once hydration for the initial part
                        // of the application is completed. This timing is similar to the unclaimed
                        // dehydrated views cleanup timing.
                        whenStable(appRef).then(() => {
                            const eventContractDetails = injector.get(JSACTION_EVENT_CONTRACT);
                            initEventReplay(eventContractDetails, injector);
                            jsactionSet.forEach(removeListeners);
                            // After hydration, we shouldn't need to do anymore work related to
                            // event replay anymore.
                            setStashFn(() => { });
                        });
                    };
                }
                return () => { }; // noop for the server code
            },
            multi: true,
        },
    ];
}
const initEventReplay = (eventDelegation, injector) => {
    const appId = injector.get(APP_ID);
    // This is set in packages/platform-server/src/utils.ts
    const earlyJsactionData = window._ejsas[appId];
    const eventContract = (eventDelegation.instance = new EventContract(new EventContractContainer(earlyJsactionData.c)));
    for (const et of earlyJsactionData.et) {
        eventContract.addEvent(et);
    }
    for (const et of earlyJsactionData.etc) {
        eventContract.addEvent(et);
    }
    const eventInfos = getAppScopedQueuedEventInfos(appId);
    eventContract.replayEarlyEventInfos(eventInfos);
    clearAppScopedEarlyEventContract(appId);
    const dispatcher = new EventDispatcher(invokeRegisteredListeners);
    registerDispatcher(eventContract, dispatcher);
};
/**
 * Extracts information about all DOM events (added in a template) registered on elements in a give
 * LView. Maps collected events to a corresponding DOM element (an element is used as a key).
 */
export function collectDomEventsInfo(tView, lView, eventTypesToReplay) {
    const domEventsInfo = new Map();
    const lCleanup = lView[CLEANUP];
    const tCleanup = tView.cleanup;
    if (!tCleanup || !lCleanup) {
        return domEventsInfo;
    }
    for (let i = 0; i < tCleanup.length;) {
        const firstParam = tCleanup[i++];
        const secondParam = tCleanup[i++];
        if (typeof firstParam !== 'string') {
            continue;
        }
        const eventType = firstParam;
        if (!isEarlyEventType(eventType)) {
            continue;
        }
        if (isCaptureEventType(eventType)) {
            eventTypesToReplay.capture.add(eventType);
        }
        else {
            eventTypesToReplay.regular.add(eventType);
        }
        const listenerElement = unwrapRNode(lView[secondParam]);
        i++; // move the cursor to the next position (location of the listener idx)
        const useCaptureOrIndx = tCleanup[i++];
        // if useCaptureOrIndx is boolean then report it as is.
        // if useCaptureOrIndx is positive number then it in unsubscribe method
        // if useCaptureOrIndx is negative number then it is a Subscription
        const isDomEvent = typeof useCaptureOrIndx === 'boolean' || useCaptureOrIndx >= 0;
        if (!isDomEvent) {
            continue;
        }
        if (!domEventsInfo.has(listenerElement)) {
            domEventsInfo.set(listenerElement, [eventType]);
        }
        else {
            domEventsInfo.get(listenerElement).push(eventType);
        }
    }
    return domEventsInfo;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZXZlbnRfcmVwbGF5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vLi4vLi4vcGFja2FnZXMvY29yZS9zcmMvaHlkcmF0aW9uL2V2ZW50X3JlcGxheS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTs7Ozs7O0dBTUc7QUFFSCxPQUFPLEVBQ0wsZ0JBQWdCLEVBQ2hCLGtCQUFrQixFQUNsQixzQkFBc0IsRUFDdEIsYUFBYSxFQUNiLGVBQWUsRUFDZixrQkFBa0IsRUFDbEIsNEJBQTRCLEVBQzVCLGdDQUFnQyxHQUNqQyxNQUFNLHlDQUF5QyxDQUFDO0FBRWpELE9BQU8sRUFBQyxzQkFBc0IsRUFBRSxjQUFjLEVBQUUsVUFBVSxFQUFDLE1BQU0sZ0NBQWdDLENBQUM7QUFDbEcsT0FBTyxFQUFDLHVCQUF1QixFQUFFLFFBQVEsRUFBQyxNQUFNLE9BQU8sQ0FBQztBQUN4RCxPQUFPLEVBQUMsTUFBTSxFQUFDLE1BQU0sOEJBQThCLENBQUM7QUFFcEQsT0FBTyxFQUFDLFVBQVUsRUFBQyxNQUFNLGtDQUFrQyxDQUFDO0FBRTVELE9BQU8sRUFBQyxPQUFPLEVBQWUsTUFBTSw0QkFBNEIsQ0FBQztBQUNqRSxPQUFPLEVBQUMsaUJBQWlCLEVBQUMsTUFBTSw0QkFBNEIsQ0FBQztBQUM3RCxPQUFPLEVBQUMsV0FBVyxFQUFDLE1BQU0sNEJBQTRCLENBQUM7QUFFdkQsT0FBTyxFQUNMLDRCQUE0QixFQUM1Qix1QkFBdUIsRUFDdkIsa0NBQWtDLEdBQ25DLE1BQU0sVUFBVSxDQUFDO0FBQ2xCLE9BQU8sRUFDTCxtQkFBbUIsRUFDbkIsZUFBZSxFQUNmLHlCQUF5QixFQUV6Qix1QkFBdUIsR0FDeEIsTUFBTSwyQkFBMkIsQ0FBQztBQUNuQyxPQUFPLEVBQUMsTUFBTSxFQUFDLE1BQU0sbUNBQW1DLENBQUM7QUFDekQsT0FBTyxFQUFDLHNCQUFzQixFQUFDLE1BQU0scUJBQXFCLENBQUM7QUFFM0Q7O0dBRUc7QUFDSCxNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsRUFBVyxDQUFDO0FBRXZDLFNBQVMsOEJBQThCLENBQUMsUUFBa0I7SUFDeEQsT0FBTyxRQUFRLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBQ2pFLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQVMsdUJBQXVCLENBQUMsUUFBa0I7SUFDakQsT0FBTyxDQUNMLFFBQVEsQ0FBQyxHQUFHLENBQUMsdUJBQXVCLEVBQUUsNEJBQTRCLENBQUM7UUFDbkUsQ0FBQyw4QkFBOEIsQ0FBQyxRQUFRLENBQUMsQ0FDMUMsQ0FBQztBQUNKLENBQUM7QUFFRDs7O0dBR0c7QUFDSCxNQUFNLFVBQVUsZUFBZTtJQUM3QixPQUFPO1FBQ0w7WUFDRSxPQUFPLEVBQUUsdUJBQXVCO1lBQ2hDLFVBQVUsRUFBRSxHQUFHLEVBQUU7Z0JBQ2YsSUFBSSxTQUFTLEdBQUcsSUFBSSxDQUFDO2dCQUNyQixJQUFJLGlCQUFpQixFQUFFLEVBQUUsQ0FBQztvQkFDeEIsb0ZBQW9GO29CQUNwRixvRkFBb0Y7b0JBQ3BGLHVFQUF1RTtvQkFDdkUsTUFBTSxLQUFLLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDO29CQUM3QixTQUFTLEdBQUcsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFDdkMsQ0FBQztnQkFDRCxJQUFJLFNBQVMsRUFBRSxDQUFDO29CQUNkLHNCQUFzQixDQUFDLGVBQWUsQ0FBQyxDQUFDO2dCQUMxQyxDQUFDO2dCQUNELE9BQU8sU0FBUyxDQUFDO1lBQ25CLENBQUM7U0FDRjtRQUNEO1lBQ0UsT0FBTyxFQUFFLHVCQUF1QjtZQUNoQyxRQUFRLEVBQUUsR0FBRyxFQUFFO2dCQUNiLE1BQU0sUUFBUSxHQUFHLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQztnQkFDbEMsSUFBSSxpQkFBaUIsQ0FBQyxRQUFRLENBQUMsSUFBSSx1QkFBdUIsQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDO29CQUNyRSxVQUFVLENBQUMsQ0FBQyxHQUFhLEVBQUUsU0FBaUIsRUFBRSxVQUF3QixFQUFFLEVBQUU7d0JBQ3hFLG1CQUFtQixDQUFDLEdBQUcsRUFBRSxTQUFTLEVBQUUsVUFBVSxDQUFDLENBQUM7d0JBQ2hELFdBQVcsQ0FBQyxHQUFHLENBQUMsR0FBeUIsQ0FBQyxDQUFDO29CQUM3QyxDQUFDLENBQUMsQ0FBQztnQkFDTCxDQUFDO1lBQ0gsQ0FBQztZQUNELEtBQUssRUFBRSxJQUFJO1NBQ1o7UUFDRDtZQUNFLE9BQU8sRUFBRSxzQkFBc0I7WUFDL0IsVUFBVSxFQUFFLEdBQUcsRUFBRTtnQkFDZixJQUFJLGlCQUFpQixFQUFFLEVBQUUsQ0FBQztvQkFDeEIsTUFBTSxRQUFRLEdBQUcsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDO29CQUNsQyxNQUFNLE1BQU0sR0FBRyxNQUFNLENBQUMsY0FBYyxDQUFDLENBQUM7b0JBQ3RDLE9BQU8sR0FBRyxFQUFFO3dCQUNWLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDOzRCQUN2QyxPQUFPO3dCQUNULENBQUM7d0JBRUQsa0VBQWtFO3dCQUNsRSwyRUFBMkU7d0JBQzNFLG1DQUFtQzt3QkFDbkMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUU7NEJBQzNCLE1BQU0sb0JBQW9CLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDOzRCQUNuRSxlQUFlLENBQUMsb0JBQW9CLEVBQUUsUUFBUSxDQUFDLENBQUM7NEJBQ2hELFdBQVcsQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLENBQUM7NEJBQ3JDLG1FQUFtRTs0QkFDbkUsd0JBQXdCOzRCQUN4QixVQUFVLENBQUMsR0FBRyxFQUFFLEdBQUUsQ0FBQyxDQUFDLENBQUM7d0JBQ3ZCLENBQUMsQ0FBQyxDQUFDO29CQUNMLENBQUMsQ0FBQztnQkFDSixDQUFDO2dCQUNELE9BQU8sR0FBRyxFQUFFLEdBQUUsQ0FBQyxDQUFDLENBQUMsMkJBQTJCO1lBQzlDLENBQUM7WUFDRCxLQUFLLEVBQUUsSUFBSTtTQUNaO0tBQ0YsQ0FBQztBQUNKLENBQUM7QUFFRCxNQUFNLGVBQWUsR0FBRyxDQUFDLGVBQXFDLEVBQUUsUUFBa0IsRUFBRSxFQUFFO0lBQ3BGLE1BQU0sS0FBSyxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDbkMsdURBQXVEO0lBQ3ZELE1BQU0saUJBQWlCLEdBQUcsTUFBTSxDQUFDLE1BQU8sQ0FBQyxLQUFLLENBQUUsQ0FBQztJQUNqRCxNQUFNLGFBQWEsR0FBRyxDQUFDLGVBQWUsQ0FBQyxRQUFRLEdBQUcsSUFBSSxhQUFhLENBQ2pFLElBQUksc0JBQXNCLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLENBQ2hELENBQUMsQ0FBQztJQUNILEtBQUssTUFBTSxFQUFFLElBQUksaUJBQWlCLENBQUMsRUFBRSxFQUFFLENBQUM7UUFDdEMsYUFBYSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUM3QixDQUFDO0lBQ0QsS0FBSyxNQUFNLEVBQUUsSUFBSSxpQkFBaUIsQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUN2QyxhQUFhLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQzdCLENBQUM7SUFDRCxNQUFNLFVBQVUsR0FBRyw0QkFBNEIsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUN2RCxhQUFhLENBQUMscUJBQXFCLENBQUMsVUFBVSxDQUFDLENBQUM7SUFDaEQsZ0NBQWdDLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDeEMsTUFBTSxVQUFVLEdBQUcsSUFBSSxlQUFlLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUNsRSxrQkFBa0IsQ0FBQyxhQUFhLEVBQUUsVUFBVSxDQUFDLENBQUM7QUFDaEQsQ0FBQyxDQUFDO0FBRUY7OztHQUdHO0FBQ0gsTUFBTSxVQUFVLG9CQUFvQixDQUNsQyxLQUFZLEVBQ1osS0FBWSxFQUNaLGtCQUFnRTtJQUVoRSxNQUFNLGFBQWEsR0FBRyxJQUFJLEdBQUcsRUFBcUIsQ0FBQztJQUNuRCxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDaEMsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztJQUMvQixJQUFJLENBQUMsUUFBUSxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDM0IsT0FBTyxhQUFhLENBQUM7SUFDdkIsQ0FBQztJQUNELEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxRQUFRLENBQUMsTUFBTSxHQUFJLENBQUM7UUFDdEMsTUFBTSxVQUFVLEdBQUcsUUFBUSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDakMsTUFBTSxXQUFXLEdBQUcsUUFBUSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDbEMsSUFBSSxPQUFPLFVBQVUsS0FBSyxRQUFRLEVBQUUsQ0FBQztZQUNuQyxTQUFTO1FBQ1gsQ0FBQztRQUNELE1BQU0sU0FBUyxHQUFHLFVBQVUsQ0FBQztRQUM3QixJQUFJLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQztZQUNqQyxTQUFTO1FBQ1gsQ0FBQztRQUNELElBQUksa0JBQWtCLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQztZQUNsQyxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQzVDLENBQUM7YUFBTSxDQUFDO1lBQ04sa0JBQWtCLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUM1QyxDQUFDO1FBQ0QsTUFBTSxlQUFlLEdBQUcsV0FBVyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBbUIsQ0FBQztRQUMxRSxDQUFDLEVBQUUsQ0FBQyxDQUFDLHNFQUFzRTtRQUMzRSxNQUFNLGdCQUFnQixHQUFHLFFBQVEsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ3ZDLHVEQUF1RDtRQUN2RCx1RUFBdUU7UUFDdkUsbUVBQW1FO1FBQ25FLE1BQU0sVUFBVSxHQUFHLE9BQU8sZ0JBQWdCLEtBQUssU0FBUyxJQUFJLGdCQUFnQixJQUFJLENBQUMsQ0FBQztRQUNsRixJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7WUFDaEIsU0FBUztRQUNYLENBQUM7UUFDRCxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxlQUFlLENBQUMsRUFBRSxDQUFDO1lBQ3hDLGFBQWEsQ0FBQyxHQUFHLENBQUMsZUFBZSxFQUFFLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztRQUNsRCxDQUFDO2FBQU0sQ0FBQztZQUNOLGFBQWEsQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFFLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ3RELENBQUM7SUFDSCxDQUFDO0lBQ0QsT0FBTyxhQUFhLENBQUM7QUFDdkIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgTExDIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuXG5pbXBvcnQge1xuICBpc0Vhcmx5RXZlbnRUeXBlLFxuICBpc0NhcHR1cmVFdmVudFR5cGUsXG4gIEV2ZW50Q29udHJhY3RDb250YWluZXIsXG4gIEV2ZW50Q29udHJhY3QsXG4gIEV2ZW50RGlzcGF0Y2hlcixcbiAgcmVnaXN0ZXJEaXNwYXRjaGVyLFxuICBnZXRBcHBTY29wZWRRdWV1ZWRFdmVudEluZm9zLFxuICBjbGVhckFwcFNjb3BlZEVhcmx5RXZlbnRDb250cmFjdCxcbn0gZnJvbSAnQGFuZ3VsYXIvY29yZS9wcmltaXRpdmVzL2V2ZW50LWRpc3BhdGNoJztcblxuaW1wb3J0IHtBUFBfQk9PVFNUUkFQX0xJU1RFTkVSLCBBcHBsaWNhdGlvblJlZiwgd2hlblN0YWJsZX0gZnJvbSAnLi4vYXBwbGljYXRpb24vYXBwbGljYXRpb25fcmVmJztcbmltcG9ydCB7RU5WSVJPTk1FTlRfSU5JVElBTElaRVIsIEluamVjdG9yfSBmcm9tICcuLi9kaSc7XG5pbXBvcnQge2luamVjdH0gZnJvbSAnLi4vZGkvaW5qZWN0b3JfY29tcGF0aWJpbGl0eSc7XG5pbXBvcnQge1Byb3ZpZGVyfSBmcm9tICcuLi9kaS9pbnRlcmZhY2UvcHJvdmlkZXInO1xuaW1wb3J0IHtzZXRTdGFzaEZufSBmcm9tICcuLi9yZW5kZXIzL2luc3RydWN0aW9ucy9saXN0ZW5lcic7XG5pbXBvcnQge1JFbGVtZW50fSBmcm9tICcuLi9yZW5kZXIzL2ludGVyZmFjZXMvcmVuZGVyZXJfZG9tJztcbmltcG9ydCB7Q0xFQU5VUCwgTFZpZXcsIFRWaWV3fSBmcm9tICcuLi9yZW5kZXIzL2ludGVyZmFjZXMvdmlldyc7XG5pbXBvcnQge2lzUGxhdGZvcm1Ccm93c2VyfSBmcm9tICcuLi9yZW5kZXIzL3V0aWwvbWlzY191dGlscyc7XG5pbXBvcnQge3Vud3JhcFJOb2RlfSBmcm9tICcuLi9yZW5kZXIzL3V0aWwvdmlld191dGlscyc7XG5cbmltcG9ydCB7XG4gIEVWRU5UX1JFUExBWV9FTkFCTEVEX0RFRkFVTFQsXG4gIElTX0VWRU5UX1JFUExBWV9FTkFCTEVELFxuICBJU19HTE9CQUxfRVZFTlRfREVMRUdBVElPTl9FTkFCTEVELFxufSBmcm9tICcuL3Rva2Vucyc7XG5pbXBvcnQge1xuICBzaGFyZWRTdGFzaEZ1bmN0aW9uLFxuICByZW1vdmVMaXN0ZW5lcnMsXG4gIGludm9rZVJlZ2lzdGVyZWRMaXN0ZW5lcnMsXG4gIEV2ZW50Q29udHJhY3REZXRhaWxzLFxuICBKU0FDVElPTl9FVkVOVF9DT05UUkFDVCxcbn0gZnJvbSAnLi4vZXZlbnRfZGVsZWdhdGlvbl91dGlscyc7XG5pbXBvcnQge0FQUF9JRH0gZnJvbSAnLi4vYXBwbGljYXRpb24vYXBwbGljYXRpb25fdG9rZW5zJztcbmltcG9ydCB7cGVyZm9ybWFuY2VNYXJrRmVhdHVyZX0gZnJvbSAnLi4vdXRpbC9wZXJmb3JtYW5jZSc7XG5cbi8qKlxuICogQSBzZXQgb2YgRE9NIGVsZW1lbnRzIHdpdGggYGpzYWN0aW9uYCBhdHRyaWJ1dGVzLlxuICovXG5jb25zdCBqc2FjdGlvblNldCA9IG5ldyBTZXQ8RWxlbWVudD4oKTtcblxuZnVuY3Rpb24gaXNHbG9iYWxFdmVudERlbGVnYXRpb25FbmFibGVkKGluamVjdG9yOiBJbmplY3Rvcikge1xuICByZXR1cm4gaW5qZWN0b3IuZ2V0KElTX0dMT0JBTF9FVkVOVF9ERUxFR0FUSU9OX0VOQUJMRUQsIGZhbHNlKTtcbn1cblxuLyoqXG4gKiBEZXRlcm1pbmVzIHdoZXRoZXIgRXZlbnQgUmVwbGF5IGZlYXR1cmUgc2hvdWxkIGJlIGFjdGl2YXRlZCBvbiB0aGUgY2xpZW50LlxuICovXG5mdW5jdGlvbiBzaG91bGRFbmFibGVFdmVudFJlcGxheShpbmplY3RvcjogSW5qZWN0b3IpIHtcbiAgcmV0dXJuIChcbiAgICBpbmplY3Rvci5nZXQoSVNfRVZFTlRfUkVQTEFZX0VOQUJMRUQsIEVWRU5UX1JFUExBWV9FTkFCTEVEX0RFRkFVTFQpICYmXG4gICAgIWlzR2xvYmFsRXZlbnREZWxlZ2F0aW9uRW5hYmxlZChpbmplY3RvcilcbiAgKTtcbn1cblxuLyoqXG4gKiBSZXR1cm5zIGEgc2V0IG9mIHByb3ZpZGVycyByZXF1aXJlZCB0byBzZXR1cCBzdXBwb3J0IGZvciBldmVudCByZXBsYXkuXG4gKiBSZXF1aXJlcyBoeWRyYXRpb24gdG8gYmUgZW5hYmxlZCBzZXBhcmF0ZWx5LlxuICovXG5leHBvcnQgZnVuY3Rpb24gd2l0aEV2ZW50UmVwbGF5KCk6IFByb3ZpZGVyW10ge1xuICByZXR1cm4gW1xuICAgIHtcbiAgICAgIHByb3ZpZGU6IElTX0VWRU5UX1JFUExBWV9FTkFCTEVELFxuICAgICAgdXNlRmFjdG9yeTogKCkgPT4ge1xuICAgICAgICBsZXQgaXNFbmFibGVkID0gdHJ1ZTtcbiAgICAgICAgaWYgKGlzUGxhdGZvcm1Ccm93c2VyKCkpIHtcbiAgICAgICAgICAvLyBOb3RlOiBnbG9iYWxUaGlzW0NPTlRSQUNUX1BST1BFUlRZXSBtYXkgYmUgdW5kZWZpbmVkIGluIGNhc2UgRXZlbnQgUmVwbGF5IGZlYXR1cmVcbiAgICAgICAgICAvLyBpcyBlbmFibGVkLCBidXQgdGhlcmUgYXJlIG5vIGV2ZW50cyBjb25maWd1cmVkIGluIHRoaXMgYXBwbGljYXRpb24sIGluIHdoaWNoIGNhc2VcbiAgICAgICAgICAvLyB3ZSBkb24ndCBhY3RpdmF0ZSB0aGlzIGZlYXR1cmUsIHNpbmNlIHRoZXJlIGFyZSBubyBldmVudHMgdG8gcmVwbGF5LlxuICAgICAgICAgIGNvbnN0IGFwcElkID0gaW5qZWN0KEFQUF9JRCk7XG4gICAgICAgICAgaXNFbmFibGVkID0gISF3aW5kb3cuX2Vqc2FzPy5bYXBwSWRdO1xuICAgICAgICB9XG4gICAgICAgIGlmIChpc0VuYWJsZWQpIHtcbiAgICAgICAgICBwZXJmb3JtYW5jZU1hcmtGZWF0dXJlKCdOZ0V2ZW50UmVwbGF5Jyk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIGlzRW5hYmxlZDtcbiAgICAgIH0sXG4gICAgfSxcbiAgICB7XG4gICAgICBwcm92aWRlOiBFTlZJUk9OTUVOVF9JTklUSUFMSVpFUixcbiAgICAgIHVzZVZhbHVlOiAoKSA9PiB7XG4gICAgICAgIGNvbnN0IGluamVjdG9yID0gaW5qZWN0KEluamVjdG9yKTtcbiAgICAgICAgaWYgKGlzUGxhdGZvcm1Ccm93c2VyKGluamVjdG9yKSAmJiBzaG91bGRFbmFibGVFdmVudFJlcGxheShpbmplY3RvcikpIHtcbiAgICAgICAgICBzZXRTdGFzaEZuKChyRWw6IFJFbGVtZW50LCBldmVudE5hbWU6IHN0cmluZywgbGlzdGVuZXJGbjogVm9pZEZ1bmN0aW9uKSA9PiB7XG4gICAgICAgICAgICBzaGFyZWRTdGFzaEZ1bmN0aW9uKHJFbCwgZXZlbnROYW1lLCBsaXN0ZW5lckZuKTtcbiAgICAgICAgICAgIGpzYWN0aW9uU2V0LmFkZChyRWwgYXMgdW5rbm93biBhcyBFbGVtZW50KTtcbiAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIG11bHRpOiB0cnVlLFxuICAgIH0sXG4gICAge1xuICAgICAgcHJvdmlkZTogQVBQX0JPT1RTVFJBUF9MSVNURU5FUixcbiAgICAgIHVzZUZhY3Rvcnk6ICgpID0+IHtcbiAgICAgICAgaWYgKGlzUGxhdGZvcm1Ccm93c2VyKCkpIHtcbiAgICAgICAgICBjb25zdCBpbmplY3RvciA9IGluamVjdChJbmplY3Rvcik7XG4gICAgICAgICAgY29uc3QgYXBwUmVmID0gaW5qZWN0KEFwcGxpY2F0aW9uUmVmKTtcbiAgICAgICAgICByZXR1cm4gKCkgPT4ge1xuICAgICAgICAgICAgaWYgKCFzaG91bGRFbmFibGVFdmVudFJlcGxheShpbmplY3RvcikpIHtcbiAgICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICAvLyBLaWNrIG9mZiBldmVudCByZXBsYXkgbG9naWMgb25jZSBoeWRyYXRpb24gZm9yIHRoZSBpbml0aWFsIHBhcnRcbiAgICAgICAgICAgIC8vIG9mIHRoZSBhcHBsaWNhdGlvbiBpcyBjb21wbGV0ZWQuIFRoaXMgdGltaW5nIGlzIHNpbWlsYXIgdG8gdGhlIHVuY2xhaW1lZFxuICAgICAgICAgICAgLy8gZGVoeWRyYXRlZCB2aWV3cyBjbGVhbnVwIHRpbWluZy5cbiAgICAgICAgICAgIHdoZW5TdGFibGUoYXBwUmVmKS50aGVuKCgpID0+IHtcbiAgICAgICAgICAgICAgY29uc3QgZXZlbnRDb250cmFjdERldGFpbHMgPSBpbmplY3Rvci5nZXQoSlNBQ1RJT05fRVZFTlRfQ09OVFJBQ1QpO1xuICAgICAgICAgICAgICBpbml0RXZlbnRSZXBsYXkoZXZlbnRDb250cmFjdERldGFpbHMsIGluamVjdG9yKTtcbiAgICAgICAgICAgICAganNhY3Rpb25TZXQuZm9yRWFjaChyZW1vdmVMaXN0ZW5lcnMpO1xuICAgICAgICAgICAgICAvLyBBZnRlciBoeWRyYXRpb24sIHdlIHNob3VsZG4ndCBuZWVkIHRvIGRvIGFueW1vcmUgd29yayByZWxhdGVkIHRvXG4gICAgICAgICAgICAgIC8vIGV2ZW50IHJlcGxheSBhbnltb3JlLlxuICAgICAgICAgICAgICBzZXRTdGFzaEZuKCgpID0+IHt9KTtcbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgIH07XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuICgpID0+IHt9OyAvLyBub29wIGZvciB0aGUgc2VydmVyIGNvZGVcbiAgICAgIH0sXG4gICAgICBtdWx0aTogdHJ1ZSxcbiAgICB9LFxuICBdO1xufVxuXG5jb25zdCBpbml0RXZlbnRSZXBsYXkgPSAoZXZlbnREZWxlZ2F0aW9uOiBFdmVudENvbnRyYWN0RGV0YWlscywgaW5qZWN0b3I6IEluamVjdG9yKSA9PiB7XG4gIGNvbnN0IGFwcElkID0gaW5qZWN0b3IuZ2V0KEFQUF9JRCk7XG4gIC8vIFRoaXMgaXMgc2V0IGluIHBhY2thZ2VzL3BsYXRmb3JtLXNlcnZlci9zcmMvdXRpbHMudHNcbiAgY29uc3QgZWFybHlKc2FjdGlvbkRhdGEgPSB3aW5kb3cuX2Vqc2FzIVthcHBJZF0hO1xuICBjb25zdCBldmVudENvbnRyYWN0ID0gKGV2ZW50RGVsZWdhdGlvbi5pbnN0YW5jZSA9IG5ldyBFdmVudENvbnRyYWN0KFxuICAgIG5ldyBFdmVudENvbnRyYWN0Q29udGFpbmVyKGVhcmx5SnNhY3Rpb25EYXRhLmMpLFxuICApKTtcbiAgZm9yIChjb25zdCBldCBvZiBlYXJseUpzYWN0aW9uRGF0YS5ldCkge1xuICAgIGV2ZW50Q29udHJhY3QuYWRkRXZlbnQoZXQpO1xuICB9XG4gIGZvciAoY29uc3QgZXQgb2YgZWFybHlKc2FjdGlvbkRhdGEuZXRjKSB7XG4gICAgZXZlbnRDb250cmFjdC5hZGRFdmVudChldCk7XG4gIH1cbiAgY29uc3QgZXZlbnRJbmZvcyA9IGdldEFwcFNjb3BlZFF1ZXVlZEV2ZW50SW5mb3MoYXBwSWQpO1xuICBldmVudENvbnRyYWN0LnJlcGxheUVhcmx5RXZlbnRJbmZvcyhldmVudEluZm9zKTtcbiAgY2xlYXJBcHBTY29wZWRFYXJseUV2ZW50Q29udHJhY3QoYXBwSWQpO1xuICBjb25zdCBkaXNwYXRjaGVyID0gbmV3IEV2ZW50RGlzcGF0Y2hlcihpbnZva2VSZWdpc3RlcmVkTGlzdGVuZXJzKTtcbiAgcmVnaXN0ZXJEaXNwYXRjaGVyKGV2ZW50Q29udHJhY3QsIGRpc3BhdGNoZXIpO1xufTtcblxuLyoqXG4gKiBFeHRyYWN0cyBpbmZvcm1hdGlvbiBhYm91dCBhbGwgRE9NIGV2ZW50cyAoYWRkZWQgaW4gYSB0ZW1wbGF0ZSkgcmVnaXN0ZXJlZCBvbiBlbGVtZW50cyBpbiBhIGdpdmVcbiAqIExWaWV3LiBNYXBzIGNvbGxlY3RlZCBldmVudHMgdG8gYSBjb3JyZXNwb25kaW5nIERPTSBlbGVtZW50IChhbiBlbGVtZW50IGlzIHVzZWQgYXMgYSBrZXkpLlxuICovXG5leHBvcnQgZnVuY3Rpb24gY29sbGVjdERvbUV2ZW50c0luZm8oXG4gIHRWaWV3OiBUVmlldyxcbiAgbFZpZXc6IExWaWV3LFxuICBldmVudFR5cGVzVG9SZXBsYXk6IHtyZWd1bGFyOiBTZXQ8c3RyaW5nPjsgY2FwdHVyZTogU2V0PHN0cmluZz59LFxuKTogTWFwPEVsZW1lbnQsIHN0cmluZ1tdPiB7XG4gIGNvbnN0IGRvbUV2ZW50c0luZm8gPSBuZXcgTWFwPEVsZW1lbnQsIHN0cmluZ1tdPigpO1xuICBjb25zdCBsQ2xlYW51cCA9IGxWaWV3W0NMRUFOVVBdO1xuICBjb25zdCB0Q2xlYW51cCA9IHRWaWV3LmNsZWFudXA7XG4gIGlmICghdENsZWFudXAgfHwgIWxDbGVhbnVwKSB7XG4gICAgcmV0dXJuIGRvbUV2ZW50c0luZm87XG4gIH1cbiAgZm9yIChsZXQgaSA9IDA7IGkgPCB0Q2xlYW51cC5sZW5ndGg7ICkge1xuICAgIGNvbnN0IGZpcnN0UGFyYW0gPSB0Q2xlYW51cFtpKytdO1xuICAgIGNvbnN0IHNlY29uZFBhcmFtID0gdENsZWFudXBbaSsrXTtcbiAgICBpZiAodHlwZW9mIGZpcnN0UGFyYW0gIT09ICdzdHJpbmcnKSB7XG4gICAgICBjb250aW51ZTtcbiAgICB9XG4gICAgY29uc3QgZXZlbnRUeXBlID0gZmlyc3RQYXJhbTtcbiAgICBpZiAoIWlzRWFybHlFdmVudFR5cGUoZXZlbnRUeXBlKSkge1xuICAgICAgY29udGludWU7XG4gICAgfVxuICAgIGlmIChpc0NhcHR1cmVFdmVudFR5cGUoZXZlbnRUeXBlKSkge1xuICAgICAgZXZlbnRUeXBlc1RvUmVwbGF5LmNhcHR1cmUuYWRkKGV2ZW50VHlwZSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGV2ZW50VHlwZXNUb1JlcGxheS5yZWd1bGFyLmFkZChldmVudFR5cGUpO1xuICAgIH1cbiAgICBjb25zdCBsaXN0ZW5lckVsZW1lbnQgPSB1bndyYXBSTm9kZShsVmlld1tzZWNvbmRQYXJhbV0pIGFzIGFueSBhcyBFbGVtZW50O1xuICAgIGkrKzsgLy8gbW92ZSB0aGUgY3Vyc29yIHRvIHRoZSBuZXh0IHBvc2l0aW9uIChsb2NhdGlvbiBvZiB0aGUgbGlzdGVuZXIgaWR4KVxuICAgIGNvbnN0IHVzZUNhcHR1cmVPckluZHggPSB0Q2xlYW51cFtpKytdO1xuICAgIC8vIGlmIHVzZUNhcHR1cmVPckluZHggaXMgYm9vbGVhbiB0aGVuIHJlcG9ydCBpdCBhcyBpcy5cbiAgICAvLyBpZiB1c2VDYXB0dXJlT3JJbmR4IGlzIHBvc2l0aXZlIG51bWJlciB0aGVuIGl0IGluIHVuc3Vic2NyaWJlIG1ldGhvZFxuICAgIC8vIGlmIHVzZUNhcHR1cmVPckluZHggaXMgbmVnYXRpdmUgbnVtYmVyIHRoZW4gaXQgaXMgYSBTdWJzY3JpcHRpb25cbiAgICBjb25zdCBpc0RvbUV2ZW50ID0gdHlwZW9mIHVzZUNhcHR1cmVPckluZHggPT09ICdib29sZWFuJyB8fCB1c2VDYXB0dXJlT3JJbmR4ID49IDA7XG4gICAgaWYgKCFpc0RvbUV2ZW50KSB7XG4gICAgICBjb250aW51ZTtcbiAgICB9XG4gICAgaWYgKCFkb21FdmVudHNJbmZvLmhhcyhsaXN0ZW5lckVsZW1lbnQpKSB7XG4gICAgICBkb21FdmVudHNJbmZvLnNldChsaXN0ZW5lckVsZW1lbnQsIFtldmVudFR5cGVdKTtcbiAgICB9IGVsc2Uge1xuICAgICAgZG9tRXZlbnRzSW5mby5nZXQobGlzdGVuZXJFbGVtZW50KSEucHVzaChldmVudFR5cGUpO1xuICAgIH1cbiAgfVxuICByZXR1cm4gZG9tRXZlbnRzSW5mbztcbn1cbiJdfQ==