/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { stringify } from '../util/stringify';
export function noComponentFactoryError(component) {
    const error = Error(`No component factory found for ${stringify(component)}.`);
    error[ERROR_COMPONENT] = component;
    return error;
}
const ERROR_COMPONENT = 'ngComponent';
export function getComponent(error) {
    return error[ERROR_COMPONENT];
}
class _NullComponentFactoryResolver {
    resolveComponentFactory(component) {
        throw noComponentFactoryError(component);
    }
}
/**
 * A simple registry that maps `Components` to generated `ComponentFactory` classes
 * that can be used to create instances of components.
 * Use to obtain the factory for a given component type,
 * then use the factory's `create()` method to create a component of that type.
 *
 * Note: since v13, dynamic component creation via
 * [`ViewContainerRef.createComponent`](api/core/ViewContainerRef#createComponent)
 * does **not** require resolving component factory: component class can be used directly.
 *
 * @publicApi
 *
 * @deprecated Angular no longer requires Component factories. Please use other APIs where
 *     Component class can be used directly.
 */
export class ComponentFactoryResolver {
    static { this.NULL = new _NullComponentFactoryResolver(); }
}
//# sourceMappingURL=data:application/json;base64,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