/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { stringify } from '../util/stringify';
/**
 * Map of module-id to the corresponding NgModule.
 */
const modules = new Map();
/**
 * Whether to check for duplicate NgModule registrations.
 *
 * This can be disabled for testing.
 */
let checkForDuplicateNgModules = true;
function assertSameOrNotExisting(id, type, incoming) {
    if (type && type !== incoming && checkForDuplicateNgModules) {
        throw new Error(`Duplicate module registered for ${id} - ${stringify(type)} vs ${stringify(type.name)}`);
    }
}
/**
 * Adds the given NgModule type to Angular's NgModule registry.
 *
 * This is generated as a side-effect of NgModule compilation. Note that the `id` is passed in
 * explicitly and not read from the NgModule definition. This is for two reasons: it avoids a
 * megamorphic read, and in JIT there's a chicken-and-egg problem where the NgModule may not be
 * fully resolved when it's registered.
 *
 * @codeGenApi
 */
export function registerNgModuleType(ngModuleType, id) {
    const existing = modules.get(id) || null;
    assertSameOrNotExisting(id, existing, ngModuleType);
    modules.set(id, ngModuleType);
}
export function clearModulesForTest() {
    modules.clear();
}
export function getRegisteredNgModuleType(id) {
    return modules.get(id);
}
/**
 * Control whether the NgModule registration system enforces that each NgModule type registered has
 * a unique id.
 *
 * This is useful for testing as the NgModule registry cannot be properly reset between tests with
 * Angular's current API.
 */
export function setAllowDuplicateNgModuleIdsForTest(allowDuplicates) {
    checkForDuplicateNgModules = !allowDuplicates;
}
//# sourceMappingURL=data:application/json;base64,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