/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { getCurrentTNode, getLView } from '../render3/state';
import { createAndRenderEmbeddedLView } from '../render3/view_manipulation';
import { ViewRef as R3_ViewRef } from '../render3/view_ref';
import { assertDefined } from '../util/assert';
import { createElementRef } from './element_ref';
/**
 * Represents an embedded template that can be used to instantiate embedded views.
 * To instantiate embedded views based on a template, use the `ViewContainerRef`
 * method `createEmbeddedView()`.
 *
 * Access a `TemplateRef` instance by placing a directive on an `<ng-template>`
 * element (or directive prefixed with `*`). The `TemplateRef` for the embedded view
 * is injected into the constructor of the directive,
 * using the `TemplateRef` token.
 *
 * You can also use a `Query` to find a `TemplateRef` associated with
 * a component or a directive.
 *
 * @see {@link ViewContainerRef}
 *
 * @publicApi
 */
export class TemplateRef {
    /**
     * @internal
     * @nocollapse
     */
    static { this.__NG_ELEMENT_ID__ = injectTemplateRef; }
}
const ViewEngineTemplateRef = TemplateRef;
// TODO(alxhub): combine interface and implementation. Currently this is challenging since something
// in g3 depends on them being separate.
const R3TemplateRef = class TemplateRef extends ViewEngineTemplateRef {
    constructor(_declarationLView, _declarationTContainer, elementRef) {
        super();
        this._declarationLView = _declarationLView;
        this._declarationTContainer = _declarationTContainer;
        this.elementRef = elementRef;
    }
    /**
     * Returns an `ssrId` associated with a TView, which was used to
     * create this instance of the `TemplateRef`.
     *
     * @internal
     */
    get ssrId() {
        return this._declarationTContainer.tView?.ssrId || null;
    }
    createEmbeddedView(context, injector) {
        return this.createEmbeddedViewImpl(context, injector);
    }
    /**
     * @internal
     */
    createEmbeddedViewImpl(context, injector, dehydratedView) {
        const embeddedLView = createAndRenderEmbeddedLView(this._declarationLView, this._declarationTContainer, context, { embeddedViewInjector: injector, dehydratedView });
        return new R3_ViewRef(embeddedLView);
    }
};
/**
 * Creates a TemplateRef given a node.
 *
 * @returns The TemplateRef instance to use
 */
export function injectTemplateRef() {
    return createTemplateRef(getCurrentTNode(), getLView());
}
/**
 * Creates a TemplateRef and stores it on the injector.
 *
 * @param hostTNode The node on which a TemplateRef is requested
 * @param hostLView The `LView` to which the node belongs
 * @returns The TemplateRef instance or null if we can't create a TemplateRef on a given node type
 */
export function createTemplateRef(hostTNode, hostLView) {
    if (hostTNode.type & 4 /* TNodeType.Container */) {
        ngDevMode && assertDefined(hostTNode.tView, 'TView must be allocated');
        return new R3TemplateRef(hostLView, hostTNode, createElementRef(hostTNode, hostLView));
    }
    return null;
}
//# sourceMappingURL=data:application/json;base64,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