/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { makePropDecorator } from '../util/decorators';
// Stores the default value of `emitDistinctChangesOnly` when the `emitDistinctChangesOnly` is not
// explicitly set.
export const emitDistinctChangesOnlyDefaultValue = true;
/**
 * Base class for query metadata.
 *
 * @see {@link ContentChildren}
 * @see {@link ContentChild}
 * @see {@link ViewChildren}
 * @see {@link ViewChild}
 *
 * @publicApi
 */
export class Query {
}
/**
 * ContentChildren decorator and metadata.
 *
 *
 * @Annotation
 * @publicApi
 */
export const ContentChildren = makePropDecorator('ContentChildren', (selector, opts = {}) => ({
    selector,
    first: false,
    isViewQuery: false,
    descendants: false,
    emitDistinctChangesOnly: emitDistinctChangesOnlyDefaultValue,
    ...opts,
}), Query);
/**
 * ContentChild decorator and metadata.
 *
 *
 * @Annotation
 *
 * @publicApi
 */
export const ContentChild = makePropDecorator('ContentChild', (selector, opts = {}) => ({
    selector,
    first: true,
    isViewQuery: false,
    descendants: true,
    ...opts,
}), Query);
/**
 * ViewChildren decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export const ViewChildren = makePropDecorator('ViewChildren', (selector, opts = {}) => ({
    selector,
    first: false,
    isViewQuery: true,
    descendants: true,
    emitDistinctChangesOnly: emitDistinctChangesOnlyDefaultValue,
    ...opts,
}), Query);
/**
 * ViewChild decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export const ViewChild = makePropDecorator('ViewChild', (selector, opts) => ({
    selector,
    first: true,
    isViewQuery: true,
    descendants: true,
    ...opts,
}), Query);
//# sourceMappingURL=data:application/json;base64,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