/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { compileComponent, compileDirective } from '../render3/jit/directive';
import { compilePipe } from '../render3/jit/pipe';
import { makeDecorator, makePropDecorator } from '../util/decorators';
/**
 * Type of the Directive metadata.
 *
 * @publicApi
 */
export const Directive = makeDecorator('Directive', (dir = {}) => dir, undefined, undefined, (type, meta) => compileDirective(type, meta));
/**
 * Component decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export const Component = makeDecorator('Component', (c = {}) => ({ changeDetection: ChangeDetectionStrategy.Default, ...c }), Directive, undefined, (type, meta) => compileComponent(type, meta));
/**
 * @Annotation
 * @publicApi
 */
export const Pipe = makeDecorator('Pipe', (p) => ({ pure: true, ...p }), undefined, undefined, (type, meta) => compilePipe(type, meta));
/**
 * @Annotation
 * @publicApi
 */
export const Input = makePropDecorator('Input', (arg) => {
    if (!arg) {
        return {};
    }
    return typeof arg === 'string' ? { alias: arg } : arg;
});
/**
 * @Annotation
 * @publicApi
 */
export const Output = makePropDecorator('Output', (alias) => ({ alias }));
/**
 * @Annotation
 * @publicApi
 */
export const HostBinding = makePropDecorator('HostBinding', (hostPropertyName) => ({ hostPropertyName }));
/**
 * @Annotation
 * @publicApi
 */
export const HostListener = makePropDecorator('HostListener', (eventName, args) => ({ eventName, args }));
//# sourceMappingURL=data:application/json;base64,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