/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { RuntimeError } from '../errors';
import { assertDefined, assertEqual, assertNumber, throwError } from '../util/assert';
import { getComponentDef, getNgModuleDef } from './definition';
import { isLContainer, isLView } from './interfaces/type_checks';
import { DECLARATION_COMPONENT_VIEW, HEADER_OFFSET, T_HOST, TVIEW, } from './interfaces/view';
// [Assert functions do not constraint type when they are guarded by a truthy
// expression.](https://github.com/microsoft/TypeScript/issues/37295)
export function assertTNodeForLView(tNode, lView) {
    assertTNodeForTView(tNode, lView[TVIEW]);
}
export function assertTNodeForTView(tNode, tView) {
    assertTNode(tNode);
    const tData = tView.data;
    for (let i = HEADER_OFFSET; i < tData.length; i++) {
        if (tData[i] === tNode) {
            return;
        }
    }
    throwError('This TNode does not belong to this TView.');
}
export function assertTNode(tNode) {
    assertDefined(tNode, 'TNode must be defined');
    if (!(tNode && typeof tNode === 'object' && tNode.hasOwnProperty('directiveStylingLast'))) {
        throwError('Not of type TNode, got: ' + tNode);
    }
}
export function assertTIcu(tIcu) {
    assertDefined(tIcu, 'Expected TIcu to be defined');
    if (!(typeof tIcu.currentCaseLViewIndex === 'number')) {
        throwError('Object is not of TIcu type.');
    }
}
export function assertComponentType(actual, msg = "Type passed in is not ComponentType, it does not have 'ɵcmp' property.") {
    if (!getComponentDef(actual)) {
        throwError(msg);
    }
}
export function assertNgModuleType(actual, msg = "Type passed in is not NgModuleType, it does not have 'ɵmod' property.") {
    if (!getNgModuleDef(actual)) {
        throwError(msg);
    }
}
export function assertCurrentTNodeIsParent(isParent) {
    assertEqual(isParent, true, 'currentTNode should be a parent');
}
export function assertHasParent(tNode) {
    assertDefined(tNode, 'currentTNode should exist!');
    assertDefined(tNode.parent, 'currentTNode should have a parent');
}
export function assertLContainer(value) {
    assertDefined(value, 'LContainer must be defined');
    assertEqual(isLContainer(value), true, 'Expecting LContainer');
}
export function assertLViewOrUndefined(value) {
    value && assertEqual(isLView(value), true, 'Expecting LView or undefined or null');
}
export function assertLView(value) {
    assertDefined(value, 'LView must be defined');
    assertEqual(isLView(value), true, 'Expecting LView');
}
export function assertFirstCreatePass(tView, errMessage) {
    assertEqual(tView.firstCreatePass, true, errMessage || 'Should only be called in first create pass.');
}
export function assertFirstUpdatePass(tView, errMessage) {
    assertEqual(tView.firstUpdatePass, true, errMessage || 'Should only be called in first update pass.');
}
/**
 * This is a basic sanity check that an object is probably a directive def. DirectiveDef is
 * an interface, so we can't do a direct instanceof check.
 */
export function assertDirectiveDef(obj) {
    if (obj.type === undefined || obj.selectors == undefined || obj.inputs === undefined) {
        throwError(`Expected a DirectiveDef/ComponentDef and this object does not seem to have the expected shape.`);
    }
}
export function assertIndexInDeclRange(tView, index) {
    assertBetween(HEADER_OFFSET, tView.bindingStartIndex, index);
}
export function assertIndexInExpandoRange(lView, index) {
    const tView = lView[1];
    assertBetween(tView.expandoStartIndex, lView.length, index);
}
export function assertBetween(lower, upper, index) {
    if (!(lower <= index && index < upper)) {
        throwError(`Index out of range (expecting ${lower} <= ${index} < ${upper})`);
    }
}
export function assertProjectionSlots(lView, errMessage) {
    assertDefined(lView[DECLARATION_COMPONENT_VIEW], 'Component views should exist.');
    assertDefined(lView[DECLARATION_COMPONENT_VIEW][T_HOST].projection, errMessage ||
        'Components with projection nodes (<ng-content>) must have projection slots defined.');
}
export function assertParentView(lView, errMessage) {
    assertDefined(lView, errMessage || "Component views should always have a parent view (component's host view)");
}
export function assertNoDuplicateDirectives(directives) {
    // The array needs at least two elements in order to have duplicates.
    if (directives.length < 2) {
        return;
    }
    const seenDirectives = new Set();
    for (const current of directives) {
        if (seenDirectives.has(current)) {
            throw new RuntimeError(309 /* RuntimeErrorCode.DUPLICATE_DIRECTIVE */, `Directive ${current.type.name} matches multiple times on the same element. ` +
                `Directives can only match an element once.`);
        }
        seenDirectives.add(current);
    }
}
/**
 * This is a basic sanity check that the `injectorIndex` seems to point to what looks like a
 * NodeInjector data structure.
 *
 * @param lView `LView` which should be checked.
 * @param injectorIndex index into the `LView` where the `NodeInjector` is expected.
 */
export function assertNodeInjector(lView, injectorIndex) {
    assertIndexInExpandoRange(lView, injectorIndex);
    assertIndexInExpandoRange(lView, injectorIndex + 8 /* NodeInjectorOffset.PARENT */);
    assertNumber(lView[injectorIndex + 0], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 1], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 2], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 3], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 4], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 5], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 6], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 7], 'injectorIndex should point to a bloom filter');
    assertNumber(lView[injectorIndex + 8 /* NodeInjectorOffset.PARENT */], 'injectorIndex should point to parent injector');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXNzZXJ0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vLi4vLi4vcGFja2FnZXMvY29yZS9zcmMvcmVuZGVyMy9hc3NlcnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7OztHQU1HO0FBRUgsT0FBTyxFQUFDLFlBQVksRUFBbUIsTUFBTSxXQUFXLENBQUM7QUFDekQsT0FBTyxFQUFDLGFBQWEsRUFBRSxXQUFXLEVBQUUsWUFBWSxFQUFFLFVBQVUsRUFBQyxNQUFNLGdCQUFnQixDQUFDO0FBRXBGLE9BQU8sRUFBQyxlQUFlLEVBQUUsY0FBYyxFQUFDLE1BQU0sY0FBYyxDQUFDO0FBTTdELE9BQU8sRUFBQyxZQUFZLEVBQUUsT0FBTyxFQUFDLE1BQU0sMEJBQTBCLENBQUM7QUFDL0QsT0FBTyxFQUNMLDBCQUEwQixFQUUxQixhQUFhLEVBR2IsTUFBTSxFQUNOLEtBQUssR0FFTixNQUFNLG1CQUFtQixDQUFDO0FBRTNCLDZFQUE2RTtBQUM3RSxxRUFBcUU7QUFFckUsTUFBTSxVQUFVLG1CQUFtQixDQUFDLEtBQVksRUFBRSxLQUFZO0lBQzVELG1CQUFtQixDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztBQUMzQyxDQUFDO0FBRUQsTUFBTSxVQUFVLG1CQUFtQixDQUFDLEtBQVksRUFBRSxLQUFZO0lBQzVELFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNuQixNQUFNLEtBQUssR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDO0lBQ3pCLEtBQUssSUFBSSxDQUFDLEdBQUcsYUFBYSxFQUFFLENBQUMsR0FBRyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUM7UUFDbEQsSUFBSSxLQUFLLENBQUMsQ0FBQyxDQUFDLEtBQUssS0FBSyxFQUFFLENBQUM7WUFDdkIsT0FBTztRQUNULENBQUM7SUFDSCxDQUFDO0lBQ0QsVUFBVSxDQUFDLDJDQUEyQyxDQUFDLENBQUM7QUFDMUQsQ0FBQztBQUVELE1BQU0sVUFBVSxXQUFXLENBQUMsS0FBWTtJQUN0QyxhQUFhLENBQUMsS0FBSyxFQUFFLHVCQUF1QixDQUFDLENBQUM7SUFDOUMsSUFBSSxDQUFDLENBQUMsS0FBSyxJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsSUFBSSxLQUFLLENBQUMsY0FBYyxDQUFDLHNCQUFzQixDQUFDLENBQUMsRUFBRSxDQUFDO1FBQzFGLFVBQVUsQ0FBQywwQkFBMEIsR0FBRyxLQUFLLENBQUMsQ0FBQztJQUNqRCxDQUFDO0FBQ0gsQ0FBQztBQUVELE1BQU0sVUFBVSxVQUFVLENBQUMsSUFBVTtJQUNuQyxhQUFhLENBQUMsSUFBSSxFQUFFLDZCQUE2QixDQUFDLENBQUM7SUFDbkQsSUFBSSxDQUFDLENBQUMsT0FBTyxJQUFJLENBQUMscUJBQXFCLEtBQUssUUFBUSxDQUFDLEVBQUUsQ0FBQztRQUN0RCxVQUFVLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0FBQ0gsQ0FBQztBQUVELE1BQU0sVUFBVSxtQkFBbUIsQ0FDakMsTUFBVyxFQUNYLE1BQWMsd0VBQXdFO0lBRXRGLElBQUksQ0FBQyxlQUFlLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQztRQUM3QixVQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDbEIsQ0FBQztBQUNILENBQUM7QUFFRCxNQUFNLFVBQVUsa0JBQWtCLENBQ2hDLE1BQVcsRUFDWCxNQUFjLHVFQUF1RTtJQUVyRixJQUFJLENBQUMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUM7UUFDNUIsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ2xCLENBQUM7QUFDSCxDQUFDO0FBRUQsTUFBTSxVQUFVLDBCQUEwQixDQUFDLFFBQWlCO0lBQzFELFdBQVcsQ0FBQyxRQUFRLEVBQUUsSUFBSSxFQUFFLGlDQUFpQyxDQUFDLENBQUM7QUFDakUsQ0FBQztBQUVELE1BQU0sVUFBVSxlQUFlLENBQUMsS0FBbUI7SUFDakQsYUFBYSxDQUFDLEtBQUssRUFBRSw0QkFBNEIsQ0FBQyxDQUFDO0lBQ25ELGFBQWEsQ0FBQyxLQUFNLENBQUMsTUFBTSxFQUFFLG1DQUFtQyxDQUFDLENBQUM7QUFDcEUsQ0FBQztBQUVELE1BQU0sVUFBVSxnQkFBZ0IsQ0FBQyxLQUFVO0lBQ3pDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsNEJBQTRCLENBQUMsQ0FBQztJQUNuRCxXQUFXLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxFQUFFLElBQUksRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO0FBQ2pFLENBQUM7QUFFRCxNQUFNLFVBQVUsc0JBQXNCLENBQUMsS0FBVTtJQUMvQyxLQUFLLElBQUksV0FBVyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRSxJQUFJLEVBQUUsc0NBQXNDLENBQUMsQ0FBQztBQUNyRixDQUFDO0FBRUQsTUFBTSxVQUFVLFdBQVcsQ0FBQyxLQUFVO0lBQ3BDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsdUJBQXVCLENBQUMsQ0FBQztJQUM5QyxXQUFXLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFLElBQUksRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO0FBQ3ZELENBQUM7QUFFRCxNQUFNLFVBQVUscUJBQXFCLENBQUMsS0FBWSxFQUFFLFVBQW1CO0lBQ3JFLFdBQVcsQ0FDVCxLQUFLLENBQUMsZUFBZSxFQUNyQixJQUFJLEVBQ0osVUFBVSxJQUFJLDZDQUE2QyxDQUM1RCxDQUFDO0FBQ0osQ0FBQztBQUVELE1BQU0sVUFBVSxxQkFBcUIsQ0FBQyxLQUFZLEVBQUUsVUFBbUI7SUFDckUsV0FBVyxDQUNULEtBQUssQ0FBQyxlQUFlLEVBQ3JCLElBQUksRUFDSixVQUFVLElBQUksNkNBQTZDLENBQzVELENBQUM7QUFDSixDQUFDO0FBRUQ7OztHQUdHO0FBQ0gsTUFBTSxVQUFVLGtCQUFrQixDQUFJLEdBQVE7SUFDNUMsSUFBSSxHQUFHLENBQUMsSUFBSSxLQUFLLFNBQVMsSUFBSSxHQUFHLENBQUMsU0FBUyxJQUFJLFNBQVMsSUFBSSxHQUFHLENBQUMsTUFBTSxLQUFLLFNBQVMsRUFBRSxDQUFDO1FBQ3JGLFVBQVUsQ0FDUixnR0FBZ0csQ0FDakcsQ0FBQztJQUNKLENBQUM7QUFDSCxDQUFDO0FBRUQsTUFBTSxVQUFVLHNCQUFzQixDQUFDLEtBQVksRUFBRSxLQUFhO0lBQ2hFLGFBQWEsQ0FBQyxhQUFhLEVBQUUsS0FBSyxDQUFDLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxDQUFDO0FBQy9ELENBQUM7QUFFRCxNQUFNLFVBQVUseUJBQXlCLENBQUMsS0FBWSxFQUFFLEtBQWE7SUFDbkUsTUFBTSxLQUFLLEdBQUcsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3ZCLGFBQWEsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsQ0FBQztBQUM5RCxDQUFDO0FBRUQsTUFBTSxVQUFVLGFBQWEsQ0FBQyxLQUFhLEVBQUUsS0FBYSxFQUFFLEtBQWE7SUFDdkUsSUFBSSxDQUFDLENBQUMsS0FBSyxJQUFJLEtBQUssSUFBSSxLQUFLLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUN2QyxVQUFVLENBQUMsaUNBQWlDLEtBQUssT0FBTyxLQUFLLE1BQU0sS0FBSyxHQUFHLENBQUMsQ0FBQztJQUMvRSxDQUFDO0FBQ0gsQ0FBQztBQUVELE1BQU0sVUFBVSxxQkFBcUIsQ0FBQyxLQUFZLEVBQUUsVUFBbUI7SUFDckUsYUFBYSxDQUFDLEtBQUssQ0FBQywwQkFBMEIsQ0FBQyxFQUFFLCtCQUErQixDQUFDLENBQUM7SUFDbEYsYUFBYSxDQUNYLEtBQUssQ0FBQywwQkFBMEIsQ0FBQyxDQUFDLE1BQU0sQ0FBRSxDQUFDLFVBQVUsRUFDckQsVUFBVTtRQUNSLHFGQUFxRixDQUN4RixDQUFDO0FBQ0osQ0FBQztBQUVELE1BQU0sVUFBVSxnQkFBZ0IsQ0FBQyxLQUFtQixFQUFFLFVBQW1CO0lBQ3ZFLGFBQWEsQ0FDWCxLQUFLLEVBQ0wsVUFBVSxJQUFJLDBFQUEwRSxDQUN6RixDQUFDO0FBQ0osQ0FBQztBQUVELE1BQU0sVUFBVSwyQkFBMkIsQ0FBQyxVQUFtQztJQUM3RSxxRUFBcUU7SUFDckUsSUFBSSxVQUFVLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRSxDQUFDO1FBQzFCLE9BQU87SUFDVCxDQUFDO0lBRUQsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLEVBQXlCLENBQUM7SUFFeEQsS0FBSyxNQUFNLE9BQU8sSUFBSSxVQUFVLEVBQUUsQ0FBQztRQUNqQyxJQUFJLGNBQWMsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUNoQyxNQUFNLElBQUksWUFBWSxpREFFcEIsYUFBYSxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksK0NBQStDO2dCQUMzRSw0Q0FBNEMsQ0FDL0MsQ0FBQztRQUNKLENBQUM7UUFDRCxjQUFjLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQzlCLENBQUM7QUFDSCxDQUFDO0FBRUQ7Ozs7OztHQU1HO0FBQ0gsTUFBTSxVQUFVLGtCQUFrQixDQUFDLEtBQVksRUFBRSxhQUFxQjtJQUNwRSx5QkFBeUIsQ0FBQyxLQUFLLEVBQUUsYUFBYSxDQUFDLENBQUM7SUFDaEQseUJBQXlCLENBQUMsS0FBSyxFQUFFLGFBQWEsb0NBQTRCLENBQUMsQ0FBQztJQUM1RSxZQUFZLENBQUMsS0FBSyxDQUFDLGFBQWEsR0FBRyxDQUFDLENBQUMsRUFBRSw4Q0FBOEMsQ0FBQyxDQUFDO0lBQ3ZGLFlBQVksQ0FBQyxLQUFLLENBQUMsYUFBYSxHQUFHLENBQUMsQ0FBQyxFQUFFLDhDQUE4QyxDQUFDLENBQUM7SUFDdkYsWUFBWSxDQUFDLEtBQUssQ0FBQyxhQUFhLEdBQUcsQ0FBQyxDQUFDLEVBQUUsOENBQThDLENBQUMsQ0FBQztJQUN2RixZQUFZLENBQUMsS0FBSyxDQUFDLGFBQWEsR0FBRyxDQUFDLENBQUMsRUFBRSw4Q0FBOEMsQ0FBQyxDQUFDO0lBQ3ZGLFlBQVksQ0FBQyxLQUFLLENBQUMsYUFBYSxHQUFHLENBQUMsQ0FBQyxFQUFFLDhDQUE4QyxDQUFDLENBQUM7SUFDdkYsWUFBWSxDQUFDLEtBQUssQ0FBQyxhQUFhLEdBQUcsQ0FBQyxDQUFDLEVBQUUsOENBQThDLENBQUMsQ0FBQztJQUN2RixZQUFZLENBQUMsS0FBSyxDQUFDLGFBQWEsR0FBRyxDQUFDLENBQUMsRUFBRSw4Q0FBOEMsQ0FBQyxDQUFDO0lBQ3ZGLFlBQVksQ0FBQyxLQUFLLENBQUMsYUFBYSxHQUFHLENBQUMsQ0FBQyxFQUFFLDhDQUE4QyxDQUFDLENBQUM7SUFDdkYsWUFBWSxDQUNWLEtBQUssQ0FBQyxhQUFhLG9DQUE0QixDQUFDLEVBQ2hELCtDQUErQyxDQUNoRCxDQUFDO0FBQ0osQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgTExDIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogVXNlIG9mIHRoaXMgc291cmNlIGNvZGUgaXMgZ292ZXJuZWQgYnkgYW4gTUlULXN0eWxlIGxpY2Vuc2UgdGhhdCBjYW4gYmVcbiAqIGZvdW5kIGluIHRoZSBMSUNFTlNFIGZpbGUgYXQgaHR0cHM6Ly9hbmd1bGFyLmlvL2xpY2Vuc2VcbiAqL1xuXG5pbXBvcnQge1J1bnRpbWVFcnJvciwgUnVudGltZUVycm9yQ29kZX0gZnJvbSAnLi4vZXJyb3JzJztcbmltcG9ydCB7YXNzZXJ0RGVmaW5lZCwgYXNzZXJ0RXF1YWwsIGFzc2VydE51bWJlciwgdGhyb3dFcnJvcn0gZnJvbSAnLi4vdXRpbC9hc3NlcnQnO1xuXG5pbXBvcnQge2dldENvbXBvbmVudERlZiwgZ2V0TmdNb2R1bGVEZWZ9IGZyb20gJy4vZGVmaW5pdGlvbic7XG5pbXBvcnQge0xDb250YWluZXJ9IGZyb20gJy4vaW50ZXJmYWNlcy9jb250YWluZXInO1xuaW1wb3J0IHtEaXJlY3RpdmVEZWZ9IGZyb20gJy4vaW50ZXJmYWNlcy9kZWZpbml0aW9uJztcbmltcG9ydCB7VEljdX0gZnJvbSAnLi9pbnRlcmZhY2VzL2kxOG4nO1xuaW1wb3J0IHtOb2RlSW5qZWN0b3JPZmZzZXR9IGZyb20gJy4vaW50ZXJmYWNlcy9pbmplY3Rvcic7XG5pbXBvcnQge1ROb2RlfSBmcm9tICcuL2ludGVyZmFjZXMvbm9kZSc7XG5pbXBvcnQge2lzTENvbnRhaW5lciwgaXNMVmlld30gZnJvbSAnLi9pbnRlcmZhY2VzL3R5cGVfY2hlY2tzJztcbmltcG9ydCB7XG4gIERFQ0xBUkFUSU9OX0NPTVBPTkVOVF9WSUVXLFxuICBGTEFHUyxcbiAgSEVBREVSX09GRlNFVCxcbiAgTFZpZXcsXG4gIExWaWV3RmxhZ3MsXG4gIFRfSE9TVCxcbiAgVFZJRVcsXG4gIFRWaWV3LFxufSBmcm9tICcuL2ludGVyZmFjZXMvdmlldyc7XG5cbi8vIFtBc3NlcnQgZnVuY3Rpb25zIGRvIG5vdCBjb25zdHJhaW50IHR5cGUgd2hlbiB0aGV5IGFyZSBndWFyZGVkIGJ5IGEgdHJ1dGh5XG4vLyBleHByZXNzaW9uLl0oaHR0cHM6Ly9naXRodWIuY29tL21pY3Jvc29mdC9UeXBlU2NyaXB0L2lzc3Vlcy8zNzI5NSlcblxuZXhwb3J0IGZ1bmN0aW9uIGFzc2VydFROb2RlRm9yTFZpZXcodE5vZGU6IFROb2RlLCBsVmlldzogTFZpZXcpIHtcbiAgYXNzZXJ0VE5vZGVGb3JUVmlldyh0Tm9kZSwgbFZpZXdbVFZJRVddKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFzc2VydFROb2RlRm9yVFZpZXcodE5vZGU6IFROb2RlLCB0VmlldzogVFZpZXcpIHtcbiAgYXNzZXJ0VE5vZGUodE5vZGUpO1xuICBjb25zdCB0RGF0YSA9IHRWaWV3LmRhdGE7XG4gIGZvciAobGV0IGkgPSBIRUFERVJfT0ZGU0VUOyBpIDwgdERhdGEubGVuZ3RoOyBpKyspIHtcbiAgICBpZiAodERhdGFbaV0gPT09IHROb2RlKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICB9XG4gIHRocm93RXJyb3IoJ1RoaXMgVE5vZGUgZG9lcyBub3QgYmVsb25nIHRvIHRoaXMgVFZpZXcuJyk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBhc3NlcnRUTm9kZSh0Tm9kZTogVE5vZGUpIHtcbiAgYXNzZXJ0RGVmaW5lZCh0Tm9kZSwgJ1ROb2RlIG11c3QgYmUgZGVmaW5lZCcpO1xuICBpZiAoISh0Tm9kZSAmJiB0eXBlb2YgdE5vZGUgPT09ICdvYmplY3QnICYmIHROb2RlLmhhc093blByb3BlcnR5KCdkaXJlY3RpdmVTdHlsaW5nTGFzdCcpKSkge1xuICAgIHRocm93RXJyb3IoJ05vdCBvZiB0eXBlIFROb2RlLCBnb3Q6ICcgKyB0Tm9kZSk7XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFzc2VydFRJY3UodEljdTogVEljdSkge1xuICBhc3NlcnREZWZpbmVkKHRJY3UsICdFeHBlY3RlZCBUSWN1IHRvIGJlIGRlZmluZWQnKTtcbiAgaWYgKCEodHlwZW9mIHRJY3UuY3VycmVudENhc2VMVmlld0luZGV4ID09PSAnbnVtYmVyJykpIHtcbiAgICB0aHJvd0Vycm9yKCdPYmplY3QgaXMgbm90IG9mIFRJY3UgdHlwZS4nKTtcbiAgfVxufVxuXG5leHBvcnQgZnVuY3Rpb24gYXNzZXJ0Q29tcG9uZW50VHlwZShcbiAgYWN0dWFsOiBhbnksXG4gIG1zZzogc3RyaW5nID0gXCJUeXBlIHBhc3NlZCBpbiBpcyBub3QgQ29tcG9uZW50VHlwZSwgaXQgZG9lcyBub3QgaGF2ZSAnybVjbXAnIHByb3BlcnR5LlwiLFxuKSB7XG4gIGlmICghZ2V0Q29tcG9uZW50RGVmKGFjdHVhbCkpIHtcbiAgICB0aHJvd0Vycm9yKG1zZyk7XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFzc2VydE5nTW9kdWxlVHlwZShcbiAgYWN0dWFsOiBhbnksXG4gIG1zZzogc3RyaW5nID0gXCJUeXBlIHBhc3NlZCBpbiBpcyBub3QgTmdNb2R1bGVUeXBlLCBpdCBkb2VzIG5vdCBoYXZlICfJtW1vZCcgcHJvcGVydHkuXCIsXG4pIHtcbiAgaWYgKCFnZXROZ01vZHVsZURlZihhY3R1YWwpKSB7XG4gICAgdGhyb3dFcnJvcihtc2cpO1xuICB9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBhc3NlcnRDdXJyZW50VE5vZGVJc1BhcmVudChpc1BhcmVudDogYm9vbGVhbikge1xuICBhc3NlcnRFcXVhbChpc1BhcmVudCwgdHJ1ZSwgJ2N1cnJlbnRUTm9kZSBzaG91bGQgYmUgYSBwYXJlbnQnKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFzc2VydEhhc1BhcmVudCh0Tm9kZTogVE5vZGUgfCBudWxsKSB7XG4gIGFzc2VydERlZmluZWQodE5vZGUsICdjdXJyZW50VE5vZGUgc2hvdWxkIGV4aXN0IScpO1xuICBhc3NlcnREZWZpbmVkKHROb2RlIS5wYXJlbnQsICdjdXJyZW50VE5vZGUgc2hvdWxkIGhhdmUgYSBwYXJlbnQnKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFzc2VydExDb250YWluZXIodmFsdWU6IGFueSk6IGFzc2VydHMgdmFsdWUgaXMgTENvbnRhaW5lciB7XG4gIGFzc2VydERlZmluZWQodmFsdWUsICdMQ29udGFpbmVyIG11c3QgYmUgZGVmaW5lZCcpO1xuICBhc3NlcnRFcXVhbChpc0xDb250YWluZXIodmFsdWUpLCB0cnVlLCAnRXhwZWN0aW5nIExDb250YWluZXInKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFzc2VydExWaWV3T3JVbmRlZmluZWQodmFsdWU6IGFueSk6IGFzc2VydHMgdmFsdWUgaXMgTFZpZXcgfCBudWxsIHwgdW5kZWZpbmVkIHtcbiAgdmFsdWUgJiYgYXNzZXJ0RXF1YWwoaXNMVmlldyh2YWx1ZSksIHRydWUsICdFeHBlY3RpbmcgTFZpZXcgb3IgdW5kZWZpbmVkIG9yIG51bGwnKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFzc2VydExWaWV3KHZhbHVlOiBhbnkpOiBhc3NlcnRzIHZhbHVlIGlzIExWaWV3IHtcbiAgYXNzZXJ0RGVmaW5lZCh2YWx1ZSwgJ0xWaWV3IG11c3QgYmUgZGVmaW5lZCcpO1xuICBhc3NlcnRFcXVhbChpc0xWaWV3KHZhbHVlKSwgdHJ1ZSwgJ0V4cGVjdGluZyBMVmlldycpO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYXNzZXJ0Rmlyc3RDcmVhdGVQYXNzKHRWaWV3OiBUVmlldywgZXJyTWVzc2FnZT86IHN0cmluZykge1xuICBhc3NlcnRFcXVhbChcbiAgICB0Vmlldy5maXJzdENyZWF0ZVBhc3MsXG4gICAgdHJ1ZSxcbiAgICBlcnJNZXNzYWdlIHx8ICdTaG91bGQgb25seSBiZSBjYWxsZWQgaW4gZmlyc3QgY3JlYXRlIHBhc3MuJyxcbiAgKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFzc2VydEZpcnN0VXBkYXRlUGFzcyh0VmlldzogVFZpZXcsIGVyck1lc3NhZ2U/OiBzdHJpbmcpIHtcbiAgYXNzZXJ0RXF1YWwoXG4gICAgdFZpZXcuZmlyc3RVcGRhdGVQYXNzLFxuICAgIHRydWUsXG4gICAgZXJyTWVzc2FnZSB8fCAnU2hvdWxkIG9ubHkgYmUgY2FsbGVkIGluIGZpcnN0IHVwZGF0ZSBwYXNzLicsXG4gICk7XG59XG5cbi8qKlxuICogVGhpcyBpcyBhIGJhc2ljIHNhbml0eSBjaGVjayB0aGF0IGFuIG9iamVjdCBpcyBwcm9iYWJseSBhIGRpcmVjdGl2ZSBkZWYuIERpcmVjdGl2ZURlZiBpc1xuICogYW4gaW50ZXJmYWNlLCBzbyB3ZSBjYW4ndCBkbyBhIGRpcmVjdCBpbnN0YW5jZW9mIGNoZWNrLlxuICovXG5leHBvcnQgZnVuY3Rpb24gYXNzZXJ0RGlyZWN0aXZlRGVmPFQ+KG9iajogYW55KTogYXNzZXJ0cyBvYmogaXMgRGlyZWN0aXZlRGVmPFQ+IHtcbiAgaWYgKG9iai50eXBlID09PSB1bmRlZmluZWQgfHwgb2JqLnNlbGVjdG9ycyA9PSB1bmRlZmluZWQgfHwgb2JqLmlucHV0cyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgdGhyb3dFcnJvcihcbiAgICAgIGBFeHBlY3RlZCBhIERpcmVjdGl2ZURlZi9Db21wb25lbnREZWYgYW5kIHRoaXMgb2JqZWN0IGRvZXMgbm90IHNlZW0gdG8gaGF2ZSB0aGUgZXhwZWN0ZWQgc2hhcGUuYCxcbiAgICApO1xuICB9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBhc3NlcnRJbmRleEluRGVjbFJhbmdlKHRWaWV3OiBUVmlldywgaW5kZXg6IG51bWJlcikge1xuICBhc3NlcnRCZXR3ZWVuKEhFQURFUl9PRkZTRVQsIHRWaWV3LmJpbmRpbmdTdGFydEluZGV4LCBpbmRleCk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBhc3NlcnRJbmRleEluRXhwYW5kb1JhbmdlKGxWaWV3OiBMVmlldywgaW5kZXg6IG51bWJlcikge1xuICBjb25zdCB0VmlldyA9IGxWaWV3WzFdO1xuICBhc3NlcnRCZXR3ZWVuKHRWaWV3LmV4cGFuZG9TdGFydEluZGV4LCBsVmlldy5sZW5ndGgsIGluZGV4KTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFzc2VydEJldHdlZW4obG93ZXI6IG51bWJlciwgdXBwZXI6IG51bWJlciwgaW5kZXg6IG51bWJlcikge1xuICBpZiAoIShsb3dlciA8PSBpbmRleCAmJiBpbmRleCA8IHVwcGVyKSkge1xuICAgIHRocm93RXJyb3IoYEluZGV4IG91dCBvZiByYW5nZSAoZXhwZWN0aW5nICR7bG93ZXJ9IDw9ICR7aW5kZXh9IDwgJHt1cHBlcn0pYCk7XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGFzc2VydFByb2plY3Rpb25TbG90cyhsVmlldzogTFZpZXcsIGVyck1lc3NhZ2U/OiBzdHJpbmcpIHtcbiAgYXNzZXJ0RGVmaW5lZChsVmlld1tERUNMQVJBVElPTl9DT01QT05FTlRfVklFV10sICdDb21wb25lbnQgdmlld3Mgc2hvdWxkIGV4aXN0LicpO1xuICBhc3NlcnREZWZpbmVkKFxuICAgIGxWaWV3W0RFQ0xBUkFUSU9OX0NPTVBPTkVOVF9WSUVXXVtUX0hPU1RdIS5wcm9qZWN0aW9uLFxuICAgIGVyck1lc3NhZ2UgfHxcbiAgICAgICdDb21wb25lbnRzIHdpdGggcHJvamVjdGlvbiBub2RlcyAoPG5nLWNvbnRlbnQ+KSBtdXN0IGhhdmUgcHJvamVjdGlvbiBzbG90cyBkZWZpbmVkLicsXG4gICk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBhc3NlcnRQYXJlbnRWaWV3KGxWaWV3OiBMVmlldyB8IG51bGwsIGVyck1lc3NhZ2U/OiBzdHJpbmcpIHtcbiAgYXNzZXJ0RGVmaW5lZChcbiAgICBsVmlldyxcbiAgICBlcnJNZXNzYWdlIHx8IFwiQ29tcG9uZW50IHZpZXdzIHNob3VsZCBhbHdheXMgaGF2ZSBhIHBhcmVudCB2aWV3IChjb21wb25lbnQncyBob3N0IHZpZXcpXCIsXG4gICk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBhc3NlcnROb0R1cGxpY2F0ZURpcmVjdGl2ZXMoZGlyZWN0aXZlczogRGlyZWN0aXZlRGVmPHVua25vd24+W10pOiB2b2lkIHtcbiAgLy8gVGhlIGFycmF5IG5lZWRzIGF0IGxlYXN0IHR3byBlbGVtZW50cyBpbiBvcmRlciB0byBoYXZlIGR1cGxpY2F0ZXMuXG4gIGlmIChkaXJlY3RpdmVzLmxlbmd0aCA8IDIpIHtcbiAgICByZXR1cm47XG4gIH1cblxuICBjb25zdCBzZWVuRGlyZWN0aXZlcyA9IG5ldyBTZXQ8RGlyZWN0aXZlRGVmPHVua25vd24+PigpO1xuXG4gIGZvciAoY29uc3QgY3VycmVudCBvZiBkaXJlY3RpdmVzKSB7XG4gICAgaWYgKHNlZW5EaXJlY3RpdmVzLmhhcyhjdXJyZW50KSkge1xuICAgICAgdGhyb3cgbmV3IFJ1bnRpbWVFcnJvcihcbiAgICAgICAgUnVudGltZUVycm9yQ29kZS5EVVBMSUNBVEVfRElSRUNUSVZFLFxuICAgICAgICBgRGlyZWN0aXZlICR7Y3VycmVudC50eXBlLm5hbWV9IG1hdGNoZXMgbXVsdGlwbGUgdGltZXMgb24gdGhlIHNhbWUgZWxlbWVudC4gYCArXG4gICAgICAgICAgYERpcmVjdGl2ZXMgY2FuIG9ubHkgbWF0Y2ggYW4gZWxlbWVudCBvbmNlLmAsXG4gICAgICApO1xuICAgIH1cbiAgICBzZWVuRGlyZWN0aXZlcy5hZGQoY3VycmVudCk7XG4gIH1cbn1cblxuLyoqXG4gKiBUaGlzIGlzIGEgYmFzaWMgc2FuaXR5IGNoZWNrIHRoYXQgdGhlIGBpbmplY3RvckluZGV4YCBzZWVtcyB0byBwb2ludCB0byB3aGF0IGxvb2tzIGxpa2UgYVxuICogTm9kZUluamVjdG9yIGRhdGEgc3RydWN0dXJlLlxuICpcbiAqIEBwYXJhbSBsVmlldyBgTFZpZXdgIHdoaWNoIHNob3VsZCBiZSBjaGVja2VkLlxuICogQHBhcmFtIGluamVjdG9ySW5kZXggaW5kZXggaW50byB0aGUgYExWaWV3YCB3aGVyZSB0aGUgYE5vZGVJbmplY3RvcmAgaXMgZXhwZWN0ZWQuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBhc3NlcnROb2RlSW5qZWN0b3IobFZpZXc6IExWaWV3LCBpbmplY3RvckluZGV4OiBudW1iZXIpIHtcbiAgYXNzZXJ0SW5kZXhJbkV4cGFuZG9SYW5nZShsVmlldywgaW5qZWN0b3JJbmRleCk7XG4gIGFzc2VydEluZGV4SW5FeHBhbmRvUmFuZ2UobFZpZXcsIGluamVjdG9ySW5kZXggKyBOb2RlSW5qZWN0b3JPZmZzZXQuUEFSRU5UKTtcbiAgYXNzZXJ0TnVtYmVyKGxWaWV3W2luamVjdG9ySW5kZXggKyAwXSwgJ2luamVjdG9ySW5kZXggc2hvdWxkIHBvaW50IHRvIGEgYmxvb20gZmlsdGVyJyk7XG4gIGFzc2VydE51bWJlcihsVmlld1tpbmplY3RvckluZGV4ICsgMV0sICdpbmplY3RvckluZGV4IHNob3VsZCBwb2ludCB0byBhIGJsb29tIGZpbHRlcicpO1xuICBhc3NlcnROdW1iZXIobFZpZXdbaW5qZWN0b3JJbmRleCArIDJdLCAnaW5qZWN0b3JJbmRleCBzaG91bGQgcG9pbnQgdG8gYSBibG9vbSBmaWx0ZXInKTtcbiAgYXNzZXJ0TnVtYmVyKGxWaWV3W2luamVjdG9ySW5kZXggKyAzXSwgJ2luamVjdG9ySW5kZXggc2hvdWxkIHBvaW50IHRvIGEgYmxvb20gZmlsdGVyJyk7XG4gIGFzc2VydE51bWJlcihsVmlld1tpbmplY3RvckluZGV4ICsgNF0sICdpbmplY3RvckluZGV4IHNob3VsZCBwb2ludCB0byBhIGJsb29tIGZpbHRlcicpO1xuICBhc3NlcnROdW1iZXIobFZpZXdbaW5qZWN0b3JJbmRleCArIDVdLCAnaW5qZWN0b3JJbmRleCBzaG91bGQgcG9pbnQgdG8gYSBibG9vbSBmaWx0ZXInKTtcbiAgYXNzZXJ0TnVtYmVyKGxWaWV3W2luamVjdG9ySW5kZXggKyA2XSwgJ2luamVjdG9ySW5kZXggc2hvdWxkIHBvaW50IHRvIGEgYmxvb20gZmlsdGVyJyk7XG4gIGFzc2VydE51bWJlcihsVmlld1tpbmplY3RvckluZGV4ICsgN10sICdpbmplY3RvckluZGV4IHNob3VsZCBwb2ludCB0byBhIGJsb29tIGZpbHRlcicpO1xuICBhc3NlcnROdW1iZXIoXG4gICAgbFZpZXdbaW5qZWN0b3JJbmRleCArIE5vZGVJbmplY3Rvck9mZnNldC5QQVJFTlRdLFxuICAgICdpbmplY3RvckluZGV4IHNob3VsZCBwb2ludCB0byBwYXJlbnQgaW5qZWN0b3InLFxuICApO1xufVxuIl19