/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { resolveForwardRef } from '../../di/forward_ref';
import { InjectionToken } from '../../di/injection_token';
import { throwError } from '../../util/assert';
let _injectorProfilerContext;
export function getInjectorProfilerContext() {
    !ngDevMode && throwError('getInjectorProfilerContext should never be called in production mode');
    return _injectorProfilerContext;
}
export function setInjectorProfilerContext(context) {
    !ngDevMode && throwError('setInjectorProfilerContext should never be called in production mode');
    const previous = _injectorProfilerContext;
    _injectorProfilerContext = context;
    return previous;
}
let injectorProfilerCallback = null;
/**
 * Sets the callback function which will be invoked during certain DI events within the
 * runtime (for example: injecting services, creating injectable instances, configuring providers)
 *
 * Warning: this function is *INTERNAL* and should not be relied upon in application's code.
 * The contract of the function might be changed in any release and/or the function can be removed
 * completely.
 *
 * @param profiler function provided by the caller or null value to disable profiling.
 */
export const setInjectorProfiler = (injectorProfiler) => {
    !ngDevMode && throwError('setInjectorProfiler should never be called in production mode');
    injectorProfilerCallback = injectorProfiler;
};
/**
 * Injector profiler function which emits on DI events executed by the runtime.
 *
 * @param event InjectorProfilerEvent corresponding to the DI event being emitted
 */
function injectorProfiler(event) {
    !ngDevMode && throwError('Injector profiler should never be called in production mode');
    if (injectorProfilerCallback != null /* both `null` and `undefined` */) {
        injectorProfilerCallback(event);
    }
}
/**
 * Emits an InjectorProfilerEventType.ProviderConfigured to the injector profiler. The data in the
 * emitted event includes the raw provider, as well as the token that provider is providing.
 *
 * @param eventProvider A provider object
 */
export function emitProviderConfiguredEvent(eventProvider, isViewProvider = false) {
    !ngDevMode && throwError('Injector profiler should never be called in production mode');
    let token;
    // if the provider is a TypeProvider (typeof provider is function) then the token is the
    // provider itself
    if (typeof eventProvider === 'function') {
        token = eventProvider;
    }
    // if the provider is an injection token, then the token is the injection token.
    else if (eventProvider instanceof InjectionToken) {
        token = eventProvider;
    }
    // in all other cases we can access the token via the `provide` property of the provider
    else {
        token = resolveForwardRef(eventProvider.provide);
    }
    let provider = eventProvider;
    // Injection tokens may define their own default provider which gets attached to the token itself
    // as `ɵprov`. In this case, we want to emit the provider that is attached to the token, not the
    // token itself.
    if (eventProvider instanceof InjectionToken) {
        provider = eventProvider.ɵprov || eventProvider;
    }
    injectorProfiler({
        type: 2 /* InjectorProfilerEventType.ProviderConfigured */,
        context: getInjectorProfilerContext(),
        providerRecord: { token, provider, isViewProvider },
    });
}
/**
 * Emits an event to the injector profiler with the instance that was created. Note that
 * the injector associated with this emission can be accessed by using getDebugInjectContext()
 *
 * @param instance an object created by an injector
 */
export function emitInstanceCreatedByInjectorEvent(instance) {
    !ngDevMode && throwError('Injector profiler should never be called in production mode');
    injectorProfiler({
        type: 1 /* InjectorProfilerEventType.InstanceCreatedByInjector */,
        context: getInjectorProfilerContext(),
        instance: { value: instance },
    });
}
/**
 * @param token DI token associated with injected service
 * @param value the instance of the injected service (i.e the result of `inject(token)`)
 * @param flags the flags that the token was injected with
 */
export function emitInjectEvent(token, value, flags) {
    !ngDevMode && throwError('Injector profiler should never be called in production mode');
    injectorProfiler({
        type: 0 /* InjectorProfilerEventType.Inject */,
        context: getInjectorProfilerContext(),
        service: { token, value, flags },
    });
}
export function runInInjectorProfilerContext(injector, token, callback) {
    !ngDevMode &&
        throwError('runInInjectorProfilerContext should never be called in production mode');
    const prevInjectContext = setInjectorProfilerContext({ injector, token });
    try {
        callback();
    }
    finally {
        setInjectorProfilerContext(prevInjectContext);
    }
}
//# sourceMappingURL=data:application/json;base64,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