/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { formatRuntimeError } from '../errors';
import { ViewEncapsulation } from '../metadata/view';
import { noSideEffects } from '../util/closure';
import { EMPTY_ARRAY, EMPTY_OBJ } from '../util/empty';
import { initNgDevMode } from '../util/ng_dev_mode';
import { stringify } from '../util/stringify';
import { NG_COMP_DEF, NG_DIR_DEF, NG_MOD_DEF, NG_PIPE_DEF } from './fields';
import { InputFlags } from './interfaces/input_flags';
import { stringifyCSSSelectorList } from './node_selector_matcher';
/**
 * Create a component definition object.
 *
 *
 * # Example
 * ```
 * class MyComponent {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵcmp = defineComponent({
 *     ...
 *   });
 * }
 * ```
 * @codeGenApi
 */
export function ɵɵdefineComponent(componentDefinition) {
    return noSideEffects(() => {
        // Initialize ngDevMode. This must be the first statement in ɵɵdefineComponent.
        // See the `initNgDevMode` docstring for more information.
        (typeof ngDevMode === 'undefined' || ngDevMode) && initNgDevMode();
        const baseDef = getNgDirectiveDef(componentDefinition);
        const def = {
            ...baseDef,
            decls: componentDefinition.decls,
            vars: componentDefinition.vars,
            template: componentDefinition.template,
            consts: componentDefinition.consts || null,
            ngContentSelectors: componentDefinition.ngContentSelectors,
            onPush: componentDefinition.changeDetection === ChangeDetectionStrategy.OnPush,
            directiveDefs: null, // assigned in noSideEffects
            pipeDefs: null, // assigned in noSideEffects
            dependencies: (baseDef.standalone && componentDefinition.dependencies) || null,
            getStandaloneInjector: null,
            signals: componentDefinition.signals ?? false,
            data: componentDefinition.data || {},
            encapsulation: componentDefinition.encapsulation || ViewEncapsulation.Emulated,
            styles: componentDefinition.styles || EMPTY_ARRAY,
            _: null,
            schemas: componentDefinition.schemas || null,
            tView: null,
            id: '',
        };
        initFeatures(def);
        const dependencies = componentDefinition.dependencies;
        def.directiveDefs = extractDefListOrFactory(dependencies, /* pipeDef */ false);
        def.pipeDefs = extractDefListOrFactory(dependencies, /* pipeDef */ true);
        def.id = getComponentId(def);
        return def;
    });
}
export function extractDirectiveDef(type) {
    return getComponentDef(type) || getDirectiveDef(type);
}
function nonNull(value) {
    return value !== null;
}
/**
 * @codeGenApi
 */
export function ɵɵdefineNgModule(def) {
    return noSideEffects(() => {
        const res = {
            type: def.type,
            bootstrap: def.bootstrap || EMPTY_ARRAY,
            declarations: def.declarations || EMPTY_ARRAY,
            imports: def.imports || EMPTY_ARRAY,
            exports: def.exports || EMPTY_ARRAY,
            transitiveCompileScopes: null,
            schemas: def.schemas || null,
            id: def.id || null,
        };
        return res;
    });
}
function parseAndConvertBindingsForDefinition(obj, declaredInputs) {
    if (obj == null)
        return EMPTY_OBJ;
    const newLookup = {};
    for (const minifiedKey in obj) {
        if (obj.hasOwnProperty(minifiedKey)) {
            const value = obj[minifiedKey];
            let publicName;
            let declaredName;
            let inputFlags = InputFlags.None;
            if (Array.isArray(value)) {
                inputFlags = value[0];
                publicName = value[1];
                declaredName = value[2] ?? publicName; // declared name might not be set to save bytes.
            }
            else {
                publicName = value;
                declaredName = value;
            }
            // For inputs, capture the declared name, or if some flags are set.
            if (declaredInputs) {
                // Perf note: An array is only allocated for the input if there are flags.
                newLookup[publicName] =
                    inputFlags !== InputFlags.None ? [minifiedKey, inputFlags] : minifiedKey;
                declaredInputs[publicName] = declaredName;
            }
            else {
                newLookup[publicName] = minifiedKey;
            }
        }
    }
    return newLookup;
}
/**
 * Create a directive definition object.
 *
 * # Example
 * ```ts
 * class MyDirective {
 *   // Generated by Angular Template Compiler
 *   // [Symbol] syntax will not be supported by TypeScript until v2.7
 *   static ɵdir = ɵɵdefineDirective({
 *     ...
 *   });
 * }
 * ```
 *
 * @codeGenApi
 */
export function ɵɵdefineDirective(directiveDefinition) {
    return noSideEffects(() => {
        const def = getNgDirectiveDef(directiveDefinition);
        initFeatures(def);
        return def;
    });
}
/**
 * Create a pipe definition object.
 *
 * # Example
 * ```
 * class MyPipe implements PipeTransform {
 *   // Generated by Angular Template Compiler
 *   static ɵpipe = definePipe({
 *     ...
 *   });
 * }
 * ```
 * @param pipeDef Pipe definition generated by the compiler
 *
 * @codeGenApi
 */
export function ɵɵdefinePipe(pipeDef) {
    return {
        type: pipeDef.type,
        name: pipeDef.name,
        factory: null,
        pure: pipeDef.pure !== false,
        standalone: pipeDef.standalone === true,
        onDestroy: pipeDef.type.prototype.ngOnDestroy || null,
    };
}
/**
 * The following getter methods retrieve the definition from the type. Currently the retrieval
 * honors inheritance, but in the future we may change the rule to require that definitions are
 * explicit. This would require some sort of migration strategy.
 */
export function getComponentDef(type) {
    return type[NG_COMP_DEF] || null;
}
export function getDirectiveDef(type) {
    return type[NG_DIR_DEF] || null;
}
export function getPipeDef(type) {
    return type[NG_PIPE_DEF] || null;
}
/**
 * Checks whether a given Component, Directive or Pipe is marked as standalone.
 * This will return false if passed anything other than a Component, Directive, or Pipe class
 * See [this guide](guide/components/importing) for additional information:
 *
 * @param type A reference to a Component, Directive or Pipe.
 * @publicApi
 */
export function isStandalone(type) {
    const def = getComponentDef(type) || getDirectiveDef(type) || getPipeDef(type);
    return def !== null ? def.standalone : false;
}
export function getNgModuleDef(type, throwNotFound) {
    const ngModuleDef = type[NG_MOD_DEF] || null;
    if (!ngModuleDef && throwNotFound === true) {
        throw new Error(`Type ${stringify(type)} does not have 'ɵmod' property.`);
    }
    return ngModuleDef;
}
function getNgDirectiveDef(directiveDefinition) {
    const declaredInputs = {};
    return {
        type: directiveDefinition.type,
        providersResolver: null,
        factory: null,
        hostBindings: directiveDefinition.hostBindings || null,
        hostVars: directiveDefinition.hostVars || 0,
        hostAttrs: directiveDefinition.hostAttrs || null,
        contentQueries: directiveDefinition.contentQueries || null,
        declaredInputs: declaredInputs,
        inputTransforms: null,
        inputConfig: directiveDefinition.inputs || EMPTY_OBJ,
        exportAs: directiveDefinition.exportAs || null,
        standalone: directiveDefinition.standalone === true,
        signals: directiveDefinition.signals === true,
        selectors: directiveDefinition.selectors || EMPTY_ARRAY,
        viewQuery: directiveDefinition.viewQuery || null,
        features: directiveDefinition.features || null,
        setInput: null,
        findHostDirectiveDefs: null,
        hostDirectives: null,
        inputs: parseAndConvertBindingsForDefinition(directiveDefinition.inputs, declaredInputs),
        outputs: parseAndConvertBindingsForDefinition(directiveDefinition.outputs),
        debugInfo: null,
    };
}
function initFeatures(definition) {
    definition.features?.forEach((fn) => fn(definition));
}
export function extractDefListOrFactory(dependencies, pipeDef) {
    if (!dependencies) {
        return null;
    }
    const defExtractor = pipeDef ? getPipeDef : extractDirectiveDef;
    return () => (typeof dependencies === 'function' ? dependencies() : dependencies)
        .map((dep) => defExtractor(dep))
        .filter(nonNull);
}
/**
 * A map that contains the generated component IDs and type.
 */
export const GENERATED_COMP_IDS = new Map();
/**
 * A method can returns a component ID from the component definition using a variant of DJB2 hash
 * algorithm.
 */
function getComponentId(componentDef) {
    let hash = 0;
    // We cannot rely solely on the component selector as the same selector can be used in different
    // modules.
    //
    // `componentDef.style` is not used, due to it causing inconsistencies. Ex: when server
    // component styles has no sourcemaps and browsers do.
    //
    // Example:
    // https://github.com/angular/components/blob/d9f82c8f95309e77a6d82fd574c65871e91354c2/src/material/core/option/option.ts#L248
    // https://github.com/angular/components/blob/285f46dc2b4c5b127d356cb7c4714b221f03ce50/src/material/legacy-core/option/option.ts#L32
    const hashSelectors = [
        componentDef.selectors,
        componentDef.ngContentSelectors,
        componentDef.hostVars,
        componentDef.hostAttrs,
        componentDef.consts,
        componentDef.vars,
        componentDef.decls,
        componentDef.encapsulation,
        componentDef.standalone,
        componentDef.signals,
        componentDef.exportAs,
        JSON.stringify(componentDef.inputs),
        JSON.stringify(componentDef.outputs),
        // We cannot use 'componentDef.type.name' as the name of the symbol will change and will not
        // match in the server and browser bundles.
        Object.getOwnPropertyNames(componentDef.type.prototype),
        !!componentDef.contentQueries,
        !!componentDef.viewQuery,
    ].join('|');
    for (const char of hashSelectors) {
        hash = (Math.imul(31, hash) + char.charCodeAt(0)) << 0;
    }
    // Force positive number hash.
    // 2147483647 = equivalent of Integer.MAX_VALUE.
    hash += 2147483647 + 1;
    const compId = 'c' + hash;
    if (typeof ngDevMode === 'undefined' || ngDevMode) {
        if (GENERATED_COMP_IDS.has(compId)) {
            const previousCompDefType = GENERATED_COMP_IDS.get(compId);
            if (previousCompDefType !== componentDef.type) {
                console.warn(formatRuntimeError(-912 /* RuntimeErrorCode.COMPONENT_ID_COLLISION */, `Component ID generation collision detected. Components '${previousCompDefType.name}' and '${componentDef.type.name}' with selector '${stringifyCSSSelectorList(componentDef.selectors)}' generated the same component ID. To fix this, you can change the selector of one of those components or add an extra host attribute to force a different ID.`));
            }
        }
        else {
            GENERATED_COMP_IDS.set(compId, componentDef.type);
        }
    }
    return compId;
}
//# sourceMappingURL=data:application/json;base64,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