/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { isEnvironmentProviders } from '../di/interface/provider';
import { RuntimeError } from '../errors';
import { stringify } from '../util/stringify';
import { stringifyForError } from './util/stringify_utils';
/** Called when directives inject each other (creating a circular dependency) */
export function throwCyclicDependencyError(token, path) {
    const depPath = path ? `. Dependency path: ${path.join(' > ')} > ${token}` : '';
    throw new RuntimeError(-200 /* RuntimeErrorCode.CYCLIC_DI_DEPENDENCY */, ngDevMode ? `Circular dependency in DI detected for ${token}${depPath}` : token);
}
export function throwMixedMultiProviderError() {
    throw new Error(`Cannot mix multi providers and regular providers`);
}
export function throwInvalidProviderError(ngModuleType, providers, provider) {
    if (ngModuleType && providers) {
        const providerDetail = providers.map((v) => (v == provider ? '?' + provider + '?' : '...'));
        throw new Error(`Invalid provider for the NgModule '${stringify(ngModuleType)}' - only instances of Provider and Type are allowed, got: [${providerDetail.join(', ')}]`);
    }
    else if (isEnvironmentProviders(provider)) {
        if (provider.ɵfromNgModule) {
            throw new RuntimeError(207 /* RuntimeErrorCode.PROVIDER_IN_WRONG_CONTEXT */, `Invalid providers from 'importProvidersFrom' present in a non-environment injector. 'importProvidersFrom' can't be used for component providers.`);
        }
        else {
            throw new RuntimeError(207 /* RuntimeErrorCode.PROVIDER_IN_WRONG_CONTEXT */, `Invalid providers present in a non-environment injector. 'EnvironmentProviders' can't be used for component providers.`);
        }
    }
    else {
        throw new Error('Invalid provider');
    }
}
/** Throws an error when a token is not found in DI. */
export function throwProviderNotFoundError(token, injectorName) {
    const errorMessage = ngDevMode &&
        `No provider for ${stringifyForError(token)} found${injectorName ? ` in ${injectorName}` : ''}`;
    throw new RuntimeError(-201 /* RuntimeErrorCode.PROVIDER_NOT_FOUND */, errorMessage);
}
//# sourceMappingURL=data:application/json;base64,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