/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { assertEqual, assertGreaterThan, assertGreaterThanOrEqual, throwError, } from '../../util/assert';
import { assertTIcu, assertTNode } from '../assert';
import { createTNodeAtIndex } from '../instructions/shared';
import { assertTNodeType } from '../node_assert';
import { setI18nHandling } from '../node_manipulation';
import { getInsertInFrontOfRNodeWithI18n, processI18nInsertBefore } from '../node_manipulation_i18n';
import { addTNodeAndUpdateInsertBeforeIndex } from './i18n_insert_before_index';
/**
 * Retrieve `TIcu` at a given `index`.
 *
 * The `TIcu` can be stored either directly (if it is nested ICU) OR
 * it is stored inside tho `TIcuContainer` if it is top level ICU.
 *
 * The reason for this is that the top level ICU need a `TNode` so that they are part of the render
 * tree, but nested ICU's have no TNode, because we don't know ahead of time if the nested ICU is
 * expressed (parent ICU may have selected a case which does not contain it.)
 *
 * @param tView Current `TView`.
 * @param index Index where the value should be read from.
 */
export function getTIcu(tView, index) {
    const value = tView.data[index];
    if (value === null || typeof value === 'string')
        return null;
    if (ngDevMode &&
        !(value.hasOwnProperty('tView') || value.hasOwnProperty('currentCaseLViewIndex'))) {
        throwError("We expect to get 'null'|'TIcu'|'TIcuContainer', but got: " + value);
    }
    // Here the `value.hasOwnProperty('currentCaseLViewIndex')` is a polymorphic read as it can be
    // either TIcu or TIcuContainerNode. This is not ideal, but we still think it is OK because it
    // will be just two cases which fits into the browser inline cache (inline cache can take up to
    // 4)
    const tIcu = value.hasOwnProperty('currentCaseLViewIndex')
        ? value
        : value.value;
    ngDevMode && assertTIcu(tIcu);
    return tIcu;
}
/**
 * Store `TIcu` at a give `index`.
 *
 * The `TIcu` can be stored either directly (if it is nested ICU) OR
 * it is stored inside tho `TIcuContainer` if it is top level ICU.
 *
 * The reason for this is that the top level ICU need a `TNode` so that they are part of the render
 * tree, but nested ICU's have no TNode, because we don't know ahead of time if the nested ICU is
 * expressed (parent ICU may have selected a case which does not contain it.)
 *
 * @param tView Current `TView`.
 * @param index Index where the value should be stored at in `Tview.data`
 * @param tIcu The TIcu to store.
 */
export function setTIcu(tView, index, tIcu) {
    const tNode = tView.data[index];
    ngDevMode &&
        assertEqual(tNode === null || tNode.hasOwnProperty('tView'), true, "We expect to get 'null'|'TIcuContainer'");
    if (tNode === null) {
        tView.data[index] = tIcu;
    }
    else {
        ngDevMode && assertTNodeType(tNode, 32 /* TNodeType.Icu */);
        tNode.value = tIcu;
    }
}
/**
 * Set `TNode.insertBeforeIndex` taking the `Array` into account.
 *
 * See `TNode.insertBeforeIndex`
 */
export function setTNodeInsertBeforeIndex(tNode, index) {
    ngDevMode && assertTNode(tNode);
    let insertBeforeIndex = tNode.insertBeforeIndex;
    if (insertBeforeIndex === null) {
        setI18nHandling(getInsertInFrontOfRNodeWithI18n, processI18nInsertBefore);
        insertBeforeIndex = tNode.insertBeforeIndex = [
            null /* may be updated to number later */,
            index,
        ];
    }
    else {
        assertEqual(Array.isArray(insertBeforeIndex), true, 'Expecting array here');
        insertBeforeIndex.push(index);
    }
}
/**
 * Create `TNode.type=TNodeType.Placeholder` node.
 *
 * See `TNodeType.Placeholder` for more information.
 */
export function createTNodePlaceholder(tView, previousTNodes, index) {
    const tNode = createTNodeAtIndex(tView, index, 64 /* TNodeType.Placeholder */, null, null);
    addTNodeAndUpdateInsertBeforeIndex(previousTNodes, tNode);
    return tNode;
}
/**
 * Returns current ICU case.
 *
 * ICU cases are stored as index into the `TIcu.cases`.
 * At times it is necessary to communicate that the ICU case just switched and that next ICU update
 * should update all bindings regardless of the mask. In such a case the we store negative numbers
 * for cases which have just been switched. This function removes the negative flag.
 */
export function getCurrentICUCaseIndex(tIcu, lView) {
    const currentCase = lView[tIcu.currentCaseLViewIndex];
    return currentCase === null ? currentCase : currentCase < 0 ? ~currentCase : currentCase;
}
export function getParentFromIcuCreateOpCode(mergedCode) {
    return mergedCode >>> 17 /* IcuCreateOpCode.SHIFT_PARENT */;
}
export function getRefFromIcuCreateOpCode(mergedCode) {
    return (mergedCode & 131070 /* IcuCreateOpCode.MASK_REF */) >>> 1 /* IcuCreateOpCode.SHIFT_REF */;
}
export function getInstructionFromIcuCreateOpCode(mergedCode) {
    return mergedCode & 1 /* IcuCreateOpCode.MASK_INSTRUCTION */;
}
export function icuCreateOpCode(opCode, parentIdx, refIdx) {
    ngDevMode && assertGreaterThanOrEqual(parentIdx, 0, 'Missing parent index');
    ngDevMode && assertGreaterThan(refIdx, 0, 'Missing ref index');
    return (opCode | (parentIdx << 17 /* IcuCreateOpCode.SHIFT_PARENT */) | (refIdx << 1 /* IcuCreateOpCode.SHIFT_REF */));
}
// Returns whether the given value corresponds to a root template message,
// or a sub-template.
export function isRootTemplateMessage(subTemplateIndex) {
    return subTemplateIndex === -1;
}
//# sourceMappingURL=data:application/json;base64,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