/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ɵRuntimeError as RuntimeError } from '@angular/core';
import { formArrayNameExample, formControlNameExample, formGroupNameExample, ngModelGroupExample, } from './error_examples';
export function controlParentException(nameOrIndex) {
    return new RuntimeError(1050 /* RuntimeErrorCode.FORM_CONTROL_NAME_MISSING_PARENT */, `formControlName must be used with a parent formGroup directive. You'll want to add a formGroup
      directive and pass it an existing FormGroup instance (you can create one in your class).

      ${describeFormControl(nameOrIndex)}

    Example:

    ${formControlNameExample}`);
}
function describeFormControl(nameOrIndex) {
    if (nameOrIndex == null || nameOrIndex === '') {
        return '';
    }
    const valueType = typeof nameOrIndex === 'string' ? 'name' : 'index';
    return `Affected Form Control ${valueType}: "${nameOrIndex}"`;
}
export function ngModelGroupException() {
    return new RuntimeError(1051 /* RuntimeErrorCode.FORM_CONTROL_NAME_INSIDE_MODEL_GROUP */, `formControlName cannot be used with an ngModelGroup parent. It is only compatible with parents
      that also have a "form" prefix: formGroupName, formArrayName, or formGroup.

      Option 1:  Update the parent to be formGroupName (reactive form strategy)

      ${formGroupNameExample}

      Option 2: Use ngModel instead of formControlName (template-driven strategy)

      ${ngModelGroupExample}`);
}
export function missingFormException() {
    return new RuntimeError(1052 /* RuntimeErrorCode.FORM_GROUP_MISSING_INSTANCE */, `formGroup expects a FormGroup instance. Please pass one in.

      Example:

      ${formControlNameExample}`);
}
export function groupParentException() {
    return new RuntimeError(1053 /* RuntimeErrorCode.FORM_GROUP_NAME_MISSING_PARENT */, `formGroupName must be used with a parent formGroup directive.  You'll want to add a formGroup
    directive and pass it an existing FormGroup instance (you can create one in your class).

    Example:

    ${formGroupNameExample}`);
}
export function arrayParentException() {
    return new RuntimeError(1054 /* RuntimeErrorCode.FORM_ARRAY_NAME_MISSING_PARENT */, `formArrayName must be used with a parent formGroup directive.  You'll want to add a formGroup
      directive and pass it an existing FormGroup instance (you can create one in your class).

      Example:

      ${formArrayNameExample}`);
}
export const disabledAttrWarning = `
  It looks like you're using the disabled attribute with a reactive form directive. If you set disabled to true
  when you set up this control in your component class, the disabled attribute will actually be set in the DOM for
  you. We recommend using this approach to avoid 'changed after checked' errors.

  Example:
  // Specify the \`disabled\` property at control creation time:
  form = new FormGroup({
    first: new FormControl({value: 'Nancy', disabled: true}, Validators.required),
    last: new FormControl('Drew', Validators.required)
  });

  // Controls can also be enabled/disabled after creation:
  form.get('first')?.enable();
  form.get('last')?.disable();
`;
export const asyncValidatorsDroppedWithOptsWarning = `
  It looks like you're constructing using a FormControl with both an options argument and an
  async validators argument. Mixing these arguments will cause your async validators to be dropped.
  You should either put all your validators in the options object, or in separate validators
  arguments. For example:

  // Using validators arguments
  fc = new FormControl(42, Validators.required, myAsyncValidator);

  // Using AbstractControlOptions
  fc = new FormControl(42, {validators: Validators.required, asyncValidators: myAV});

  // Do NOT mix them: async validators will be dropped!
  fc = new FormControl(42, {validators: Validators.required}, /* Oops! */ myAsyncValidator);
`;
export function ngModelWarning(directiveName) {
    return `
  It looks like you're using ngModel on the same form field as ${directiveName}.
  Support for using the ngModel input property and ngModelChange event with
  reactive form directives has been deprecated in Angular v6 and will be removed
  in a future version of Angular.

  For more information on this, see our API docs here:
  https://angular.io/api/forms/${directiveName === 'formControl' ? 'FormControlDirective' : 'FormControlName'}#use-with-ngmodel
  `;
}
function describeKey(isFormGroup, key) {
    return isFormGroup ? `with name: '${key}'` : `at index: ${key}`;
}
export function noControlsError(isFormGroup) {
    return `
    There are no form controls registered with this ${isFormGroup ? 'group' : 'array'} yet. If you're using ngModel,
    you may want to check next tick (e.g. use setTimeout).
  `;
}
export function missingControlError(isFormGroup, key) {
    return `Cannot find form control ${describeKey(isFormGroup, key)}`;
}
export function missingControlValueError(isFormGroup, key) {
    return `Must supply a value for form control ${describeKey(isFormGroup, key)}`;
}
//# sourceMappingURL=data:application/json;base64,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