/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { FocusMonitor, FocusTrapFactory, InteractivityChecker } from '@angular/cdk/a11y';
import { OverlayRef } from '@angular/cdk/overlay';
import { DOCUMENT } from '@angular/common';
import { ChangeDetectionStrategy, Component, ElementRef, EventEmitter, Inject, NgZone, Optional, ViewEncapsulation, ANIMATION_MODULE_TYPE, } from '@angular/core';
import { MatDialogConfig } from './dialog-config';
import { CdkDialogContainer } from '@angular/cdk/dialog';
import { coerceNumberProperty } from '@angular/cdk/coercion';
import { CdkPortalOutlet } from '@angular/cdk/portal';
import * as i0 from "@angular/core";
import * as i1 from "@angular/cdk/a11y";
import * as i2 from "./dialog-config";
import * as i3 from "@angular/cdk/overlay";
/** Class added when the dialog is open. */
const OPEN_CLASS = 'mdc-dialog--open';
/** Class added while the dialog is opening. */
const OPENING_CLASS = 'mdc-dialog--opening';
/** Class added while the dialog is closing. */
const CLOSING_CLASS = 'mdc-dialog--closing';
/** Duration of the opening animation in milliseconds. */
export const OPEN_ANIMATION_DURATION = 150;
/** Duration of the closing animation in milliseconds. */
export const CLOSE_ANIMATION_DURATION = 75;
export class MatDialogContainer extends CdkDialogContainer {
    constructor(elementRef, focusTrapFactory, _document, dialogConfig, interactivityChecker, ngZone, overlayRef, _animationMode, focusMonitor) {
        super(elementRef, focusTrapFactory, _document, dialogConfig, interactivityChecker, ngZone, overlayRef, focusMonitor);
        this._animationMode = _animationMode;
        /** Emits when an animation state changes. */
        this._animationStateChanged = new EventEmitter();
        /** Whether animations are enabled. */
        this._animationsEnabled = this._animationMode !== 'NoopAnimations';
        /** Number of actions projected in the dialog. */
        this._actionSectionCount = 0;
        /** Host element of the dialog container component. */
        this._hostElement = this._elementRef.nativeElement;
        /** Duration of the dialog open animation. */
        this._enterAnimationDuration = this._animationsEnabled
            ? parseCssTime(this._config.enterAnimationDuration) ?? OPEN_ANIMATION_DURATION
            : 0;
        /** Duration of the dialog close animation. */
        this._exitAnimationDuration = this._animationsEnabled
            ? parseCssTime(this._config.exitAnimationDuration) ?? CLOSE_ANIMATION_DURATION
            : 0;
        /** Current timer for dialog animations. */
        this._animationTimer = null;
        /**
         * Completes the dialog open by clearing potential animation classes, trapping
         * focus and emitting an opened event.
         */
        this._finishDialogOpen = () => {
            this._clearAnimationClasses();
            this._openAnimationDone(this._enterAnimationDuration);
        };
        /**
         * Completes the dialog close by clearing potential animation classes, restoring
         * focus and emitting a closed event.
         */
        this._finishDialogClose = () => {
            this._clearAnimationClasses();
            this._animationStateChanged.emit({ state: 'closed', totalTime: this._exitAnimationDuration });
        };
    }
    _contentAttached() {
        // Delegate to the original dialog-container initialization (i.e. saving the
        // previous element, setting up the focus trap and moving focus to the container).
        super._contentAttached();
        // Note: Usually we would be able to use the MDC dialog foundation here to handle
        // the dialog animation for us, but there are a few reasons why we just leverage
        // their styles and not use the runtime foundation code:
        //   1. Foundation does not allow us to disable animations.
        //   2. Foundation contains unnecessary features we don't need and aren't
        //      tree-shakeable. e.g. background scrim, keyboard event handlers for ESC button.
        this._startOpenAnimation();
    }
    /** Starts the dialog open animation if enabled. */
    _startOpenAnimation() {
        this._animationStateChanged.emit({ state: 'opening', totalTime: this._enterAnimationDuration });
        if (this._animationsEnabled) {
            this._hostElement.style.setProperty(TRANSITION_DURATION_PROPERTY, `${this._enterAnimationDuration}ms`);
            // We need to give the `setProperty` call from above some time to be applied.
            // One would expect that the open class is added once the animation finished, but MDC
            // uses the open class in combination with the opening class to start the animation.
            this._requestAnimationFrame(() => this._hostElement.classList.add(OPENING_CLASS, OPEN_CLASS));
            this._waitForAnimationToComplete(this._enterAnimationDuration, this._finishDialogOpen);
        }
        else {
            this._hostElement.classList.add(OPEN_CLASS);
            // Note: We could immediately finish the dialog opening here with noop animations,
            // but we defer until next tick so that consumers can subscribe to `afterOpened`.
            // Executing this immediately would mean that `afterOpened` emits synchronously
            // on `dialog.open` before the consumer had a change to subscribe to `afterOpened`.
            Promise.resolve().then(() => this._finishDialogOpen());
        }
    }
    /**
     * Starts the exit animation of the dialog if enabled. This method is
     * called by the dialog ref.
     */
    _startExitAnimation() {
        this._animationStateChanged.emit({ state: 'closing', totalTime: this._exitAnimationDuration });
        this._hostElement.classList.remove(OPEN_CLASS);
        if (this._animationsEnabled) {
            this._hostElement.style.setProperty(TRANSITION_DURATION_PROPERTY, `${this._exitAnimationDuration}ms`);
            // We need to give the `setProperty` call from above some time to be applied.
            this._requestAnimationFrame(() => this._hostElement.classList.add(CLOSING_CLASS));
            this._waitForAnimationToComplete(this._exitAnimationDuration, this._finishDialogClose);
        }
        else {
            // This subscription to the `OverlayRef#backdropClick` observable in the `DialogRef` is
            // set up before any user can subscribe to the backdrop click. The subscription triggers
            // the dialog close and this method synchronously. If we'd synchronously emit the `CLOSED`
            // animation state event if animations are disabled, the overlay would be disposed
            // immediately and all other subscriptions to `DialogRef#backdropClick` would be silently
            // skipped. We work around this by waiting with the dialog close until the next tick when
            // all subscriptions have been fired as expected. This is not an ideal solution, but
            // there doesn't seem to be any other good way. Alternatives that have been considered:
            //   1. Deferring `DialogRef.close`. This could be a breaking change due to a new microtask.
            //      Also this issue is specific to the MDC implementation where the dialog could
            //      technically be closed synchronously. In the non-MDC one, Angular animations are used
            //      and closing always takes at least a tick.
            //   2. Ensuring that user subscriptions to `backdropClick`, `keydownEvents` in the dialog
            //      ref are first. This would solve the issue, but has the risk of memory leaks and also
            //      doesn't solve the case where consumers call `DialogRef.close` in their subscriptions.
            // Based on the fact that this is specific to the MDC-based implementation of the dialog
            // animations, the defer is applied here.
            Promise.resolve().then(() => this._finishDialogClose());
        }
    }
    /**
     * Updates the number action sections.
     * @param delta Increase/decrease in the number of sections.
     */
    _updateActionSectionCount(delta) {
        this._actionSectionCount += delta;
        this._changeDetectorRef.markForCheck();
    }
    /** Clears all dialog animation classes. */
    _clearAnimationClasses() {
        this._hostElement.classList.remove(OPENING_CLASS, CLOSING_CLASS);
    }
    _waitForAnimationToComplete(duration, callback) {
        if (this._animationTimer !== null) {
            clearTimeout(this._animationTimer);
        }
        // Note that we want this timer to run inside the NgZone, because we want
        // the related events like `afterClosed` to be inside the zone as well.
        this._animationTimer = setTimeout(callback, duration);
    }
    /** Runs a callback in `requestAnimationFrame`, if available. */
    _requestAnimationFrame(callback) {
        this._ngZone.runOutsideAngular(() => {
            if (typeof requestAnimationFrame === 'function') {
                requestAnimationFrame(callback);
            }
            else {
                callback();
            }
        });
    }
    _captureInitialFocus() {
        if (!this._config.delayFocusTrap) {
            this._trapFocus();
        }
    }
    /**
     * Callback for when the open dialog animation has finished. Intended to
     * be called by sub-classes that use different animation implementations.
     */
    _openAnimationDone(totalTime) {
        if (this._config.delayFocusTrap) {
            this._trapFocus();
        }
        this._animationStateChanged.next({ state: 'opened', totalTime });
    }
    ngOnDestroy() {
        super.ngOnDestroy();
        if (this._animationTimer !== null) {
            clearTimeout(this._animationTimer);
        }
    }
    attachComponentPortal(portal) {
        // When a component is passed into the dialog, the host element interrupts
        // the `display:flex` from affecting the dialog title, content, and
        // actions. To fix this, we make the component host `display: contents` by
        // marking its host with the `mat-mdc-dialog-component-host` class.
        //
        // Note that this problem does not exist when a template ref is used since
        // the title, contents, and actions are then nested directly under the
        // dialog surface.
        const ref = super.attachComponentPortal(portal);
        ref.location.nativeElement.classList.add('mat-mdc-dialog-component-host');
        return ref;
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.1.0", ngImport: i0, type: MatDialogContainer, deps: [{ token: i0.ElementRef }, { token: i1.FocusTrapFactory }, { token: DOCUMENT, optional: true }, { token: i2.MatDialogConfig }, { token: i1.InteractivityChecker }, { token: i0.NgZone }, { token: i3.OverlayRef }, { token: ANIMATION_MODULE_TYPE, optional: true }, { token: i1.FocusMonitor }], target: i0.ɵɵFactoryTarget.Component }); }
    static { this.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "14.0.0", version: "18.1.0", type: MatDialogContainer, isStandalone: true, selector: "mat-dialog-container", host: { attributes: { "tabindex": "-1" }, properties: { "attr.aria-modal": "_config.ariaModal", "id": "_config.id", "attr.role": "_config.role", "attr.aria-labelledby": "_config.ariaLabel ? null : _ariaLabelledByQueue[0]", "attr.aria-label": "_config.ariaLabel", "attr.aria-describedby": "_config.ariaDescribedBy || null", "class._mat-animation-noopable": "!_animationsEnabled", "class.mat-mdc-dialog-container-with-actions": "_actionSectionCount > 0" }, classAttribute: "mat-mdc-dialog-container mdc-dialog" }, usesInheritance: true, ngImport: i0, template: "<div class=\"mat-mdc-dialog-inner-container mdc-dialog__container\">\n  <div class=\"mat-mdc-dialog-surface mdc-dialog__surface\">\n    <ng-template cdkPortalOutlet />\n  </div>\n</div>\n", styles: [".mat-mdc-dialog-container{width:100%;height:100%;display:block;box-sizing:border-box;max-height:inherit;min-height:inherit;min-width:inherit;max-width:inherit;outline:0}.cdk-overlay-pane.mat-mdc-dialog-panel{max-width:var(--mat-dialog-container-max-width, 80vw);min-width:var(--mat-dialog-container-min-width, 0)}@media(max-width: 599px){.cdk-overlay-pane.mat-mdc-dialog-panel{max-width:var(--mat-dialog-container-small-max-width, 80vw)}}.mat-mdc-dialog-inner-container{display:flex;flex-direction:row;align-items:center;justify-content:space-around;box-sizing:border-box;height:100%;opacity:0;transition:opacity linear var(--mat-dialog-transition-duration, 0ms);max-height:inherit;min-height:inherit;min-width:inherit;max-width:inherit}.mdc-dialog--closing .mat-mdc-dialog-inner-container{transition:opacity 75ms linear;transform:none}.mdc-dialog--open .mat-mdc-dialog-inner-container{opacity:1}._mat-animation-noopable .mat-mdc-dialog-inner-container{transition:none}.mat-mdc-dialog-surface{display:flex;flex-direction:column;flex-grow:0;flex-shrink:0;box-sizing:border-box;width:100%;height:100%;position:relative;overflow-y:auto;outline:0;transform:scale(0.8);transition:transform var(--mat-dialog-transition-duration, 0ms) cubic-bezier(0, 0, 0.2, 1);max-height:inherit;min-height:inherit;min-width:inherit;max-width:inherit;box-shadow:var(--mat-dialog-container-elevation-shadow, 0px 11px 15px -7px rgba(0, 0, 0, 0.2), 0px 24px 38px 3px rgba(0, 0, 0, 0.14), 0px 9px 46px 8px rgba(0, 0, 0, 0.12));border-radius:var(--mdc-dialog-container-shape, 4px);background-color:var(--mdc-dialog-container-color, white)}[dir=rtl] .mat-mdc-dialog-surface{text-align:right}.mdc-dialog--open .mat-mdc-dialog-surface,.mdc-dialog--closing .mat-mdc-dialog-surface{transform:none}._mat-animation-noopable .mat-mdc-dialog-surface{transition:none}.mat-mdc-dialog-surface::before{position:absolute;box-sizing:border-box;width:100%;height:100%;top:0;left:0;border:2px solid rgba(0,0,0,0);border-radius:inherit;content:\"\";pointer-events:none}.mat-mdc-dialog-title{display:block;position:relative;flex-shrink:0;box-sizing:border-box;margin:0 0 1px;padding:var(--mat-dialog-headline-padding, 0 24px 9px)}.mat-mdc-dialog-title::before{display:inline-block;width:0;height:40px;content:\"\";vertical-align:0}[dir=rtl] .mat-mdc-dialog-title{text-align:right}.mat-mdc-dialog-container .mat-mdc-dialog-title{color:var(--mdc-dialog-subhead-color, rgba(0, 0, 0, 0.87));font-family:var(--mdc-dialog-subhead-font, Roboto, sans-serif);line-height:var(--mdc-dialog-subhead-line-height, 1.5rem);font-size:var(--mdc-dialog-subhead-size, 1rem);font-weight:var(--mdc-dialog-subhead-weight, 400);letter-spacing:var(--mdc-dialog-subhead-tracking, 0.03125em)}.mat-mdc-dialog-content{display:block;flex-grow:1;box-sizing:border-box;margin:0;overflow:auto;max-height:65vh}.mat-mdc-dialog-content>:first-child{margin-top:0}.mat-mdc-dialog-content>:last-child{margin-bottom:0}.mat-mdc-dialog-container .mat-mdc-dialog-content{color:var(--mdc-dialog-supporting-text-color, rgba(0, 0, 0, 0.6));font-family:var(--mdc-dialog-supporting-text-font, Roboto, sans-serif);line-height:var(--mdc-dialog-supporting-text-line-height, 1.5rem);font-size:var(--mdc-dialog-supporting-text-size, 1rem);font-weight:var(--mdc-dialog-supporting-text-weight, 400);letter-spacing:var(--mdc-dialog-supporting-text-tracking, 0.03125em)}.mat-mdc-dialog-container .mat-mdc-dialog-content{padding:var(--mat-dialog-content-padding, 20px 24px)}.mat-mdc-dialog-container-with-actions .mat-mdc-dialog-content{padding:var(--mat-dialog-with-actions-content-padding, 20px 24px)}.mat-mdc-dialog-container .mat-mdc-dialog-title+.mat-mdc-dialog-content{padding-top:0}.mat-mdc-dialog-actions{display:flex;position:relative;flex-shrink:0;flex-wrap:wrap;align-items:center;justify-content:flex-end;box-sizing:border-box;min-height:52px;margin:0;padding:8px;border-top:1px solid rgba(0,0,0,0);padding:var(--mat-dialog-actions-padding, 8px);justify-content:var(--mat-dialog-actions-alignment, start)}.cdk-high-contrast-active .mat-mdc-dialog-actions{border-top-color:CanvasText}.mat-mdc-dialog-actions.mat-mdc-dialog-actions-align-start,.mat-mdc-dialog-actions[align=start]{justify-content:start}.mat-mdc-dialog-actions.mat-mdc-dialog-actions-align-center,.mat-mdc-dialog-actions[align=center]{justify-content:center}.mat-mdc-dialog-actions.mat-mdc-dialog-actions-align-end,.mat-mdc-dialog-actions[align=end]{justify-content:flex-end}.mat-mdc-dialog-actions .mat-button-base+.mat-button-base,.mat-mdc-dialog-actions .mat-mdc-button-base+.mat-mdc-button-base{margin-left:8px}[dir=rtl] .mat-mdc-dialog-actions .mat-button-base+.mat-button-base,[dir=rtl] .mat-mdc-dialog-actions .mat-mdc-button-base+.mat-mdc-button-base{margin-left:0;margin-right:8px}.mat-mdc-dialog-component-host{display:contents}"], dependencies: [{ kind: "directive", type: CdkPortalOutlet, selector: "[cdkPortalOutlet]", inputs: ["cdkPortalOutlet"], outputs: ["attached"], exportAs: ["cdkPortalOutlet"] }], changeDetection: i0.ChangeDetectionStrategy.Default, encapsulation: i0.ViewEncapsulation.None }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.1.0", ngImport: i0, type: MatDialogContainer, decorators: [{
            type: Component,
            args: [{ selector: 'mat-dialog-container', encapsulation: ViewEncapsulation.None, changeDetection: ChangeDetectionStrategy.Default, standalone: true, imports: [CdkPortalOutlet], host: {
                        'class': 'mat-mdc-dialog-container mdc-dialog',
                        'tabindex': '-1',
                        '[attr.aria-modal]': '_config.ariaModal',
                        '[id]': '_config.id',
                        '[attr.role]': '_config.role',
                        '[attr.aria-labelledby]': '_config.ariaLabel ? null : _ariaLabelledByQueue[0]',
                        '[attr.aria-label]': '_config.ariaLabel',
                        '[attr.aria-describedby]': '_config.ariaDescribedBy || null',
                        '[class._mat-animation-noopable]': '!_animationsEnabled',
                        '[class.mat-mdc-dialog-container-with-actions]': '_actionSectionCount > 0',
                    }, template: "<div class=\"mat-mdc-dialog-inner-container mdc-dialog__container\">\n  <div class=\"mat-mdc-dialog-surface mdc-dialog__surface\">\n    <ng-template cdkPortalOutlet />\n  </div>\n</div>\n", styles: [".mat-mdc-dialog-container{width:100%;height:100%;display:block;box-sizing:border-box;max-height:inherit;min-height:inherit;min-width:inherit;max-width:inherit;outline:0}.cdk-overlay-pane.mat-mdc-dialog-panel{max-width:var(--mat-dialog-container-max-width, 80vw);min-width:var(--mat-dialog-container-min-width, 0)}@media(max-width: 599px){.cdk-overlay-pane.mat-mdc-dialog-panel{max-width:var(--mat-dialog-container-small-max-width, 80vw)}}.mat-mdc-dialog-inner-container{display:flex;flex-direction:row;align-items:center;justify-content:space-around;box-sizing:border-box;height:100%;opacity:0;transition:opacity linear var(--mat-dialog-transition-duration, 0ms);max-height:inherit;min-height:inherit;min-width:inherit;max-width:inherit}.mdc-dialog--closing .mat-mdc-dialog-inner-container{transition:opacity 75ms linear;transform:none}.mdc-dialog--open .mat-mdc-dialog-inner-container{opacity:1}._mat-animation-noopable .mat-mdc-dialog-inner-container{transition:none}.mat-mdc-dialog-surface{display:flex;flex-direction:column;flex-grow:0;flex-shrink:0;box-sizing:border-box;width:100%;height:100%;position:relative;overflow-y:auto;outline:0;transform:scale(0.8);transition:transform var(--mat-dialog-transition-duration, 0ms) cubic-bezier(0, 0, 0.2, 1);max-height:inherit;min-height:inherit;min-width:inherit;max-width:inherit;box-shadow:var(--mat-dialog-container-elevation-shadow, 0px 11px 15px -7px rgba(0, 0, 0, 0.2), 0px 24px 38px 3px rgba(0, 0, 0, 0.14), 0px 9px 46px 8px rgba(0, 0, 0, 0.12));border-radius:var(--mdc-dialog-container-shape, 4px);background-color:var(--mdc-dialog-container-color, white)}[dir=rtl] .mat-mdc-dialog-surface{text-align:right}.mdc-dialog--open .mat-mdc-dialog-surface,.mdc-dialog--closing .mat-mdc-dialog-surface{transform:none}._mat-animation-noopable .mat-mdc-dialog-surface{transition:none}.mat-mdc-dialog-surface::before{position:absolute;box-sizing:border-box;width:100%;height:100%;top:0;left:0;border:2px solid rgba(0,0,0,0);border-radius:inherit;content:\"\";pointer-events:none}.mat-mdc-dialog-title{display:block;position:relative;flex-shrink:0;box-sizing:border-box;margin:0 0 1px;padding:var(--mat-dialog-headline-padding, 0 24px 9px)}.mat-mdc-dialog-title::before{display:inline-block;width:0;height:40px;content:\"\";vertical-align:0}[dir=rtl] .mat-mdc-dialog-title{text-align:right}.mat-mdc-dialog-container .mat-mdc-dialog-title{color:var(--mdc-dialog-subhead-color, rgba(0, 0, 0, 0.87));font-family:var(--mdc-dialog-subhead-font, Roboto, sans-serif);line-height:var(--mdc-dialog-subhead-line-height, 1.5rem);font-size:var(--mdc-dialog-subhead-size, 1rem);font-weight:var(--mdc-dialog-subhead-weight, 400);letter-spacing:var(--mdc-dialog-subhead-tracking, 0.03125em)}.mat-mdc-dialog-content{display:block;flex-grow:1;box-sizing:border-box;margin:0;overflow:auto;max-height:65vh}.mat-mdc-dialog-content>:first-child{margin-top:0}.mat-mdc-dialog-content>:last-child{margin-bottom:0}.mat-mdc-dialog-container .mat-mdc-dialog-content{color:var(--mdc-dialog-supporting-text-color, rgba(0, 0, 0, 0.6));font-family:var(--mdc-dialog-supporting-text-font, Roboto, sans-serif);line-height:var(--mdc-dialog-supporting-text-line-height, 1.5rem);font-size:var(--mdc-dialog-supporting-text-size, 1rem);font-weight:var(--mdc-dialog-supporting-text-weight, 400);letter-spacing:var(--mdc-dialog-supporting-text-tracking, 0.03125em)}.mat-mdc-dialog-container .mat-mdc-dialog-content{padding:var(--mat-dialog-content-padding, 20px 24px)}.mat-mdc-dialog-container-with-actions .mat-mdc-dialog-content{padding:var(--mat-dialog-with-actions-content-padding, 20px 24px)}.mat-mdc-dialog-container .mat-mdc-dialog-title+.mat-mdc-dialog-content{padding-top:0}.mat-mdc-dialog-actions{display:flex;position:relative;flex-shrink:0;flex-wrap:wrap;align-items:center;justify-content:flex-end;box-sizing:border-box;min-height:52px;margin:0;padding:8px;border-top:1px solid rgba(0,0,0,0);padding:var(--mat-dialog-actions-padding, 8px);justify-content:var(--mat-dialog-actions-alignment, start)}.cdk-high-contrast-active .mat-mdc-dialog-actions{border-top-color:CanvasText}.mat-mdc-dialog-actions.mat-mdc-dialog-actions-align-start,.mat-mdc-dialog-actions[align=start]{justify-content:start}.mat-mdc-dialog-actions.mat-mdc-dialog-actions-align-center,.mat-mdc-dialog-actions[align=center]{justify-content:center}.mat-mdc-dialog-actions.mat-mdc-dialog-actions-align-end,.mat-mdc-dialog-actions[align=end]{justify-content:flex-end}.mat-mdc-dialog-actions .mat-button-base+.mat-button-base,.mat-mdc-dialog-actions .mat-mdc-button-base+.mat-mdc-button-base{margin-left:8px}[dir=rtl] .mat-mdc-dialog-actions .mat-button-base+.mat-button-base,[dir=rtl] .mat-mdc-dialog-actions .mat-mdc-button-base+.mat-mdc-button-base{margin-left:0;margin-right:8px}.mat-mdc-dialog-component-host{display:contents}"] }]
        }], ctorParameters: () => [{ type: i0.ElementRef }, { type: i1.FocusTrapFactory }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [DOCUMENT]
                }] }, { type: i2.MatDialogConfig }, { type: i1.InteractivityChecker }, { type: i0.NgZone }, { type: i3.OverlayRef }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [ANIMATION_MODULE_TYPE]
                }] }, { type: i1.FocusMonitor }] });
const TRANSITION_DURATION_PROPERTY = '--mat-dialog-transition-duration';
// TODO(mmalerba): Remove this function after animation durations are required
//  to be numbers.
/**
 * Converts a CSS time string to a number in ms. If the given time is already a
 * number, it is assumed to be in ms.
 */
function parseCssTime(time) {
    if (time == null) {
        return null;
    }
    if (typeof time === 'number') {
        return time;
    }
    if (time.endsWith('ms')) {
        return coerceNumberProperty(time.substring(0, time.length - 2));
    }
    if (time.endsWith('s')) {
        return coerceNumberProperty(time.substring(0, time.length - 1)) * 1000;
    }
    if (time === '0') {
        return 0;
    }
    return null; // anything else is invalid.
}
//# sourceMappingURL=data:application/json;base64,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