/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Directionality } from '@angular/cdk/bidi';
import { coerceBooleanProperty } from '@angular/cdk/coercion';
import { Platform } from '@angular/cdk/platform';
import { DOCUMENT, NgTemplateOutlet } from '@angular/common';
import { ANIMATION_MODULE_TYPE, ChangeDetectionStrategy, ChangeDetectorRef, Component, ContentChild, ContentChildren, ElementRef, Inject, InjectionToken, Injector, Input, NgZone, Optional, QueryList, ViewChild, ViewEncapsulation, afterRender, computed, contentChild, inject, } from '@angular/core';
import { Subject, merge } from 'rxjs';
import { takeUntil } from 'rxjs/operators';
import { MAT_ERROR } from './directives/error';
import { FLOATING_LABEL_PARENT, MatFormFieldFloatingLabel, } from './directives/floating-label';
import { MatHint } from './directives/hint';
import { MatLabel } from './directives/label';
import { MatFormFieldLineRipple } from './directives/line-ripple';
import { MatFormFieldNotchedOutline } from './directives/notched-outline';
import { MAT_PREFIX } from './directives/prefix';
import { MAT_SUFFIX } from './directives/suffix';
import { matFormFieldAnimations } from './form-field-animations';
import { MatFormFieldControl as _MatFormFieldControl } from './form-field-control';
import { getMatFormFieldDuplicatedHintError, getMatFormFieldMissingControlError, } from './form-field-errors';
import * as i0 from "@angular/core";
import * as i1 from "@angular/cdk/bidi";
import * as i2 from "@angular/cdk/platform";
/**
 * Injection token that can be used to inject an instances of `MatFormField`. It serves
 * as alternative token to the actual `MatFormField` class which would cause unnecessary
 * retention of the `MatFormField` class and its component metadata.
 */
export const MAT_FORM_FIELD = new InjectionToken('MatFormField');
/**
 * Injection token that can be used to configure the
 * default options for all form field within an app.
 */
export const MAT_FORM_FIELD_DEFAULT_OPTIONS = new InjectionToken('MAT_FORM_FIELD_DEFAULT_OPTIONS');
let nextUniqueId = 0;
/** Default appearance used by the form field. */
const DEFAULT_APPEARANCE = 'fill';
/**
 * Whether the label for form fields should by default float `always`,
 * `never`, or `auto`.
 */
const DEFAULT_FLOAT_LABEL = 'auto';
/** Default way that the subscript element height is set. */
const DEFAULT_SUBSCRIPT_SIZING = 'fixed';
/**
 * Default transform for docked floating labels in a MDC text-field. This value has been
 * extracted from the MDC text-field styles because we programmatically modify the docked
 * label transform, but do not want to accidentally discard the default label transform.
 */
const FLOATING_LABEL_DEFAULT_DOCKED_TRANSFORM = `translateY(-50%)`;
/** Container for form controls that applies Material Design styling and behavior. */
export class MatFormField {
    /** Whether the required marker should be hidden. */
    get hideRequiredMarker() {
        return this._hideRequiredMarker;
    }
    set hideRequiredMarker(value) {
        this._hideRequiredMarker = coerceBooleanProperty(value);
    }
    /** Whether the label should always float or float as the user types. */
    get floatLabel() {
        return this._floatLabel || this._defaults?.floatLabel || DEFAULT_FLOAT_LABEL;
    }
    set floatLabel(value) {
        if (value !== this._floatLabel) {
            this._floatLabel = value;
            // For backwards compatibility. Custom form field controls or directives might set
            // the "floatLabel" input and expect the form field view to be updated automatically.
            // e.g. autocomplete trigger. Ideally we'd get rid of this and the consumers would just
            // emit the "stateChanges" observable. TODO(devversion): consider removing.
            this._changeDetectorRef.markForCheck();
        }
    }
    /** The form field appearance style. */
    get appearance() {
        return this._appearance;
    }
    set appearance(value) {
        const oldValue = this._appearance;
        const newAppearance = value || this._defaults?.appearance || DEFAULT_APPEARANCE;
        if (typeof ngDevMode === 'undefined' || ngDevMode) {
            if (newAppearance !== 'fill' && newAppearance !== 'outline') {
                throw new Error(`MatFormField: Invalid appearance "${newAppearance}", valid values are "fill" or "outline".`);
            }
        }
        this._appearance = newAppearance;
        if (this._appearance === 'outline' && this._appearance !== oldValue) {
            // If the appearance has been switched to `outline`, the label offset needs to be updated.
            // The update can happen once the view has been re-checked, but not immediately because
            // the view has not been updated and the notched-outline floating label is not present.
            this._needsOutlineLabelOffsetUpdate = true;
        }
    }
    /**
     * Whether the form field should reserve space for one line of hint/error text (default)
     * or to have the spacing grow from 0px as needed based on the size of the hint/error content.
     * Note that when using dynamic sizing, layout shifts will occur when hint/error text changes.
     */
    get subscriptSizing() {
        return this._subscriptSizing || this._defaults?.subscriptSizing || DEFAULT_SUBSCRIPT_SIZING;
    }
    set subscriptSizing(value) {
        this._subscriptSizing = value || this._defaults?.subscriptSizing || DEFAULT_SUBSCRIPT_SIZING;
    }
    /** Text for the form field hint. */
    get hintLabel() {
        return this._hintLabel;
    }
    set hintLabel(value) {
        this._hintLabel = value;
        this._processHints();
    }
    /** Gets the current form field control */
    get _control() {
        return this._explicitFormFieldControl || this._formFieldControl;
    }
    set _control(value) {
        this._explicitFormFieldControl = value;
    }
    constructor(_elementRef, _changeDetectorRef, 
    /**
     * @deprecated not needed, to be removed.
     * @breaking-change 19.0.0 remove this param
     */
    _unusedNgZone, _dir, _platform, _defaults, _animationMode, 
    /**
     * @deprecated not needed, to be removed.
     * @breaking-change 17.0.0 remove this param
     */
    _unusedDocument) {
        this._elementRef = _elementRef;
        this._changeDetectorRef = _changeDetectorRef;
        this._dir = _dir;
        this._platform = _platform;
        this._defaults = _defaults;
        this._animationMode = _animationMode;
        this._labelChild = contentChild(MatLabel);
        this._hideRequiredMarker = false;
        /**
         * Theme color of the form field. This API is supported in M2 themes only, it
         * has no effect in M3 themes.
         *
         * For information on applying color variants in M3, see
         * https://material.angular.io/guide/theming#using-component-color-variants.
         */
        this.color = 'primary';
        this._appearance = DEFAULT_APPEARANCE;
        this._subscriptSizing = null;
        this._hintLabel = '';
        this._hasIconPrefix = false;
        this._hasTextPrefix = false;
        this._hasIconSuffix = false;
        this._hasTextSuffix = false;
        // Unique id for the internal form field label.
        this._labelId = `mat-mdc-form-field-label-${nextUniqueId++}`;
        // Unique id for the hint label.
        this._hintLabelId = `mat-mdc-hint-${nextUniqueId++}`;
        /** State of the mat-hint and mat-error animations. */
        this._subscriptAnimationState = '';
        this._destroyed = new Subject();
        this._isFocused = null;
        this._needsOutlineLabelOffsetUpdate = false;
        this._injector = inject(Injector);
        /**
         * Gets the id of the label element. If no label is present, returns `null`.
         */
        this.getLabelId = computed(() => (this._hasFloatingLabel() ? this._labelId : null));
        this._hasFloatingLabel = computed(() => !!this._labelChild());
        if (_defaults) {
            if (_defaults.appearance) {
                this.appearance = _defaults.appearance;
            }
            this._hideRequiredMarker = Boolean(_defaults?.hideRequiredMarker);
            if (_defaults.color) {
                this.color = _defaults.color;
            }
        }
    }
    ngAfterViewInit() {
        // Initial focus state sync. This happens rarely, but we want to account for
        // it in case the form field control has "focused" set to true on init.
        this._updateFocusState();
        // Enable animations now. This ensures we don't animate on initial render.
        this._subscriptAnimationState = 'enter';
        // Because the above changes a value used in the template after it was checked, we need
        // to trigger CD or the change might not be reflected if there is no other CD scheduled.
        this._changeDetectorRef.detectChanges();
    }
    ngAfterContentInit() {
        this._assertFormFieldControl();
        this._initializeControl();
        this._initializeSubscript();
        this._initializePrefixAndSuffix();
        this._initializeOutlineLabelOffsetSubscriptions();
    }
    ngAfterContentChecked() {
        this._assertFormFieldControl();
    }
    ngOnDestroy() {
        this._destroyed.next();
        this._destroyed.complete();
    }
    /**
     * Gets an ElementRef for the element that a overlay attached to the form field
     * should be positioned relative to.
     */
    getConnectedOverlayOrigin() {
        return this._textField || this._elementRef;
    }
    /** Animates the placeholder up and locks it in position. */
    _animateAndLockLabel() {
        // This is for backwards compatibility only. Consumers of the form field might use
        // this method. e.g. the autocomplete trigger. This method has been added to the non-MDC
        // form field because setting "floatLabel" to "always" caused the label to float without
        // animation. This is different in MDC where the label always animates, so this method
        // is no longer necessary. There doesn't seem any benefit in adding logic to allow changing
        // the floating label state without animations. The non-MDC implementation was inconsistent
        // because it always animates if "floatLabel" is set away from "always".
        // TODO(devversion): consider removing this method when releasing the MDC form field.
        if (this._hasFloatingLabel()) {
            this.floatLabel = 'always';
        }
    }
    /** Initializes the registered form field control. */
    _initializeControl() {
        const control = this._control;
        if (control.controlType) {
            this._elementRef.nativeElement.classList.add(`mat-mdc-form-field-type-${control.controlType}`);
        }
        // Subscribe to changes in the child control state in order to update the form field UI.
        control.stateChanges.subscribe(() => {
            this._updateFocusState();
            this._syncDescribedByIds();
            this._changeDetectorRef.markForCheck();
        });
        // Run change detection if the value changes.
        if (control.ngControl && control.ngControl.valueChanges) {
            control.ngControl.valueChanges
                .pipe(takeUntil(this._destroyed))
                .subscribe(() => this._changeDetectorRef.markForCheck());
        }
    }
    _checkPrefixAndSuffixTypes() {
        this._hasIconPrefix = !!this._prefixChildren.find(p => !p._isText);
        this._hasTextPrefix = !!this._prefixChildren.find(p => p._isText);
        this._hasIconSuffix = !!this._suffixChildren.find(s => !s._isText);
        this._hasTextSuffix = !!this._suffixChildren.find(s => s._isText);
    }
    /** Initializes the prefix and suffix containers. */
    _initializePrefixAndSuffix() {
        this._checkPrefixAndSuffixTypes();
        // Mark the form field as dirty whenever the prefix or suffix children change. This
        // is necessary because we conditionally display the prefix/suffix containers based
        // on whether there is projected content.
        merge(this._prefixChildren.changes, this._suffixChildren.changes).subscribe(() => {
            this._checkPrefixAndSuffixTypes();
            this._changeDetectorRef.markForCheck();
        });
    }
    /**
     * Initializes the subscript by validating hints and synchronizing "aria-describedby" ids
     * with the custom form field control. Also subscribes to hint and error changes in order
     * to be able to validate and synchronize ids on change.
     */
    _initializeSubscript() {
        // Re-validate when the number of hints changes.
        this._hintChildren.changes.subscribe(() => {
            this._processHints();
            this._changeDetectorRef.markForCheck();
        });
        // Update the aria-described by when the number of errors changes.
        this._errorChildren.changes.subscribe(() => {
            this._syncDescribedByIds();
            this._changeDetectorRef.markForCheck();
        });
        // Initial mat-hint validation and subscript describedByIds sync.
        this._validateHints();
        this._syncDescribedByIds();
    }
    /** Throws an error if the form field's control is missing. */
    _assertFormFieldControl() {
        if (!this._control && (typeof ngDevMode === 'undefined' || ngDevMode)) {
            throw getMatFormFieldMissingControlError();
        }
    }
    _updateFocusState() {
        // Usually the MDC foundation would call "activateFocus" and "deactivateFocus" whenever
        // certain DOM events are emitted. This is not possible in our implementation of the
        // form field because we support abstract form field controls which are not necessarily
        // of type input, nor do we have a reference to a native form field control element. Instead
        // we handle the focus by checking if the abstract form field control focused state changes.
        if (this._control.focused && !this._isFocused) {
            this._isFocused = true;
            this._lineRipple?.activate();
        }
        else if (!this._control.focused && (this._isFocused || this._isFocused === null)) {
            this._isFocused = false;
            this._lineRipple?.deactivate();
        }
        this._textField?.nativeElement.classList.toggle('mdc-text-field--focused', this._control.focused);
    }
    /**
     * The floating label in the docked state needs to account for prefixes. The horizontal offset
     * is calculated whenever the appearance changes to `outline`, the prefixes change, or when the
     * form field is added to the DOM. This method sets up all subscriptions which are needed to
     * trigger the label offset update.
     */
    _initializeOutlineLabelOffsetSubscriptions() {
        // Whenever the prefix changes, schedule an update of the label offset.
        // TODO(mmalerba): Use ResizeObserver to better support dynamically changing prefix content.
        this._prefixChildren.changes.subscribe(() => (this._needsOutlineLabelOffsetUpdate = true));
        // TODO(mmalerba): Split this into separate `afterRender` calls using the `EarlyRead` and
        //  `Write` phases.
        afterRender(() => {
            if (this._needsOutlineLabelOffsetUpdate) {
                this._needsOutlineLabelOffsetUpdate = false;
                this._updateOutlineLabelOffset();
            }
        }, {
            injector: this._injector,
        });
        this._dir.change
            .pipe(takeUntil(this._destroyed))
            .subscribe(() => (this._needsOutlineLabelOffsetUpdate = true));
    }
    /** Whether the floating label should always float or not. */
    _shouldAlwaysFloat() {
        return this.floatLabel === 'always';
    }
    _hasOutline() {
        return this.appearance === 'outline';
    }
    /**
     * Whether the label should display in the infix. Labels in the outline appearance are
     * displayed as part of the notched-outline and are horizontally offset to account for
     * form field prefix content. This won't work in server side rendering since we cannot
     * measure the width of the prefix container. To make the docked label appear as if the
     * right offset has been calculated, we forcibly render the label inside the infix. Since
     * the label is part of the infix, the label cannot overflow the prefix content.
     */
    _forceDisplayInfixLabel() {
        return !this._platform.isBrowser && this._prefixChildren.length && !this._shouldLabelFloat();
    }
    _shouldLabelFloat() {
        if (!this._hasFloatingLabel()) {
            return false;
        }
        return this._control.shouldLabelFloat || this._shouldAlwaysFloat();
    }
    /**
     * Determines whether a class from the AbstractControlDirective
     * should be forwarded to the host element.
     */
    _shouldForward(prop) {
        const control = this._control ? this._control.ngControl : null;
        return control && control[prop];
    }
    /** Determines whether to display hints or errors. */
    _getDisplayedMessages() {
        return this._errorChildren && this._errorChildren.length > 0 && this._control.errorState
            ? 'error'
            : 'hint';
    }
    /** Handle label resize events. */
    _handleLabelResized() {
        this._refreshOutlineNotchWidth();
    }
    /** Refreshes the width of the outline-notch, if present. */
    _refreshOutlineNotchWidth() {
        if (!this._hasOutline() || !this._floatingLabel || !this._shouldLabelFloat()) {
            this._notchedOutline?._setNotchWidth(0);
        }
        else {
            this._notchedOutline?._setNotchWidth(this._floatingLabel.getWidth());
        }
    }
    /** Does any extra processing that is required when handling the hints. */
    _processHints() {
        this._validateHints();
        this._syncDescribedByIds();
    }
    /**
     * Ensure that there is a maximum of one of each "mat-hint" alignment specified. The hint
     * label specified set through the input is being considered as "start" aligned.
     *
     * This method is a noop if Angular runs in production mode.
     */
    _validateHints() {
        if (this._hintChildren && (typeof ngDevMode === 'undefined' || ngDevMode)) {
            let startHint;
            let endHint;
            this._hintChildren.forEach((hint) => {
                if (hint.align === 'start') {
                    if (startHint || this.hintLabel) {
                        throw getMatFormFieldDuplicatedHintError('start');
                    }
                    startHint = hint;
                }
                else if (hint.align === 'end') {
                    if (endHint) {
                        throw getMatFormFieldDuplicatedHintError('end');
                    }
                    endHint = hint;
                }
            });
        }
    }
    /**
     * Sets the list of element IDs that describe the child control. This allows the control to update
     * its `aria-describedby` attribute accordingly.
     */
    _syncDescribedByIds() {
        if (this._control) {
            let ids = [];
            // TODO(wagnermaciel): Remove the type check when we find the root cause of this bug.
            if (this._control.userAriaDescribedBy &&
                typeof this._control.userAriaDescribedBy === 'string') {
                ids.push(...this._control.userAriaDescribedBy.split(' '));
            }
            if (this._getDisplayedMessages() === 'hint') {
                const startHint = this._hintChildren
                    ? this._hintChildren.find(hint => hint.align === 'start')
                    : null;
                const endHint = this._hintChildren
                    ? this._hintChildren.find(hint => hint.align === 'end')
                    : null;
                if (startHint) {
                    ids.push(startHint.id);
                }
                else if (this._hintLabel) {
                    ids.push(this._hintLabelId);
                }
                if (endHint) {
                    ids.push(endHint.id);
                }
            }
            else if (this._errorChildren) {
                ids.push(...this._errorChildren.map(error => error.id));
            }
            this._control.setDescribedByIds(ids);
        }
    }
    /**
     * Updates the horizontal offset of the label in the outline appearance. In the outline
     * appearance, the notched-outline and label are not relative to the infix container because
     * the outline intends to surround prefixes, suffixes and the infix. This means that the
     * floating label by default overlaps prefixes in the docked state. To avoid this, we need to
     * horizontally offset the label by the width of the prefix container. The MDC text-field does
     * not need to do this because they use a fixed width for prefixes. Hence, they can simply
     * incorporate the horizontal offset into their default text-field styles.
     */
    _updateOutlineLabelOffset() {
        if (!this._hasOutline() || !this._floatingLabel) {
            return;
        }
        const floatingLabel = this._floatingLabel.element;
        // If no prefix is displayed, reset the outline label offset from potential
        // previous label offset updates.
        if (!(this._iconPrefixContainer || this._textPrefixContainer)) {
            floatingLabel.style.transform = '';
            return;
        }
        // If the form field is not attached to the DOM yet (e.g. in a tab), we defer
        // the label offset update until the zone stabilizes.
        if (!this._isAttachedToDom()) {
            this._needsOutlineLabelOffsetUpdate = true;
            return;
        }
        const iconPrefixContainer = this._iconPrefixContainer?.nativeElement;
        const textPrefixContainer = this._textPrefixContainer?.nativeElement;
        const iconPrefixContainerWidth = iconPrefixContainer?.getBoundingClientRect().width ?? 0;
        const textPrefixContainerWidth = textPrefixContainer?.getBoundingClientRect().width ?? 0;
        // If the directionality is RTL, the x-axis transform needs to be inverted. This
        // is because `transformX` does not change based on the page directionality.
        const negate = this._dir.value === 'rtl' ? '-1' : '1';
        const prefixWidth = `${iconPrefixContainerWidth + textPrefixContainerWidth}px`;
        const labelOffset = `var(--mat-mdc-form-field-label-offset-x, 0px)`;
        const labelHorizontalOffset = `calc(${negate} * (${prefixWidth} + ${labelOffset}))`;
        // Update the translateX of the floating label to account for the prefix container,
        // but allow the CSS to override this setting via a CSS variable when the label is
        // floating.
        floatingLabel.style.transform = `var(
        --mat-mdc-form-field-label-transform,
        ${FLOATING_LABEL_DEFAULT_DOCKED_TRANSFORM} translateX(${labelHorizontalOffset})
    )`;
    }
    /** Checks whether the form field is attached to the DOM. */
    _isAttachedToDom() {
        const element = this._elementRef.nativeElement;
        if (element.getRootNode) {
            const rootNode = element.getRootNode();
            // If the element is inside the DOM the root node will be either the document
            // or the closest shadow root, otherwise it'll be the element itself.
            return rootNode && rootNode !== element;
        }
        // Otherwise fall back to checking if it's in the document. This doesn't account for
        // shadow DOM, however browser that support shadow DOM should support `getRootNode` as well.
        return document.documentElement.contains(element);
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.1.0", ngImport: i0, type: MatFormField, deps: [{ token: i0.ElementRef }, { token: i0.ChangeDetectorRef }, { token: i0.NgZone }, { token: i1.Directionality }, { token: i2.Platform }, { token: MAT_FORM_FIELD_DEFAULT_OPTIONS, optional: true }, { token: ANIMATION_MODULE_TYPE, optional: true }, { token: DOCUMENT }], target: i0.ɵɵFactoryTarget.Component }); }
    static { this.ɵcmp = i0.ɵɵngDeclareComponent({ minVersion: "17.0.0", version: "18.1.0", type: MatFormField, isStandalone: true, selector: "mat-form-field", inputs: { hideRequiredMarker: "hideRequiredMarker", color: "color", floatLabel: "floatLabel", appearance: "appearance", subscriptSizing: "subscriptSizing", hintLabel: "hintLabel" }, host: { properties: { "class.mat-mdc-form-field-label-always-float": "_shouldAlwaysFloat()", "class.mat-mdc-form-field-has-icon-prefix": "_hasIconPrefix", "class.mat-mdc-form-field-has-icon-suffix": "_hasIconSuffix", "class.mat-form-field-invalid": "_control.errorState", "class.mat-form-field-disabled": "_control.disabled", "class.mat-form-field-autofilled": "_control.autofilled", "class.mat-form-field-no-animations": "_animationMode === \"NoopAnimations\"", "class.mat-form-field-appearance-fill": "appearance == \"fill\"", "class.mat-form-field-appearance-outline": "appearance == \"outline\"", "class.mat-form-field-hide-placeholder": "_hasFloatingLabel() && !_shouldLabelFloat()", "class.mat-focused": "_control.focused", "class.mat-primary": "color !== \"accent\" && color !== \"warn\"", "class.mat-accent": "color === \"accent\"", "class.mat-warn": "color === \"warn\"", "class.ng-untouched": "_shouldForward(\"untouched\")", "class.ng-touched": "_shouldForward(\"touched\")", "class.ng-pristine": "_shouldForward(\"pristine\")", "class.ng-dirty": "_shouldForward(\"dirty\")", "class.ng-valid": "_shouldForward(\"valid\")", "class.ng-invalid": "_shouldForward(\"invalid\")", "class.ng-pending": "_shouldForward(\"pending\")" }, classAttribute: "mat-mdc-form-field" }, providers: [
            { provide: MAT_FORM_FIELD, useExisting: MatFormField },
            { provide: FLOATING_LABEL_PARENT, useExisting: MatFormField },
        ], queries: [{ propertyName: "_labelChild", first: true, predicate: MatLabel, descendants: true, isSignal: true }, { propertyName: "_formFieldControl", first: true, predicate: _MatFormFieldControl, descendants: true }, { propertyName: "_prefixChildren", predicate: MAT_PREFIX, descendants: true }, { propertyName: "_suffixChildren", predicate: MAT_SUFFIX, descendants: true }, { propertyName: "_errorChildren", predicate: MAT_ERROR, descendants: true }, { propertyName: "_hintChildren", predicate: MatHint, descendants: true }], viewQueries: [{ propertyName: "_textField", first: true, predicate: ["textField"], descendants: true }, { propertyName: "_iconPrefixContainer", first: true, predicate: ["iconPrefixContainer"], descendants: true }, { propertyName: "_textPrefixContainer", first: true, predicate: ["textPrefixContainer"], descendants: true }, { propertyName: "_floatingLabel", first: true, predicate: MatFormFieldFloatingLabel, descendants: true }, { propertyName: "_notchedOutline", first: true, predicate: MatFormFieldNotchedOutline, descendants: true }, { propertyName: "_lineRipple", first: true, predicate: MatFormFieldLineRipple, descendants: true }], exportAs: ["matFormField"], ngImport: i0, template: "<ng-template #labelTemplate>\n  <!--\n    MDC recommends that the text-field is a `<label>` element. This rather complicates the\n    setup because it would require every form-field control to explicitly set `aria-labelledby`.\n    This is because the `<label>` itself contains more than the actual label (e.g. prefix, suffix\n    or other projected content), and screen readers could potentially read out undesired content.\n    Excluding elements from being printed out requires them to be marked with `aria-hidden`, or\n    the form control is set to a scoped element for the label (using `aria-labelledby`). Both of\n    these options seem to complicate the setup because we know exactly what content is rendered\n    as part of the label, and we don't want to spend resources on walking through projected content\n    to set `aria-hidden`. Nor do we want to set `aria-labelledby` on every form control if we could\n    simply link the label to the control using the label `for` attribute.\n  -->\n  @if (_hasFloatingLabel()) {\n    <label matFormFieldFloatingLabel\n           [floating]=\"_shouldLabelFloat()\"\n           [monitorResize]=\"_hasOutline()\"\n           [id]=\"_labelId\"\n           [attr.for]=\"_control.disableAutomaticLabeling ? null : _control.id\">\n      <ng-content select=\"mat-label\"></ng-content>\n      <!--\n        We set the required marker as a separate element, in order to make it easier to target if\n        apps want to override it and to be able to set `aria-hidden` so that screen readers don't\n        pick it up.\n       -->\n       @if (!hideRequiredMarker && _control.required) {\n         <span\n           aria-hidden=\"true\"\n           class=\"mat-mdc-form-field-required-marker mdc-floating-label--required\"></span>\n       }\n    </label>\n  }\n</ng-template>\n\n<div class=\"mat-mdc-text-field-wrapper mdc-text-field\" #textField\n     [class.mdc-text-field--filled]=\"!_hasOutline()\"\n     [class.mdc-text-field--outlined]=\"_hasOutline()\"\n     [class.mdc-text-field--no-label]=\"!_hasFloatingLabel()\"\n     [class.mdc-text-field--disabled]=\"_control.disabled\"\n     [class.mdc-text-field--invalid]=\"_control.errorState\"\n     (click)=\"_control.onContainerClick($event)\">\n  @if (!_hasOutline() && !_control.disabled) {\n    <div class=\"mat-mdc-form-field-focus-overlay\"></div>\n  }\n  <div class=\"mat-mdc-form-field-flex\">\n    @if (_hasOutline()) {\n      <div matFormFieldNotchedOutline [matFormFieldNotchedOutlineOpen]=\"_shouldLabelFloat()\">\n        @if (!_forceDisplayInfixLabel()) {\n          <ng-template [ngTemplateOutlet]=\"labelTemplate\"></ng-template>\n        }\n      </div>\n    }\n\n    @if (_hasIconPrefix) {\n      <div class=\"mat-mdc-form-field-icon-prefix\" #iconPrefixContainer>\n        <ng-content select=\"[matPrefix], [matIconPrefix]\"></ng-content>\n      </div>\n    }\n\n    @if (_hasTextPrefix) {\n      <div class=\"mat-mdc-form-field-text-prefix\" #textPrefixContainer>\n        <ng-content select=\"[matTextPrefix]\"></ng-content>\n      </div>\n    }\n\n    <div class=\"mat-mdc-form-field-infix\">\n      @if (!_hasOutline() || _forceDisplayInfixLabel()) {\n        <ng-template [ngTemplateOutlet]=\"labelTemplate\"></ng-template>\n      }\n\n      <ng-content></ng-content>\n    </div>\n\n    @if (_hasTextSuffix) {\n      <div class=\"mat-mdc-form-field-text-suffix\">\n        <ng-content select=\"[matTextSuffix]\"></ng-content>\n      </div>\n    }\n\n    @if (_hasIconSuffix) {\n      <div class=\"mat-mdc-form-field-icon-suffix\">\n        <ng-content select=\"[matSuffix], [matIconSuffix]\"></ng-content>\n      </div>\n    }\n  </div>\n\n  @if (!_hasOutline()) {\n    <div matFormFieldLineRipple></div>\n  }\n</div>\n\n<div class=\"mat-mdc-form-field-subscript-wrapper mat-mdc-form-field-bottom-align\"\n     [class.mat-mdc-form-field-subscript-dynamic-size]=\"subscriptSizing === 'dynamic'\">\n\n  @switch (_getDisplayedMessages()) {\n    @case ('error') {\n      <div class=\"mat-mdc-form-field-error-wrapper\"\n           [@transitionMessages]=\"_subscriptAnimationState\">\n        <ng-content select=\"mat-error, [matError]\"></ng-content>\n      </div>\n    }\n\n    @case ('hint') {\n      <div class=\"mat-mdc-form-field-hint-wrapper\" [@transitionMessages]=\"_subscriptAnimationState\">\n        @if (hintLabel) {\n          <mat-hint [id]=\"_hintLabelId\">{{hintLabel}}</mat-hint>\n        }\n        <ng-content select=\"mat-hint:not([align='end'])\"></ng-content>\n        <div class=\"mat-mdc-form-field-hint-spacer\"></div>\n        <ng-content select=\"mat-hint[align='end']\"></ng-content>\n      </div>\n    }\n  }\n</div>\n", styles: [".mdc-text-field{display:inline-flex;align-items:baseline;padding:0 16px;position:relative;box-sizing:border-box;overflow:hidden;will-change:opacity,transform,color;border-top-left-radius:4px;border-top-right-radius:4px;border-bottom-right-radius:0;border-bottom-left-radius:0}.mdc-text-field__input{width:100%;min-width:0;border:none;border-radius:0;background:none;padding:0;-moz-appearance:none;-webkit-appearance:none;height:28px}.mdc-text-field__input::-webkit-calendar-picker-indicator{display:none}.mdc-text-field__input::-ms-clear{display:none}.mdc-text-field__input:focus{outline:none}.mdc-text-field__input:invalid{box-shadow:none}.mdc-text-field__input::placeholder{opacity:0}.mdc-text-field__input::-moz-placeholder{opacity:0}.mdc-text-field__input::-webkit-input-placeholder{opacity:0}.mdc-text-field__input:-ms-input-placeholder{opacity:0}.mdc-text-field--no-label .mdc-text-field__input::placeholder,.mdc-text-field--focused .mdc-text-field__input::placeholder{opacity:1}.mdc-text-field--no-label .mdc-text-field__input::-moz-placeholder,.mdc-text-field--focused .mdc-text-field__input::-moz-placeholder{opacity:1}.mdc-text-field--no-label .mdc-text-field__input::-webkit-input-placeholder,.mdc-text-field--focused .mdc-text-field__input::-webkit-input-placeholder{opacity:1}.mdc-text-field--no-label .mdc-text-field__input:-ms-input-placeholder,.mdc-text-field--focused .mdc-text-field__input:-ms-input-placeholder{opacity:1}.mdc-text-field--outlined .mdc-text-field__input,.mdc-text-field--filled.mdc-text-field--no-label .mdc-text-field__input{height:100%}.mdc-text-field--outlined .mdc-text-field__input{display:flex;border:none !important;background-color:rgba(0,0,0,0)}.mdc-text-field--disabled .mdc-text-field__input{pointer-events:auto}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-text-field__input{color:var(--mdc-filled-text-field-input-text-color);caret-color:var(--mdc-filled-text-field-caret-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-text-field__input::placeholder{color:var(--mdc-filled-text-field-input-text-placeholder-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-text-field__input::-moz-placeholder{color:var(--mdc-filled-text-field-input-text-placeholder-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-text-field__input::-webkit-input-placeholder{color:var(--mdc-filled-text-field-input-text-placeholder-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-text-field__input:-ms-input-placeholder{color:var(--mdc-filled-text-field-input-text-placeholder-color)}.mdc-text-field--filled.mdc-text-field--invalid:not(.mdc-text-field--disabled) .mdc-text-field__input{caret-color:var(--mdc-filled-text-field-error-caret-color)}.mdc-text-field--filled.mdc-text-field--disabled .mdc-text-field__input{color:var(--mdc-filled-text-field-disabled-input-text-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled) .mdc-text-field__input{color:var(--mdc-outlined-text-field-input-text-color);caret-color:var(--mdc-outlined-text-field-caret-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled) .mdc-text-field__input::placeholder{color:var(--mdc-outlined-text-field-input-text-placeholder-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled) .mdc-text-field__input::-moz-placeholder{color:var(--mdc-outlined-text-field-input-text-placeholder-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled) .mdc-text-field__input::-webkit-input-placeholder{color:var(--mdc-outlined-text-field-input-text-placeholder-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled) .mdc-text-field__input:-ms-input-placeholder{color:var(--mdc-outlined-text-field-input-text-placeholder-color)}.mdc-text-field--outlined.mdc-text-field--invalid:not(.mdc-text-field--disabled) .mdc-text-field__input{caret-color:var(--mdc-outlined-text-field-error-caret-color)}.mdc-text-field--outlined.mdc-text-field--disabled .mdc-text-field__input{color:var(--mdc-outlined-text-field-disabled-input-text-color)}.mdc-text-field--disabled .cdk-high-contrast-active .mdc-text-field__input{background-color:Window}.mdc-text-field--filled{height:56px;border-bottom-right-radius:0;border-bottom-left-radius:0;border-top-left-radius:var(--mdc-filled-text-field-container-shape);border-top-right-radius:var(--mdc-filled-text-field-container-shape)}.mdc-text-field--filled:not(.mdc-text-field--disabled){background-color:var(--mdc-filled-text-field-container-color)}.mdc-text-field--filled.mdc-text-field--disabled{background-color:var(--mdc-filled-text-field-disabled-container-color)}.mdc-text-field--outlined{height:56px;overflow:visible;padding-left:16px;padding-right:16px}@supports(top: max(0%)){.mdc-text-field--outlined{padding-right:max(16px,var(--mdc-outlined-text-field-container-shape));padding-left:max(16px,var(--mdc-outlined-text-field-container-shape) + 4px)}[dir=rtl] .mdc-text-field--outlined{padding-right:max(16px,var(--mdc-outlined-text-field-container-shape) + 4px);padding-left:max(16px,var(--mdc-outlined-text-field-container-shape))}}.mdc-floating-label{position:absolute;left:0;transform-origin:left top;line-height:1.15rem;text-align:left;text-overflow:ellipsis;white-space:nowrap;cursor:text;overflow:hidden;will-change:transform}[dir=rtl] .mdc-floating-label{right:0;left:auto;transform-origin:right top;text-align:right}.mdc-text-field .mdc-floating-label{top:50%;transform:translateY(-50%);pointer-events:none}.mdc-notched-outline .mdc-floating-label{display:inline-block;position:relative;max-width:100%}.mdc-text-field--outlined .mdc-floating-label{left:4px;right:auto}[dir=rtl] .mdc-text-field--outlined .mdc-floating-label{left:auto;right:4px}.mdc-text-field--filled .mdc-floating-label{left:16px;right:auto}[dir=rtl] .mdc-text-field--filled .mdc-floating-label{left:auto;right:16px}.mdc-text-field--disabled .mdc-floating-label{cursor:default}.cdk-high-contrast-active .mdc-text-field--disabled .mdc-floating-label{z-index:1}.mdc-text-field--filled.mdc-text-field--no-label .mdc-floating-label{display:none}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-floating-label{color:var(--mdc-filled-text-field-label-text-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled).mdc-text-field--focused .mdc-floating-label{color:var(--mdc-filled-text-field-focus-label-text-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled):not(.mdc-text-field--focused):hover .mdc-floating-label{color:var(--mdc-filled-text-field-hover-label-text-color)}.mdc-text-field--filled.mdc-text-field--disabled .mdc-floating-label{color:var(--mdc-filled-text-field-disabled-label-text-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled).mdc-text-field--invalid .mdc-floating-label{color:var(--mdc-filled-text-field-error-label-text-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled).mdc-text-field--invalid.mdc-text-field--focused .mdc-floating-label{color:var(--mdc-filled-text-field-error-focus-label-text-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled).mdc-text-field--invalid:not(.mdc-text-field--disabled):hover .mdc-floating-label{color:var(--mdc-filled-text-field-error-hover-label-text-color)}.mdc-text-field--filled .mdc-floating-label{font-family:var(--mdc-filled-text-field-label-text-font);font-size:var(--mdc-filled-text-field-label-text-size);font-weight:var(--mdc-filled-text-field-label-text-weight);letter-spacing:var(--mdc-filled-text-field-label-text-tracking)}.mdc-text-field--outlined:not(.mdc-text-field--disabled) .mdc-floating-label{color:var(--mdc-outlined-text-field-label-text-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--focused .mdc-floating-label{color:var(--mdc-outlined-text-field-focus-label-text-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled):not(.mdc-text-field--focused):hover .mdc-floating-label{color:var(--mdc-outlined-text-field-hover-label-text-color)}.mdc-text-field--outlined.mdc-text-field--disabled .mdc-floating-label{color:var(--mdc-outlined-text-field-disabled-label-text-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--invalid .mdc-floating-label{color:var(--mdc-outlined-text-field-error-label-text-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--invalid.mdc-text-field--focused .mdc-floating-label{color:var(--mdc-outlined-text-field-error-focus-label-text-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--invalid:not(.mdc-text-field--disabled):hover .mdc-floating-label{color:var(--mdc-outlined-text-field-error-hover-label-text-color)}.mdc-text-field--outlined .mdc-floating-label{font-family:var(--mdc-outlined-text-field-label-text-font);font-size:var(--mdc-outlined-text-field-label-text-size);font-weight:var(--mdc-outlined-text-field-label-text-weight);letter-spacing:var(--mdc-outlined-text-field-label-text-tracking)}.mdc-floating-label--float-above{cursor:auto;transform:translateY(-106%) scale(0.75)}.mdc-text-field--filled .mdc-floating-label--float-above{transform:translateY(-106%) scale(0.75)}.mdc-text-field--outlined .mdc-floating-label--float-above{transform:translateY(-37.25px) scale(1);font-size:.75rem}.mdc-notched-outline .mdc-floating-label--float-above{text-overflow:clip}.mdc-notched-outline--upgraded .mdc-floating-label--float-above{max-width:133.3333333333%}.mdc-text-field--outlined.mdc-notched-outline--upgraded .mdc-floating-label--float-above,.mdc-text-field--outlined .mdc-notched-outline--upgraded .mdc-floating-label--float-above{transform:translateY(-34.75px) scale(0.75)}.mdc-text-field--outlined.mdc-notched-outline--upgraded .mdc-floating-label--float-above,.mdc-text-field--outlined .mdc-notched-outline--upgraded .mdc-floating-label--float-above{font-size:1rem}.mdc-floating-label--required:not(.mdc-floating-label--hide-required-marker)::after{margin-left:1px;margin-right:0;content:\"*\"}[dir=rtl] .mdc-floating-label--required:not(.mdc-floating-label--hide-required-marker)::after{margin-left:0;margin-right:1px}.mdc-notched-outline{display:flex;position:absolute;top:0;right:0;left:0;box-sizing:border-box;width:100%;max-width:100%;height:100%;text-align:left;pointer-events:none}[dir=rtl] .mdc-notched-outline{text-align:right}.mdc-text-field--outlined .mdc-notched-outline{z-index:1}.mat-mdc-notch-piece{box-sizing:border-box;height:100%;pointer-events:none;border-top:1px solid;border-bottom:1px solid}.mdc-text-field--focused .mat-mdc-notch-piece{border-width:2px}.mdc-text-field--outlined:not(.mdc-text-field--disabled) .mat-mdc-notch-piece{border-color:var(--mdc-outlined-text-field-outline-color);border-width:var(--mdc-outlined-text-field-outline-width)}.mdc-text-field--outlined:not(.mdc-text-field--disabled):not(.mdc-text-field--focused):hover .mat-mdc-notch-piece{border-color:var(--mdc-outlined-text-field-hover-outline-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--focused .mat-mdc-notch-piece{border-color:var(--mdc-outlined-text-field-focus-outline-color)}.mdc-text-field--outlined.mdc-text-field--disabled .mat-mdc-notch-piece{border-color:var(--mdc-outlined-text-field-disabled-outline-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--invalid .mat-mdc-notch-piece{border-color:var(--mdc-outlined-text-field-error-outline-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--invalid:not(.mdc-text-field--focused):hover .mdc-notched-outline .mat-mdc-notch-piece{border-color:var(--mdc-outlined-text-field-error-hover-outline-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--invalid.mdc-text-field--focused .mat-mdc-notch-piece{border-color:var(--mdc-outlined-text-field-error-focus-outline-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--focused .mdc-notched-outline .mat-mdc-notch-piece{border-width:var(--mdc-outlined-text-field-focus-outline-width)}.mdc-notched-outline__leading{border-left:1px solid;border-right:none;border-top-right-radius:0;border-bottom-right-radius:0;width:12px;border-top-left-radius:var(--mdc-outlined-text-field-container-shape);border-bottom-left-radius:var(--mdc-outlined-text-field-container-shape)}@supports(top: max(0%)){.mdc-text-field--outlined .mdc-notched-outline .mdc-notched-outline__leading{width:max(12px,var(--mdc-outlined-text-field-container-shape))}}[dir=rtl] .mdc-notched-outline__leading{border-left:none;border-right:1px solid;border-bottom-left-radius:0;border-top-left-radius:0;border-top-right-radius:var(--mdc-outlined-text-field-container-shape);border-bottom-right-radius:var(--mdc-outlined-text-field-container-shape)}.mdc-notched-outline__trailing{flex-grow:1;border-left:none;border-right:1px solid;border-top-left-radius:0;border-bottom-left-radius:0;border-top-right-radius:var(--mdc-outlined-text-field-container-shape);border-bottom-right-radius:var(--mdc-outlined-text-field-container-shape)}[dir=rtl] .mdc-notched-outline__trailing{border-left:1px solid;border-right:none;border-top-right-radius:0;border-bottom-right-radius:0;border-top-left-radius:var(--mdc-outlined-text-field-container-shape);border-bottom-left-radius:var(--mdc-outlined-text-field-container-shape)}.mdc-notched-outline__notch{flex:0 0 auto;width:auto;max-width:calc(100% - 24px)}@supports(top: max(0%)){.mdc-text-field--outlined .mdc-notched-outline .mdc-notched-outline__notch{max-width:calc(100% - max(12px,var(--mdc-outlined-text-field-container-shape))*2)}}.mdc-text-field--outlined .mdc-notched-outline--notched .mdc-notched-outline__notch{padding-top:1px}.mdc-text-field--focused.mdc-text-field--outlined .mdc-notched-outline--notched .mdc-notched-outline__notch{padding-top:2px}.mdc-notched-outline--notched .mdc-notched-outline__notch{padding-left:0;padding-right:8px;border-top:none}[dir=rtl] .mdc-notched-outline--notched .mdc-notched-outline__notch{padding-left:8px;padding-right:0}.mdc-notched-outline--no-label .mdc-notched-outline__notch{display:none}.mdc-line-ripple::before,.mdc-line-ripple::after{position:absolute;bottom:0;left:0;width:100%;border-bottom-style:solid;content:\"\"}.mdc-line-ripple::before{z-index:1;border-bottom-width:var(--mdc-filled-text-field-active-indicator-height)}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-line-ripple::before{border-bottom-color:var(--mdc-filled-text-field-active-indicator-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled):not(.mdc-text-field--focused):hover .mdc-line-ripple::before{border-bottom-color:var(--mdc-filled-text-field-hover-active-indicator-color)}.mdc-text-field--filled.mdc-text-field--disabled .mdc-line-ripple::before{border-bottom-color:var(--mdc-filled-text-field-disabled-active-indicator-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled).mdc-text-field--invalid .mdc-line-ripple::before{border-bottom-color:var(--mdc-filled-text-field-error-active-indicator-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled).mdc-text-field--invalid:not(.mdc-text-field--focused):hover .mdc-line-ripple::before{border-bottom-color:var(--mdc-filled-text-field-error-hover-active-indicator-color)}.mdc-line-ripple::after{transform:scaleX(0);opacity:0;z-index:2}.mdc-text-field--filled .mdc-line-ripple::after{border-bottom-width:var(--mdc-filled-text-field-focus-active-indicator-height)}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-line-ripple::after{border-bottom-color:var(--mdc-filled-text-field-focus-active-indicator-color)}.mdc-text-field--filled.mdc-text-field--invalid:not(.mdc-text-field--disabled) .mdc-line-ripple::after{border-bottom-color:var(--mdc-filled-text-field-error-focus-active-indicator-color)}.mdc-line-ripple--active::after{transform:scaleX(1);opacity:1}.mdc-line-ripple--deactivating::after{opacity:0}.mdc-text-field--disabled{pointer-events:none}.mat-mdc-form-field-textarea-control{vertical-align:middle;resize:vertical;box-sizing:border-box;height:auto;margin:0;padding:0;border:none;overflow:auto}.mat-mdc-form-field-input-control.mat-mdc-form-field-input-control{-moz-osx-font-smoothing:grayscale;-webkit-font-smoothing:antialiased;font:inherit;letter-spacing:inherit;text-decoration:inherit;text-transform:inherit;border:none}.mat-mdc-form-field .mat-mdc-floating-label.mdc-floating-label{-moz-osx-font-smoothing:grayscale;-webkit-font-smoothing:antialiased;line-height:normal;pointer-events:all;will-change:auto}.mat-mdc-form-field:not(.mat-form-field-disabled) .mat-mdc-floating-label.mdc-floating-label{cursor:inherit}.mdc-text-field--no-label:not(.mdc-text-field--textarea) .mat-mdc-form-field-input-control.mdc-text-field__input,.mat-mdc-text-field-wrapper .mat-mdc-form-field-input-control{height:auto}.mat-mdc-text-field-wrapper .mat-mdc-form-field-input-control.mdc-text-field__input[type=color]{height:23px}.mat-mdc-text-field-wrapper{height:auto;flex:auto;will-change:auto}.mat-mdc-form-field-has-icon-prefix .mat-mdc-text-field-wrapper{padding-left:0;--mat-mdc-form-field-label-offset-x: -16px}.mat-mdc-form-field-has-icon-suffix .mat-mdc-text-field-wrapper{padding-right:0}[dir=rtl] .mat-mdc-text-field-wrapper{padding-left:16px;padding-right:16px}[dir=rtl] .mat-mdc-form-field-has-icon-suffix .mat-mdc-text-field-wrapper{padding-left:0}[dir=rtl] .mat-mdc-form-field-has-icon-prefix .mat-mdc-text-field-wrapper{padding-right:0}.mat-form-field-disabled .mdc-text-field__input::placeholder{color:var(--mat-form-field-disabled-input-text-placeholder-color)}.mat-form-field-disabled .mdc-text-field__input::-moz-placeholder{color:var(--mat-form-field-disabled-input-text-placeholder-color)}.mat-form-field-disabled .mdc-text-field__input::-webkit-input-placeholder{color:var(--mat-form-field-disabled-input-text-placeholder-color)}.mat-form-field-disabled .mdc-text-field__input:-ms-input-placeholder{color:var(--mat-form-field-disabled-input-text-placeholder-color)}.mat-mdc-form-field-label-always-float .mdc-text-field__input::placeholder{transition-delay:40ms;transition-duration:110ms;opacity:1}.mat-mdc-text-field-wrapper .mat-mdc-form-field-infix .mat-mdc-floating-label{left:auto;right:auto}.mat-mdc-text-field-wrapper.mdc-text-field--outlined .mdc-text-field__input{display:inline-block}.mat-mdc-form-field .mat-mdc-text-field-wrapper.mdc-text-field .mdc-notched-outline__notch{padding-top:0}.mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field .mdc-notched-outline__notch{border-left:1px solid rgba(0,0,0,0)}[dir=rtl] .mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field .mdc-notched-outline__notch{border-left:none;border-right:1px solid rgba(0,0,0,0)}.mat-mdc-form-field-infix{min-height:var(--mat-form-field-container-height);padding-top:var(--mat-form-field-filled-with-label-container-padding-top);padding-bottom:var(--mat-form-field-filled-with-label-container-padding-bottom)}.mdc-text-field--outlined .mat-mdc-form-field-infix,.mdc-text-field--no-label .mat-mdc-form-field-infix{padding-top:var(--mat-form-field-container-vertical-padding);padding-bottom:var(--mat-form-field-container-vertical-padding)}.mat-mdc-text-field-wrapper .mat-mdc-form-field-flex .mat-mdc-floating-label{top:calc(var(--mat-form-field-container-height)/2)}.mdc-text-field--filled .mat-mdc-floating-label{display:var(--mat-form-field-filled-label-display, block)}.mat-mdc-text-field-wrapper.mdc-text-field--outlined .mdc-notched-outline--upgraded .mdc-floating-label--float-above{--mat-mdc-form-field-label-transform: translateY(calc(calc(6.75px + var(--mat-form-field-container-height) / 2) * -1)) scale(var(--mat-mdc-form-field-floating-label-scale, 0.75));transform:var(--mat-mdc-form-field-label-transform)}.mat-mdc-form-field-subscript-wrapper{box-sizing:border-box;width:100%;position:relative}.mat-mdc-form-field-hint-wrapper,.mat-mdc-form-field-error-wrapper{position:absolute;top:0;left:0;right:0;padding:0 16px}.mat-mdc-form-field-subscript-dynamic-size .mat-mdc-form-field-hint-wrapper,.mat-mdc-form-field-subscript-dynamic-size .mat-mdc-form-field-error-wrapper{position:static}.mat-mdc-form-field-bottom-align::before{content:\"\";display:inline-block;height:16px}.mat-mdc-form-field-bottom-align.mat-mdc-form-field-subscript-dynamic-size::before{content:unset}.mat-mdc-form-field-hint-end{order:1}.mat-mdc-form-field-hint-wrapper{display:flex}.mat-mdc-form-field-hint-spacer{flex:1 0 1em}.mat-mdc-form-field-error{display:block;color:var(--mat-form-field-error-text-color)}.mat-mdc-form-field-subscript-wrapper,.mat-mdc-form-field-bottom-align::before{-moz-osx-font-smoothing:grayscale;-webkit-font-smoothing:antialiased;font-family:var(--mat-form-field-subscript-text-font);line-height:var(--mat-form-field-subscript-text-line-height);font-size:var(--mat-form-field-subscript-text-size);letter-spacing:var(--mat-form-field-subscript-text-tracking);font-weight:var(--mat-form-field-subscript-text-weight)}.mat-mdc-form-field-focus-overlay{top:0;left:0;right:0;bottom:0;position:absolute;opacity:0;pointer-events:none;background-color:var(--mat-form-field-state-layer-color)}.mat-mdc-text-field-wrapper:hover .mat-mdc-form-field-focus-overlay{opacity:var(--mat-form-field-hover-state-layer-opacity)}.mat-mdc-form-field.mat-focused .mat-mdc-form-field-focus-overlay{opacity:var(--mat-form-field-focus-state-layer-opacity)}select.mat-mdc-form-field-input-control{-moz-appearance:none;-webkit-appearance:none;background-color:rgba(0,0,0,0);display:inline-flex;box-sizing:border-box}select.mat-mdc-form-field-input-control:not(:disabled){cursor:pointer}select.mat-mdc-form-field-input-control:not(.mat-mdc-native-select-inline) option{color:var(--mat-form-field-select-option-text-color)}select.mat-mdc-form-field-input-control:not(.mat-mdc-native-select-inline) option:disabled{color:var(--mat-form-field-select-disabled-option-text-color)}.mat-mdc-form-field-type-mat-native-select .mat-mdc-form-field-infix::after{content:\"\";width:0;height:0;border-left:5px solid rgba(0,0,0,0);border-right:5px solid rgba(0,0,0,0);border-top:5px solid;position:absolute;right:0;top:50%;margin-top:-2.5px;pointer-events:none;color:var(--mat-form-field-enabled-select-arrow-color)}[dir=rtl] .mat-mdc-form-field-type-mat-native-select .mat-mdc-form-field-infix::after{right:auto;left:0}.mat-mdc-form-field-type-mat-native-select.mat-focused .mat-mdc-form-field-infix::after{color:var(--mat-form-field-focus-select-arrow-color)}.mat-mdc-form-field-type-mat-native-select.mat-form-field-disabled .mat-mdc-form-field-infix::after{color:var(--mat-form-field-disabled-select-arrow-color)}.mat-mdc-form-field-type-mat-native-select .mat-mdc-form-field-input-control{padding-right:15px}[dir=rtl] .mat-mdc-form-field-type-mat-native-select .mat-mdc-form-field-input-control{padding-right:0;padding-left:15px}.cdk-high-contrast-active .mat-form-field-appearance-fill .mat-mdc-text-field-wrapper{outline:solid 1px}.cdk-high-contrast-active .mat-form-field-appearance-fill.mat-form-field-disabled .mat-mdc-text-field-wrapper{outline-color:GrayText}.cdk-high-contrast-active .mat-form-field-appearance-fill.mat-focused .mat-mdc-text-field-wrapper{outline:dashed 3px}.cdk-high-contrast-active .mat-mdc-form-field.mat-focused .mdc-notched-outline{border:dashed 3px}.mat-mdc-form-field-input-control[type=date],.mat-mdc-form-field-input-control[type=datetime],.mat-mdc-form-field-input-control[type=datetime-local],.mat-mdc-form-field-input-control[type=month],.mat-mdc-form-field-input-control[type=week],.mat-mdc-form-field-input-control[type=time]{line-height:1}.mat-mdc-form-field-input-control::-webkit-datetime-edit{line-height:1;padding:0;margin-bottom:-2px}.mat-mdc-form-field{--mat-mdc-form-field-floating-label-scale: 0.75;display:inline-flex;flex-direction:column;min-width:0;text-align:left;-moz-osx-font-smoothing:grayscale;-webkit-font-smoothing:antialiased;font-family:var(--mat-form-field-container-text-font);line-height:var(--mat-form-field-container-text-line-height);font-size:var(--mat-form-field-container-text-size);letter-spacing:var(--mat-form-field-container-text-tracking);font-weight:var(--mat-form-field-container-text-weight)}[dir=rtl] .mat-mdc-form-field{text-align:right}.mat-mdc-form-field .mdc-text-field--outlined .mdc-floating-label--float-above{font-size:calc(var(--mat-form-field-outlined-label-text-populated-size)*var(--mat-mdc-form-field-floating-label-scale))}.mat-mdc-form-field .mdc-text-field--outlined .mdc-notched-outline--upgraded .mdc-floating-label--float-above{font-size:var(--mat-form-field-outlined-label-text-populated-size)}.mat-mdc-form-field-flex{display:inline-flex;align-items:baseline;box-sizing:border-box;width:100%}.mat-mdc-text-field-wrapper{width:100%;z-index:0}.mat-mdc-form-field-icon-prefix,.mat-mdc-form-field-icon-suffix{align-self:center;line-height:0;pointer-events:auto;position:relative;z-index:1}.mat-mdc-form-field-icon-prefix>.mat-icon,.mat-mdc-form-field-icon-suffix>.mat-icon{padding:0 12px;box-sizing:content-box}.mat-mdc-form-field-icon-prefix{color:var(--mat-form-field-leading-icon-color)}.mat-form-field-disabled .mat-mdc-form-field-icon-prefix{color:var(--mat-form-field-disabled-leading-icon-color)}.mat-mdc-form-field-icon-suffix{color:var(--mat-form-field-trailing-icon-color)}.mat-form-field-disabled .mat-mdc-form-field-icon-suffix{color:var(--mat-form-field-disabled-trailing-icon-color)}.mat-form-field-invalid .mat-mdc-form-field-icon-suffix{color:var(--mat-form-field-error-trailing-icon-color)}.mat-form-field-invalid:not(.mat-focused):not(.mat-form-field-disabled) .mat-mdc-text-field-wrapper:hover .mat-mdc-form-field-icon-suffix{color:var(--mat-form-field-error-hover-trailing-icon-color)}.mat-form-field-invalid.mat-focused .mat-mdc-text-field-wrapper .mat-mdc-form-field-icon-suffix{color:var(--mat-form-field-error-focus-trailing-icon-color)}.mat-mdc-form-field-icon-prefix,[dir=rtl] .mat-mdc-form-field-icon-suffix{padding:0 4px 0 0}.mat-mdc-form-field-icon-suffix,[dir=rtl] .mat-mdc-form-field-icon-prefix{padding:0 0 0 4px}.mat-mdc-form-field-subscript-wrapper .mat-icon,.mat-mdc-form-field label .mat-icon{width:1em;height:1em;font-size:inherit}.mat-mdc-form-field-infix{flex:auto;min-width:0;width:180px;position:relative;box-sizing:border-box}.mat-mdc-form-field .mdc-notched-outline__notch{margin-left:-1px;-webkit-clip-path:inset(-9em -999em -9em 1px);clip-path:inset(-9em -999em -9em 1px)}[dir=rtl] .mat-mdc-form-field .mdc-notched-outline__notch{margin-left:0;margin-right:-1px;-webkit-clip-path:inset(-9em 1px -9em -999em);clip-path:inset(-9em 1px -9em -999em)}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-floating-label{transition:transform 150ms cubic-bezier(0.4, 0, 0.2, 1),color 150ms cubic-bezier(0.4, 0, 0.2, 1)}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-text-field__input{transition:opacity 150ms cubic-bezier(0.4, 0, 0.2, 1)}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-text-field__input::placeholder{transition:opacity 67ms cubic-bezier(0.4, 0, 0.2, 1)}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-text-field__input::-moz-placeholder{transition:opacity 67ms cubic-bezier(0.4, 0, 0.2, 1)}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-text-field__input::-webkit-input-placeholder{transition:opacity 67ms cubic-bezier(0.4, 0, 0.2, 1)}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-text-field__input:-ms-input-placeholder{transition:opacity 67ms cubic-bezier(0.4, 0, 0.2, 1)}.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--no-label .mdc-text-field__input::placeholder,.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--focused .mdc-text-field__input::placeholder{transition-delay:40ms;transition-duration:110ms}.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--no-label .mdc-text-field__input::-moz-placeholder,.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--focused .mdc-text-field__input::-moz-placeholder{transition-delay:40ms;transition-duration:110ms}.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--no-label .mdc-text-field__input::-webkit-input-placeholder,.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--focused .mdc-text-field__input::-webkit-input-placeholder{transition-delay:40ms;transition-duration:110ms}.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--no-label .mdc-text-field__input:-ms-input-placeholder,.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--focused .mdc-text-field__input:-ms-input-placeholder{transition-delay:40ms;transition-duration:110ms}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-text-field--filled:not(.mdc-ripple-upgraded):focus .mdc-text-field__ripple::before{transition-duration:75ms}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-line-ripple::after{transition:transform 180ms cubic-bezier(0.4, 0, 0.2, 1),opacity 180ms cubic-bezier(0.4, 0, 0.2, 1)}.mdc-notched-outline .mdc-floating-label{max-width:calc(100% + 1px)}.mdc-notched-outline--upgraded .mdc-floating-label--float-above{max-width:calc(133.3333333333% + 1px)}"], dependencies: [{ kind: "directive", type: MatFormFieldFloatingLabel, selector: "label[matFormFieldFloatingLabel]", inputs: ["floating", "monitorResize"] }, { kind: "component", type: MatFormFieldNotchedOutline, selector: "div[matFormFieldNotchedOutline]", inputs: ["matFormFieldNotchedOutlineOpen"] }, { kind: "directive", type: NgTemplateOutlet, selector: "[ngTemplateOutlet]", inputs: ["ngTemplateOutletContext", "ngTemplateOutlet", "ngTemplateOutletInjector"] }, { kind: "directive", type: MatFormFieldLineRipple, selector: "div[matFormFieldLineRipple]" }, { kind: "directive", type: MatHint, selector: "mat-hint", inputs: ["align", "id"] }], animations: [matFormFieldAnimations.transitionMessages], changeDetection: i0.ChangeDetectionStrategy.OnPush, encapsulation: i0.ViewEncapsulation.None }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.1.0", ngImport: i0, type: MatFormField, decorators: [{
            type: Component,
            args: [{ selector: 'mat-form-field', exportAs: 'matFormField', animations: [matFormFieldAnimations.transitionMessages], host: {
                        'class': 'mat-mdc-form-field',
                        '[class.mat-mdc-form-field-label-always-float]': '_shouldAlwaysFloat()',
                        '[class.mat-mdc-form-field-has-icon-prefix]': '_hasIconPrefix',
                        '[class.mat-mdc-form-field-has-icon-suffix]': '_hasIconSuffix',
                        // Note that these classes reuse the same names as the non-MDC version, because they can be
                        // considered a public API since custom form controls may use them to style themselves.
                        // See https://github.com/angular/components/pull/20502#discussion_r486124901.
                        '[class.mat-form-field-invalid]': '_control.errorState',
                        '[class.mat-form-field-disabled]': '_control.disabled',
                        '[class.mat-form-field-autofilled]': '_control.autofilled',
                        '[class.mat-form-field-no-animations]': '_animationMode === "NoopAnimations"',
                        '[class.mat-form-field-appearance-fill]': 'appearance == "fill"',
                        '[class.mat-form-field-appearance-outline]': 'appearance == "outline"',
                        '[class.mat-form-field-hide-placeholder]': '_hasFloatingLabel() && !_shouldLabelFloat()',
                        '[class.mat-focused]': '_control.focused',
                        '[class.mat-primary]': 'color !== "accent" && color !== "warn"',
                        '[class.mat-accent]': 'color === "accent"',
                        '[class.mat-warn]': 'color === "warn"',
                        '[class.ng-untouched]': '_shouldForward("untouched")',
                        '[class.ng-touched]': '_shouldForward("touched")',
                        '[class.ng-pristine]': '_shouldForward("pristine")',
                        '[class.ng-dirty]': '_shouldForward("dirty")',
                        '[class.ng-valid]': '_shouldForward("valid")',
                        '[class.ng-invalid]': '_shouldForward("invalid")',
                        '[class.ng-pending]': '_shouldForward("pending")',
                    }, encapsulation: ViewEncapsulation.None, changeDetection: ChangeDetectionStrategy.OnPush, providers: [
                        { provide: MAT_FORM_FIELD, useExisting: MatFormField },
                        { provide: FLOATING_LABEL_PARENT, useExisting: MatFormField },
                    ], standalone: true, imports: [
                        MatFormFieldFloatingLabel,
                        MatFormFieldNotchedOutline,
                        NgTemplateOutlet,
                        MatFormFieldLineRipple,
                        MatHint,
                    ], template: "<ng-template #labelTemplate>\n  <!--\n    MDC recommends that the text-field is a `<label>` element. This rather complicates the\n    setup because it would require every form-field control to explicitly set `aria-labelledby`.\n    This is because the `<label>` itself contains more than the actual label (e.g. prefix, suffix\n    or other projected content), and screen readers could potentially read out undesired content.\n    Excluding elements from being printed out requires them to be marked with `aria-hidden`, or\n    the form control is set to a scoped element for the label (using `aria-labelledby`). Both of\n    these options seem to complicate the setup because we know exactly what content is rendered\n    as part of the label, and we don't want to spend resources on walking through projected content\n    to set `aria-hidden`. Nor do we want to set `aria-labelledby` on every form control if we could\n    simply link the label to the control using the label `for` attribute.\n  -->\n  @if (_hasFloatingLabel()) {\n    <label matFormFieldFloatingLabel\n           [floating]=\"_shouldLabelFloat()\"\n           [monitorResize]=\"_hasOutline()\"\n           [id]=\"_labelId\"\n           [attr.for]=\"_control.disableAutomaticLabeling ? null : _control.id\">\n      <ng-content select=\"mat-label\"></ng-content>\n      <!--\n        We set the required marker as a separate element, in order to make it easier to target if\n        apps want to override it and to be able to set `aria-hidden` so that screen readers don't\n        pick it up.\n       -->\n       @if (!hideRequiredMarker && _control.required) {\n         <span\n           aria-hidden=\"true\"\n           class=\"mat-mdc-form-field-required-marker mdc-floating-label--required\"></span>\n       }\n    </label>\n  }\n</ng-template>\n\n<div class=\"mat-mdc-text-field-wrapper mdc-text-field\" #textField\n     [class.mdc-text-field--filled]=\"!_hasOutline()\"\n     [class.mdc-text-field--outlined]=\"_hasOutline()\"\n     [class.mdc-text-field--no-label]=\"!_hasFloatingLabel()\"\n     [class.mdc-text-field--disabled]=\"_control.disabled\"\n     [class.mdc-text-field--invalid]=\"_control.errorState\"\n     (click)=\"_control.onContainerClick($event)\">\n  @if (!_hasOutline() && !_control.disabled) {\n    <div class=\"mat-mdc-form-field-focus-overlay\"></div>\n  }\n  <div class=\"mat-mdc-form-field-flex\">\n    @if (_hasOutline()) {\n      <div matFormFieldNotchedOutline [matFormFieldNotchedOutlineOpen]=\"_shouldLabelFloat()\">\n        @if (!_forceDisplayInfixLabel()) {\n          <ng-template [ngTemplateOutlet]=\"labelTemplate\"></ng-template>\n        }\n      </div>\n    }\n\n    @if (_hasIconPrefix) {\n      <div class=\"mat-mdc-form-field-icon-prefix\" #iconPrefixContainer>\n        <ng-content select=\"[matPrefix], [matIconPrefix]\"></ng-content>\n      </div>\n    }\n\n    @if (_hasTextPrefix) {\n      <div class=\"mat-mdc-form-field-text-prefix\" #textPrefixContainer>\n        <ng-content select=\"[matTextPrefix]\"></ng-content>\n      </div>\n    }\n\n    <div class=\"mat-mdc-form-field-infix\">\n      @if (!_hasOutline() || _forceDisplayInfixLabel()) {\n        <ng-template [ngTemplateOutlet]=\"labelTemplate\"></ng-template>\n      }\n\n      <ng-content></ng-content>\n    </div>\n\n    @if (_hasTextSuffix) {\n      <div class=\"mat-mdc-form-field-text-suffix\">\n        <ng-content select=\"[matTextSuffix]\"></ng-content>\n      </div>\n    }\n\n    @if (_hasIconSuffix) {\n      <div class=\"mat-mdc-form-field-icon-suffix\">\n        <ng-content select=\"[matSuffix], [matIconSuffix]\"></ng-content>\n      </div>\n    }\n  </div>\n\n  @if (!_hasOutline()) {\n    <div matFormFieldLineRipple></div>\n  }\n</div>\n\n<div class=\"mat-mdc-form-field-subscript-wrapper mat-mdc-form-field-bottom-align\"\n     [class.mat-mdc-form-field-subscript-dynamic-size]=\"subscriptSizing === 'dynamic'\">\n\n  @switch (_getDisplayedMessages()) {\n    @case ('error') {\n      <div class=\"mat-mdc-form-field-error-wrapper\"\n           [@transitionMessages]=\"_subscriptAnimationState\">\n        <ng-content select=\"mat-error, [matError]\"></ng-content>\n      </div>\n    }\n\n    @case ('hint') {\n      <div class=\"mat-mdc-form-field-hint-wrapper\" [@transitionMessages]=\"_subscriptAnimationState\">\n        @if (hintLabel) {\n          <mat-hint [id]=\"_hintLabelId\">{{hintLabel}}</mat-hint>\n        }\n        <ng-content select=\"mat-hint:not([align='end'])\"></ng-content>\n        <div class=\"mat-mdc-form-field-hint-spacer\"></div>\n        <ng-content select=\"mat-hint[align='end']\"></ng-content>\n      </div>\n    }\n  }\n</div>\n", styles: [".mdc-text-field{display:inline-flex;align-items:baseline;padding:0 16px;position:relative;box-sizing:border-box;overflow:hidden;will-change:opacity,transform,color;border-top-left-radius:4px;border-top-right-radius:4px;border-bottom-right-radius:0;border-bottom-left-radius:0}.mdc-text-field__input{width:100%;min-width:0;border:none;border-radius:0;background:none;padding:0;-moz-appearance:none;-webkit-appearance:none;height:28px}.mdc-text-field__input::-webkit-calendar-picker-indicator{display:none}.mdc-text-field__input::-ms-clear{display:none}.mdc-text-field__input:focus{outline:none}.mdc-text-field__input:invalid{box-shadow:none}.mdc-text-field__input::placeholder{opacity:0}.mdc-text-field__input::-moz-placeholder{opacity:0}.mdc-text-field__input::-webkit-input-placeholder{opacity:0}.mdc-text-field__input:-ms-input-placeholder{opacity:0}.mdc-text-field--no-label .mdc-text-field__input::placeholder,.mdc-text-field--focused .mdc-text-field__input::placeholder{opacity:1}.mdc-text-field--no-label .mdc-text-field__input::-moz-placeholder,.mdc-text-field--focused .mdc-text-field__input::-moz-placeholder{opacity:1}.mdc-text-field--no-label .mdc-text-field__input::-webkit-input-placeholder,.mdc-text-field--focused .mdc-text-field__input::-webkit-input-placeholder{opacity:1}.mdc-text-field--no-label .mdc-text-field__input:-ms-input-placeholder,.mdc-text-field--focused .mdc-text-field__input:-ms-input-placeholder{opacity:1}.mdc-text-field--outlined .mdc-text-field__input,.mdc-text-field--filled.mdc-text-field--no-label .mdc-text-field__input{height:100%}.mdc-text-field--outlined .mdc-text-field__input{display:flex;border:none !important;background-color:rgba(0,0,0,0)}.mdc-text-field--disabled .mdc-text-field__input{pointer-events:auto}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-text-field__input{color:var(--mdc-filled-text-field-input-text-color);caret-color:var(--mdc-filled-text-field-caret-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-text-field__input::placeholder{color:var(--mdc-filled-text-field-input-text-placeholder-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-text-field__input::-moz-placeholder{color:var(--mdc-filled-text-field-input-text-placeholder-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-text-field__input::-webkit-input-placeholder{color:var(--mdc-filled-text-field-input-text-placeholder-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-text-field__input:-ms-input-placeholder{color:var(--mdc-filled-text-field-input-text-placeholder-color)}.mdc-text-field--filled.mdc-text-field--invalid:not(.mdc-text-field--disabled) .mdc-text-field__input{caret-color:var(--mdc-filled-text-field-error-caret-color)}.mdc-text-field--filled.mdc-text-field--disabled .mdc-text-field__input{color:var(--mdc-filled-text-field-disabled-input-text-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled) .mdc-text-field__input{color:var(--mdc-outlined-text-field-input-text-color);caret-color:var(--mdc-outlined-text-field-caret-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled) .mdc-text-field__input::placeholder{color:var(--mdc-outlined-text-field-input-text-placeholder-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled) .mdc-text-field__input::-moz-placeholder{color:var(--mdc-outlined-text-field-input-text-placeholder-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled) .mdc-text-field__input::-webkit-input-placeholder{color:var(--mdc-outlined-text-field-input-text-placeholder-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled) .mdc-text-field__input:-ms-input-placeholder{color:var(--mdc-outlined-text-field-input-text-placeholder-color)}.mdc-text-field--outlined.mdc-text-field--invalid:not(.mdc-text-field--disabled) .mdc-text-field__input{caret-color:var(--mdc-outlined-text-field-error-caret-color)}.mdc-text-field--outlined.mdc-text-field--disabled .mdc-text-field__input{color:var(--mdc-outlined-text-field-disabled-input-text-color)}.mdc-text-field--disabled .cdk-high-contrast-active .mdc-text-field__input{background-color:Window}.mdc-text-field--filled{height:56px;border-bottom-right-radius:0;border-bottom-left-radius:0;border-top-left-radius:var(--mdc-filled-text-field-container-shape);border-top-right-radius:var(--mdc-filled-text-field-container-shape)}.mdc-text-field--filled:not(.mdc-text-field--disabled){background-color:var(--mdc-filled-text-field-container-color)}.mdc-text-field--filled.mdc-text-field--disabled{background-color:var(--mdc-filled-text-field-disabled-container-color)}.mdc-text-field--outlined{height:56px;overflow:visible;padding-left:16px;padding-right:16px}@supports(top: max(0%)){.mdc-text-field--outlined{padding-right:max(16px,var(--mdc-outlined-text-field-container-shape));padding-left:max(16px,var(--mdc-outlined-text-field-container-shape) + 4px)}[dir=rtl] .mdc-text-field--outlined{padding-right:max(16px,var(--mdc-outlined-text-field-container-shape) + 4px);padding-left:max(16px,var(--mdc-outlined-text-field-container-shape))}}.mdc-floating-label{position:absolute;left:0;transform-origin:left top;line-height:1.15rem;text-align:left;text-overflow:ellipsis;white-space:nowrap;cursor:text;overflow:hidden;will-change:transform}[dir=rtl] .mdc-floating-label{right:0;left:auto;transform-origin:right top;text-align:right}.mdc-text-field .mdc-floating-label{top:50%;transform:translateY(-50%);pointer-events:none}.mdc-notched-outline .mdc-floating-label{display:inline-block;position:relative;max-width:100%}.mdc-text-field--outlined .mdc-floating-label{left:4px;right:auto}[dir=rtl] .mdc-text-field--outlined .mdc-floating-label{left:auto;right:4px}.mdc-text-field--filled .mdc-floating-label{left:16px;right:auto}[dir=rtl] .mdc-text-field--filled .mdc-floating-label{left:auto;right:16px}.mdc-text-field--disabled .mdc-floating-label{cursor:default}.cdk-high-contrast-active .mdc-text-field--disabled .mdc-floating-label{z-index:1}.mdc-text-field--filled.mdc-text-field--no-label .mdc-floating-label{display:none}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-floating-label{color:var(--mdc-filled-text-field-label-text-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled).mdc-text-field--focused .mdc-floating-label{color:var(--mdc-filled-text-field-focus-label-text-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled):not(.mdc-text-field--focused):hover .mdc-floating-label{color:var(--mdc-filled-text-field-hover-label-text-color)}.mdc-text-field--filled.mdc-text-field--disabled .mdc-floating-label{color:var(--mdc-filled-text-field-disabled-label-text-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled).mdc-text-field--invalid .mdc-floating-label{color:var(--mdc-filled-text-field-error-label-text-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled).mdc-text-field--invalid.mdc-text-field--focused .mdc-floating-label{color:var(--mdc-filled-text-field-error-focus-label-text-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled).mdc-text-field--invalid:not(.mdc-text-field--disabled):hover .mdc-floating-label{color:var(--mdc-filled-text-field-error-hover-label-text-color)}.mdc-text-field--filled .mdc-floating-label{font-family:var(--mdc-filled-text-field-label-text-font);font-size:var(--mdc-filled-text-field-label-text-size);font-weight:var(--mdc-filled-text-field-label-text-weight);letter-spacing:var(--mdc-filled-text-field-label-text-tracking)}.mdc-text-field--outlined:not(.mdc-text-field--disabled) .mdc-floating-label{color:var(--mdc-outlined-text-field-label-text-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--focused .mdc-floating-label{color:var(--mdc-outlined-text-field-focus-label-text-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled):not(.mdc-text-field--focused):hover .mdc-floating-label{color:var(--mdc-outlined-text-field-hover-label-text-color)}.mdc-text-field--outlined.mdc-text-field--disabled .mdc-floating-label{color:var(--mdc-outlined-text-field-disabled-label-text-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--invalid .mdc-floating-label{color:var(--mdc-outlined-text-field-error-label-text-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--invalid.mdc-text-field--focused .mdc-floating-label{color:var(--mdc-outlined-text-field-error-focus-label-text-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--invalid:not(.mdc-text-field--disabled):hover .mdc-floating-label{color:var(--mdc-outlined-text-field-error-hover-label-text-color)}.mdc-text-field--outlined .mdc-floating-label{font-family:var(--mdc-outlined-text-field-label-text-font);font-size:var(--mdc-outlined-text-field-label-text-size);font-weight:var(--mdc-outlined-text-field-label-text-weight);letter-spacing:var(--mdc-outlined-text-field-label-text-tracking)}.mdc-floating-label--float-above{cursor:auto;transform:translateY(-106%) scale(0.75)}.mdc-text-field--filled .mdc-floating-label--float-above{transform:translateY(-106%) scale(0.75)}.mdc-text-field--outlined .mdc-floating-label--float-above{transform:translateY(-37.25px) scale(1);font-size:.75rem}.mdc-notched-outline .mdc-floating-label--float-above{text-overflow:clip}.mdc-notched-outline--upgraded .mdc-floating-label--float-above{max-width:133.3333333333%}.mdc-text-field--outlined.mdc-notched-outline--upgraded .mdc-floating-label--float-above,.mdc-text-field--outlined .mdc-notched-outline--upgraded .mdc-floating-label--float-above{transform:translateY(-34.75px) scale(0.75)}.mdc-text-field--outlined.mdc-notched-outline--upgraded .mdc-floating-label--float-above,.mdc-text-field--outlined .mdc-notched-outline--upgraded .mdc-floating-label--float-above{font-size:1rem}.mdc-floating-label--required:not(.mdc-floating-label--hide-required-marker)::after{margin-left:1px;margin-right:0;content:\"*\"}[dir=rtl] .mdc-floating-label--required:not(.mdc-floating-label--hide-required-marker)::after{margin-left:0;margin-right:1px}.mdc-notched-outline{display:flex;position:absolute;top:0;right:0;left:0;box-sizing:border-box;width:100%;max-width:100%;height:100%;text-align:left;pointer-events:none}[dir=rtl] .mdc-notched-outline{text-align:right}.mdc-text-field--outlined .mdc-notched-outline{z-index:1}.mat-mdc-notch-piece{box-sizing:border-box;height:100%;pointer-events:none;border-top:1px solid;border-bottom:1px solid}.mdc-text-field--focused .mat-mdc-notch-piece{border-width:2px}.mdc-text-field--outlined:not(.mdc-text-field--disabled) .mat-mdc-notch-piece{border-color:var(--mdc-outlined-text-field-outline-color);border-width:var(--mdc-outlined-text-field-outline-width)}.mdc-text-field--outlined:not(.mdc-text-field--disabled):not(.mdc-text-field--focused):hover .mat-mdc-notch-piece{border-color:var(--mdc-outlined-text-field-hover-outline-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--focused .mat-mdc-notch-piece{border-color:var(--mdc-outlined-text-field-focus-outline-color)}.mdc-text-field--outlined.mdc-text-field--disabled .mat-mdc-notch-piece{border-color:var(--mdc-outlined-text-field-disabled-outline-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--invalid .mat-mdc-notch-piece{border-color:var(--mdc-outlined-text-field-error-outline-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--invalid:not(.mdc-text-field--focused):hover .mdc-notched-outline .mat-mdc-notch-piece{border-color:var(--mdc-outlined-text-field-error-hover-outline-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--invalid.mdc-text-field--focused .mat-mdc-notch-piece{border-color:var(--mdc-outlined-text-field-error-focus-outline-color)}.mdc-text-field--outlined:not(.mdc-text-field--disabled).mdc-text-field--focused .mdc-notched-outline .mat-mdc-notch-piece{border-width:var(--mdc-outlined-text-field-focus-outline-width)}.mdc-notched-outline__leading{border-left:1px solid;border-right:none;border-top-right-radius:0;border-bottom-right-radius:0;width:12px;border-top-left-radius:var(--mdc-outlined-text-field-container-shape);border-bottom-left-radius:var(--mdc-outlined-text-field-container-shape)}@supports(top: max(0%)){.mdc-text-field--outlined .mdc-notched-outline .mdc-notched-outline__leading{width:max(12px,var(--mdc-outlined-text-field-container-shape))}}[dir=rtl] .mdc-notched-outline__leading{border-left:none;border-right:1px solid;border-bottom-left-radius:0;border-top-left-radius:0;border-top-right-radius:var(--mdc-outlined-text-field-container-shape);border-bottom-right-radius:var(--mdc-outlined-text-field-container-shape)}.mdc-notched-outline__trailing{flex-grow:1;border-left:none;border-right:1px solid;border-top-left-radius:0;border-bottom-left-radius:0;border-top-right-radius:var(--mdc-outlined-text-field-container-shape);border-bottom-right-radius:var(--mdc-outlined-text-field-container-shape)}[dir=rtl] .mdc-notched-outline__trailing{border-left:1px solid;border-right:none;border-top-right-radius:0;border-bottom-right-radius:0;border-top-left-radius:var(--mdc-outlined-text-field-container-shape);border-bottom-left-radius:var(--mdc-outlined-text-field-container-shape)}.mdc-notched-outline__notch{flex:0 0 auto;width:auto;max-width:calc(100% - 24px)}@supports(top: max(0%)){.mdc-text-field--outlined .mdc-notched-outline .mdc-notched-outline__notch{max-width:calc(100% - max(12px,var(--mdc-outlined-text-field-container-shape))*2)}}.mdc-text-field--outlined .mdc-notched-outline--notched .mdc-notched-outline__notch{padding-top:1px}.mdc-text-field--focused.mdc-text-field--outlined .mdc-notched-outline--notched .mdc-notched-outline__notch{padding-top:2px}.mdc-notched-outline--notched .mdc-notched-outline__notch{padding-left:0;padding-right:8px;border-top:none}[dir=rtl] .mdc-notched-outline--notched .mdc-notched-outline__notch{padding-left:8px;padding-right:0}.mdc-notched-outline--no-label .mdc-notched-outline__notch{display:none}.mdc-line-ripple::before,.mdc-line-ripple::after{position:absolute;bottom:0;left:0;width:100%;border-bottom-style:solid;content:\"\"}.mdc-line-ripple::before{z-index:1;border-bottom-width:var(--mdc-filled-text-field-active-indicator-height)}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-line-ripple::before{border-bottom-color:var(--mdc-filled-text-field-active-indicator-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled):not(.mdc-text-field--focused):hover .mdc-line-ripple::before{border-bottom-color:var(--mdc-filled-text-field-hover-active-indicator-color)}.mdc-text-field--filled.mdc-text-field--disabled .mdc-line-ripple::before{border-bottom-color:var(--mdc-filled-text-field-disabled-active-indicator-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled).mdc-text-field--invalid .mdc-line-ripple::before{border-bottom-color:var(--mdc-filled-text-field-error-active-indicator-color)}.mdc-text-field--filled:not(.mdc-text-field--disabled).mdc-text-field--invalid:not(.mdc-text-field--focused):hover .mdc-line-ripple::before{border-bottom-color:var(--mdc-filled-text-field-error-hover-active-indicator-color)}.mdc-line-ripple::after{transform:scaleX(0);opacity:0;z-index:2}.mdc-text-field--filled .mdc-line-ripple::after{border-bottom-width:var(--mdc-filled-text-field-focus-active-indicator-height)}.mdc-text-field--filled:not(.mdc-text-field--disabled) .mdc-line-ripple::after{border-bottom-color:var(--mdc-filled-text-field-focus-active-indicator-color)}.mdc-text-field--filled.mdc-text-field--invalid:not(.mdc-text-field--disabled) .mdc-line-ripple::after{border-bottom-color:var(--mdc-filled-text-field-error-focus-active-indicator-color)}.mdc-line-ripple--active::after{transform:scaleX(1);opacity:1}.mdc-line-ripple--deactivating::after{opacity:0}.mdc-text-field--disabled{pointer-events:none}.mat-mdc-form-field-textarea-control{vertical-align:middle;resize:vertical;box-sizing:border-box;height:auto;margin:0;padding:0;border:none;overflow:auto}.mat-mdc-form-field-input-control.mat-mdc-form-field-input-control{-moz-osx-font-smoothing:grayscale;-webkit-font-smoothing:antialiased;font:inherit;letter-spacing:inherit;text-decoration:inherit;text-transform:inherit;border:none}.mat-mdc-form-field .mat-mdc-floating-label.mdc-floating-label{-moz-osx-font-smoothing:grayscale;-webkit-font-smoothing:antialiased;line-height:normal;pointer-events:all;will-change:auto}.mat-mdc-form-field:not(.mat-form-field-disabled) .mat-mdc-floating-label.mdc-floating-label{cursor:inherit}.mdc-text-field--no-label:not(.mdc-text-field--textarea) .mat-mdc-form-field-input-control.mdc-text-field__input,.mat-mdc-text-field-wrapper .mat-mdc-form-field-input-control{height:auto}.mat-mdc-text-field-wrapper .mat-mdc-form-field-input-control.mdc-text-field__input[type=color]{height:23px}.mat-mdc-text-field-wrapper{height:auto;flex:auto;will-change:auto}.mat-mdc-form-field-has-icon-prefix .mat-mdc-text-field-wrapper{padding-left:0;--mat-mdc-form-field-label-offset-x: -16px}.mat-mdc-form-field-has-icon-suffix .mat-mdc-text-field-wrapper{padding-right:0}[dir=rtl] .mat-mdc-text-field-wrapper{padding-left:16px;padding-right:16px}[dir=rtl] .mat-mdc-form-field-has-icon-suffix .mat-mdc-text-field-wrapper{padding-left:0}[dir=rtl] .mat-mdc-form-field-has-icon-prefix .mat-mdc-text-field-wrapper{padding-right:0}.mat-form-field-disabled .mdc-text-field__input::placeholder{color:var(--mat-form-field-disabled-input-text-placeholder-color)}.mat-form-field-disabled .mdc-text-field__input::-moz-placeholder{color:var(--mat-form-field-disabled-input-text-placeholder-color)}.mat-form-field-disabled .mdc-text-field__input::-webkit-input-placeholder{color:var(--mat-form-field-disabled-input-text-placeholder-color)}.mat-form-field-disabled .mdc-text-field__input:-ms-input-placeholder{color:var(--mat-form-field-disabled-input-text-placeholder-color)}.mat-mdc-form-field-label-always-float .mdc-text-field__input::placeholder{transition-delay:40ms;transition-duration:110ms;opacity:1}.mat-mdc-text-field-wrapper .mat-mdc-form-field-infix .mat-mdc-floating-label{left:auto;right:auto}.mat-mdc-text-field-wrapper.mdc-text-field--outlined .mdc-text-field__input{display:inline-block}.mat-mdc-form-field .mat-mdc-text-field-wrapper.mdc-text-field .mdc-notched-outline__notch{padding-top:0}.mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field .mdc-notched-outline__notch{border-left:1px solid rgba(0,0,0,0)}[dir=rtl] .mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field.mat-mdc-form-field .mdc-notched-outline__notch{border-left:none;border-right:1px solid rgba(0,0,0,0)}.mat-mdc-form-field-infix{min-height:var(--mat-form-field-container-height);padding-top:var(--mat-form-field-filled-with-label-container-padding-top);padding-bottom:var(--mat-form-field-filled-with-label-container-padding-bottom)}.mdc-text-field--outlined .mat-mdc-form-field-infix,.mdc-text-field--no-label .mat-mdc-form-field-infix{padding-top:var(--mat-form-field-container-vertical-padding);padding-bottom:var(--mat-form-field-container-vertical-padding)}.mat-mdc-text-field-wrapper .mat-mdc-form-field-flex .mat-mdc-floating-label{top:calc(var(--mat-form-field-container-height)/2)}.mdc-text-field--filled .mat-mdc-floating-label{display:var(--mat-form-field-filled-label-display, block)}.mat-mdc-text-field-wrapper.mdc-text-field--outlined .mdc-notched-outline--upgraded .mdc-floating-label--float-above{--mat-mdc-form-field-label-transform: translateY(calc(calc(6.75px + var(--mat-form-field-container-height) / 2) * -1)) scale(var(--mat-mdc-form-field-floating-label-scale, 0.75));transform:var(--mat-mdc-form-field-label-transform)}.mat-mdc-form-field-subscript-wrapper{box-sizing:border-box;width:100%;position:relative}.mat-mdc-form-field-hint-wrapper,.mat-mdc-form-field-error-wrapper{position:absolute;top:0;left:0;right:0;padding:0 16px}.mat-mdc-form-field-subscript-dynamic-size .mat-mdc-form-field-hint-wrapper,.mat-mdc-form-field-subscript-dynamic-size .mat-mdc-form-field-error-wrapper{position:static}.mat-mdc-form-field-bottom-align::before{content:\"\";display:inline-block;height:16px}.mat-mdc-form-field-bottom-align.mat-mdc-form-field-subscript-dynamic-size::before{content:unset}.mat-mdc-form-field-hint-end{order:1}.mat-mdc-form-field-hint-wrapper{display:flex}.mat-mdc-form-field-hint-spacer{flex:1 0 1em}.mat-mdc-form-field-error{display:block;color:var(--mat-form-field-error-text-color)}.mat-mdc-form-field-subscript-wrapper,.mat-mdc-form-field-bottom-align::before{-moz-osx-font-smoothing:grayscale;-webkit-font-smoothing:antialiased;font-family:var(--mat-form-field-subscript-text-font);line-height:var(--mat-form-field-subscript-text-line-height);font-size:var(--mat-form-field-subscript-text-size);letter-spacing:var(--mat-form-field-subscript-text-tracking);font-weight:var(--mat-form-field-subscript-text-weight)}.mat-mdc-form-field-focus-overlay{top:0;left:0;right:0;bottom:0;position:absolute;opacity:0;pointer-events:none;background-color:var(--mat-form-field-state-layer-color)}.mat-mdc-text-field-wrapper:hover .mat-mdc-form-field-focus-overlay{opacity:var(--mat-form-field-hover-state-layer-opacity)}.mat-mdc-form-field.mat-focused .mat-mdc-form-field-focus-overlay{opacity:var(--mat-form-field-focus-state-layer-opacity)}select.mat-mdc-form-field-input-control{-moz-appearance:none;-webkit-appearance:none;background-color:rgba(0,0,0,0);display:inline-flex;box-sizing:border-box}select.mat-mdc-form-field-input-control:not(:disabled){cursor:pointer}select.mat-mdc-form-field-input-control:not(.mat-mdc-native-select-inline) option{color:var(--mat-form-field-select-option-text-color)}select.mat-mdc-form-field-input-control:not(.mat-mdc-native-select-inline) option:disabled{color:var(--mat-form-field-select-disabled-option-text-color)}.mat-mdc-form-field-type-mat-native-select .mat-mdc-form-field-infix::after{content:\"\";width:0;height:0;border-left:5px solid rgba(0,0,0,0);border-right:5px solid rgba(0,0,0,0);border-top:5px solid;position:absolute;right:0;top:50%;margin-top:-2.5px;pointer-events:none;color:var(--mat-form-field-enabled-select-arrow-color)}[dir=rtl] .mat-mdc-form-field-type-mat-native-select .mat-mdc-form-field-infix::after{right:auto;left:0}.mat-mdc-form-field-type-mat-native-select.mat-focused .mat-mdc-form-field-infix::after{color:var(--mat-form-field-focus-select-arrow-color)}.mat-mdc-form-field-type-mat-native-select.mat-form-field-disabled .mat-mdc-form-field-infix::after{color:var(--mat-form-field-disabled-select-arrow-color)}.mat-mdc-form-field-type-mat-native-select .mat-mdc-form-field-input-control{padding-right:15px}[dir=rtl] .mat-mdc-form-field-type-mat-native-select .mat-mdc-form-field-input-control{padding-right:0;padding-left:15px}.cdk-high-contrast-active .mat-form-field-appearance-fill .mat-mdc-text-field-wrapper{outline:solid 1px}.cdk-high-contrast-active .mat-form-field-appearance-fill.mat-form-field-disabled .mat-mdc-text-field-wrapper{outline-color:GrayText}.cdk-high-contrast-active .mat-form-field-appearance-fill.mat-focused .mat-mdc-text-field-wrapper{outline:dashed 3px}.cdk-high-contrast-active .mat-mdc-form-field.mat-focused .mdc-notched-outline{border:dashed 3px}.mat-mdc-form-field-input-control[type=date],.mat-mdc-form-field-input-control[type=datetime],.mat-mdc-form-field-input-control[type=datetime-local],.mat-mdc-form-field-input-control[type=month],.mat-mdc-form-field-input-control[type=week],.mat-mdc-form-field-input-control[type=time]{line-height:1}.mat-mdc-form-field-input-control::-webkit-datetime-edit{line-height:1;padding:0;margin-bottom:-2px}.mat-mdc-form-field{--mat-mdc-form-field-floating-label-scale: 0.75;display:inline-flex;flex-direction:column;min-width:0;text-align:left;-moz-osx-font-smoothing:grayscale;-webkit-font-smoothing:antialiased;font-family:var(--mat-form-field-container-text-font);line-height:var(--mat-form-field-container-text-line-height);font-size:var(--mat-form-field-container-text-size);letter-spacing:var(--mat-form-field-container-text-tracking);font-weight:var(--mat-form-field-container-text-weight)}[dir=rtl] .mat-mdc-form-field{text-align:right}.mat-mdc-form-field .mdc-text-field--outlined .mdc-floating-label--float-above{font-size:calc(var(--mat-form-field-outlined-label-text-populated-size)*var(--mat-mdc-form-field-floating-label-scale))}.mat-mdc-form-field .mdc-text-field--outlined .mdc-notched-outline--upgraded .mdc-floating-label--float-above{font-size:var(--mat-form-field-outlined-label-text-populated-size)}.mat-mdc-form-field-flex{display:inline-flex;align-items:baseline;box-sizing:border-box;width:100%}.mat-mdc-text-field-wrapper{width:100%;z-index:0}.mat-mdc-form-field-icon-prefix,.mat-mdc-form-field-icon-suffix{align-self:center;line-height:0;pointer-events:auto;position:relative;z-index:1}.mat-mdc-form-field-icon-prefix>.mat-icon,.mat-mdc-form-field-icon-suffix>.mat-icon{padding:0 12px;box-sizing:content-box}.mat-mdc-form-field-icon-prefix{color:var(--mat-form-field-leading-icon-color)}.mat-form-field-disabled .mat-mdc-form-field-icon-prefix{color:var(--mat-form-field-disabled-leading-icon-color)}.mat-mdc-form-field-icon-suffix{color:var(--mat-form-field-trailing-icon-color)}.mat-form-field-disabled .mat-mdc-form-field-icon-suffix{color:var(--mat-form-field-disabled-trailing-icon-color)}.mat-form-field-invalid .mat-mdc-form-field-icon-suffix{color:var(--mat-form-field-error-trailing-icon-color)}.mat-form-field-invalid:not(.mat-focused):not(.mat-form-field-disabled) .mat-mdc-text-field-wrapper:hover .mat-mdc-form-field-icon-suffix{color:var(--mat-form-field-error-hover-trailing-icon-color)}.mat-form-field-invalid.mat-focused .mat-mdc-text-field-wrapper .mat-mdc-form-field-icon-suffix{color:var(--mat-form-field-error-focus-trailing-icon-color)}.mat-mdc-form-field-icon-prefix,[dir=rtl] .mat-mdc-form-field-icon-suffix{padding:0 4px 0 0}.mat-mdc-form-field-icon-suffix,[dir=rtl] .mat-mdc-form-field-icon-prefix{padding:0 0 0 4px}.mat-mdc-form-field-subscript-wrapper .mat-icon,.mat-mdc-form-field label .mat-icon{width:1em;height:1em;font-size:inherit}.mat-mdc-form-field-infix{flex:auto;min-width:0;width:180px;position:relative;box-sizing:border-box}.mat-mdc-form-field .mdc-notched-outline__notch{margin-left:-1px;-webkit-clip-path:inset(-9em -999em -9em 1px);clip-path:inset(-9em -999em -9em 1px)}[dir=rtl] .mat-mdc-form-field .mdc-notched-outline__notch{margin-left:0;margin-right:-1px;-webkit-clip-path:inset(-9em 1px -9em -999em);clip-path:inset(-9em 1px -9em -999em)}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-floating-label{transition:transform 150ms cubic-bezier(0.4, 0, 0.2, 1),color 150ms cubic-bezier(0.4, 0, 0.2, 1)}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-text-field__input{transition:opacity 150ms cubic-bezier(0.4, 0, 0.2, 1)}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-text-field__input::placeholder{transition:opacity 67ms cubic-bezier(0.4, 0, 0.2, 1)}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-text-field__input::-moz-placeholder{transition:opacity 67ms cubic-bezier(0.4, 0, 0.2, 1)}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-text-field__input::-webkit-input-placeholder{transition:opacity 67ms cubic-bezier(0.4, 0, 0.2, 1)}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-text-field__input:-ms-input-placeholder{transition:opacity 67ms cubic-bezier(0.4, 0, 0.2, 1)}.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--no-label .mdc-text-field__input::placeholder,.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--focused .mdc-text-field__input::placeholder{transition-delay:40ms;transition-duration:110ms}.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--no-label .mdc-text-field__input::-moz-placeholder,.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--focused .mdc-text-field__input::-moz-placeholder{transition-delay:40ms;transition-duration:110ms}.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--no-label .mdc-text-field__input::-webkit-input-placeholder,.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--focused .mdc-text-field__input::-webkit-input-placeholder{transition-delay:40ms;transition-duration:110ms}.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--no-label .mdc-text-field__input:-ms-input-placeholder,.mat-mdc-form-field:not(.mat-form-field-no-animations).mdc-text-field--focused .mdc-text-field__input:-ms-input-placeholder{transition-delay:40ms;transition-duration:110ms}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-text-field--filled:not(.mdc-ripple-upgraded):focus .mdc-text-field__ripple::before{transition-duration:75ms}.mat-mdc-form-field:not(.mat-form-field-no-animations) .mdc-line-ripple::after{transition:transform 180ms cubic-bezier(0.4, 0, 0.2, 1),opacity 180ms cubic-bezier(0.4, 0, 0.2, 1)}.mdc-notched-outline .mdc-floating-label{max-width:calc(100% + 1px)}.mdc-notched-outline--upgraded .mdc-floating-label--float-above{max-width:calc(133.3333333333% + 1px)}"] }]
        }], ctorParameters: () => [{ type: i0.ElementRef }, { type: i0.ChangeDetectorRef }, { type: i0.NgZone }, { type: i1.Directionality }, { type: i2.Platform }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [MAT_FORM_FIELD_DEFAULT_OPTIONS]
                }] }, { type: undefined, decorators: [{
                    type: Optional
                }, {
                    type: Inject,
                    args: [ANIMATION_MODULE_TYPE]
                }] }, { type: undefined, decorators: [{
                    type: Inject,
                    args: [DOCUMENT]
                }] }], propDecorators: { _textField: [{
                type: ViewChild,
                args: ['textField']
            }], _iconPrefixContainer: [{
                type: ViewChild,
                args: ['iconPrefixContainer']
            }], _textPrefixContainer: [{
                type: ViewChild,
                args: ['textPrefixContainer']
            }], _floatingLabel: [{
                type: ViewChild,
                args: [MatFormFieldFloatingLabel]
            }], _notchedOutline: [{
                type: ViewChild,
                args: [MatFormFieldNotchedOutline]
            }], _lineRipple: [{
                type: ViewChild,
                args: [MatFormFieldLineRipple]
            }], _formFieldControl: [{
                type: ContentChild,
                args: [_MatFormFieldControl]
            }], _prefixChildren: [{
                type: ContentChildren,
                args: [MAT_PREFIX, { descendants: true }]
            }], _suffixChildren: [{
                type: ContentChildren,
                args: [MAT_SUFFIX, { descendants: true }]
            }], _errorChildren: [{
                type: ContentChildren,
                args: [MAT_ERROR, { descendants: true }]
            }], _hintChildren: [{
                type: ContentChildren,
                args: [MatHint, { descendants: true }]
            }], hideRequiredMarker: [{
                type: Input
            }], color: [{
                type: Input
            }], floatLabel: [{
                type: Input
            }], appearance: [{
                type: Input
            }], subscriptSizing: [{
                type: Input
            }], hintLabel: [{
                type: Input
            }] } });
//# sourceMappingURL=data:application/json;base64,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