/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
/**
 * Can be returned by a `Router` guard to instruct the `Router` to redirect rather than continue
 * processing the path of the in-flight navigation. The `redirectTo` indicates _where_ the new
 * navigation should go to and the optional `navigationBehaviorOptions` can provide more information
 * about _how_ to perform the navigation.
 *
 * ```ts
 * const route: Route = {
 *   path: "user/:userId",
 *   component: User,
 *   canActivate: [
 *     () => {
 *       const router = inject(Router);
 *       const authService = inject(AuthenticationService);
 *
 *       if (!authService.isLoggedIn()) {
 *         const loginPath = router.parseUrl("/login");
 *         return new RedirectCommand(loginPath, {
 *           skipLocationChange: "true",
 *         });
 *       }
 *
 *       return true;
 *     },
 *   ],
 * };
 * ```
 * @see [Routing guide](guide/routing/common-router-tasks#preventing-unauthorized-access)
 *
 * @publicApi
 */
export class RedirectCommand {
    constructor(redirectTo, navigationBehaviorOptions) {
        this.redirectTo = redirectTo;
        this.navigationBehaviorOptions = navigationBehaviorOptions;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibW9kZWxzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vLi4vcGFja2FnZXMvcm91dGVyL3NyYy9tb2RlbHMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7OztHQU1HO0FBb0VIOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0E4Qkc7QUFDSCxNQUFNLE9BQU8sZUFBZTtJQUMxQixZQUNXLFVBQW1CLEVBQ25CLHlCQUFxRDtRQURyRCxlQUFVLEdBQVYsVUFBVSxDQUFTO1FBQ25CLDhCQUF5QixHQUF6Qix5QkFBeUIsQ0FBNEI7SUFDN0QsQ0FBQztDQUNMIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IEdvb2dsZSBMTEMgQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiBVc2Ugb2YgdGhpcyBzb3VyY2UgY29kZSBpcyBnb3Zlcm5lZCBieSBhbiBNSVQtc3R5bGUgbGljZW5zZSB0aGF0IGNhbiBiZVxuICogZm91bmQgaW4gdGhlIExJQ0VOU0UgZmlsZSBhdCBodHRwczovL2FuZ3VsYXIuaW8vbGljZW5zZVxuICovXG5cbmltcG9ydCB7XG4gIEVudmlyb25tZW50SW5qZWN0b3IsXG4gIEVudmlyb25tZW50UHJvdmlkZXJzLFxuICBOZ01vZHVsZUZhY3RvcnksXG4gIFByb3ZpZGVyLFxuICBQcm92aWRlclRva2VuLFxuICBUeXBlLFxufSBmcm9tICdAYW5ndWxhci9jb3JlJztcbmltcG9ydCB7T2JzZXJ2YWJsZX0gZnJvbSAncnhqcyc7XG5cbmltcG9ydCB7QWN0aXZhdGVkUm91dGVTbmFwc2hvdCwgUm91dGVyU3RhdGVTbmFwc2hvdH0gZnJvbSAnLi9yb3V0ZXJfc3RhdGUnO1xuaW1wb3J0IHtVcmxTZWdtZW50LCBVcmxTZWdtZW50R3JvdXAsIFVybFRyZWV9IGZyb20gJy4vdXJsX3RyZWUnO1xuXG4vKipcbiAqIEhvdyB0byBoYW5kbGUgYSBuYXZpZ2F0aW9uIHJlcXVlc3QgdG8gdGhlIGN1cnJlbnQgVVJMLiBPbmUgb2Y6XG4gKlxuICogLSBgJ2lnbm9yZSdgIDogIFRoZSByb3V0ZXIgaWdub3JlcyB0aGUgcmVxdWVzdCBpdCBpcyB0aGUgc2FtZSBhcyB0aGUgY3VycmVudCBzdGF0ZS5cbiAqIC0gYCdyZWxvYWQnYCA6IFRoZSByb3V0ZXIgcHJvY2Vzc2VzIHRoZSBVUkwgZXZlbiBpZiBpdCBpcyBub3QgZGlmZmVyZW50IGZyb20gdGhlIGN1cnJlbnQgc3RhdGUuXG4gKiBPbmUgZXhhbXBsZSBvZiB3aGVuIHlvdSBtaWdodCB3YW50IHRoaXMgb3B0aW9uIGlzIGlmIGEgYGNhbk1hdGNoYCBndWFyZCBkZXBlbmRzIG9uXG4gKiBhcHBsaWNhdGlvbiBzdGF0ZSBhbmQgaW5pdGlhbGx5IHJlamVjdHMgbmF2aWdhdGlvbiB0byBhIHJvdXRlLiBBZnRlciBmaXhpbmcgdGhlIHN0YXRlLCB5b3Ugd2FudFxuICogdG8gcmUtbmF2aWdhdGUgdG8gdGhlIHNhbWUgVVJMIHNvIHRoZSByb3V0ZSB3aXRoIHRoZSBgY2FuTWF0Y2hgIGd1YXJkIGNhbiBhY3RpdmF0ZS5cbiAqXG4gKiBOb3RlIHRoYXQgdGhpcyBvbmx5IGNvbmZpZ3VyZXMgd2hldGhlciB0aGUgUm91dGUgcmVwcm9jZXNzZXMgdGhlIFVSTCBhbmQgdHJpZ2dlcnMgcmVsYXRlZFxuICogYWN0aW9uIGFuZCBldmVudHMgbGlrZSByZWRpcmVjdHMsIGd1YXJkcywgYW5kIHJlc29sdmVycy4gQnkgZGVmYXVsdCwgdGhlIHJvdXRlciByZS11c2VzIGFcbiAqIGNvbXBvbmVudCBpbnN0YW5jZSB3aGVuIGl0IHJlLW5hdmlnYXRlcyB0byB0aGUgc2FtZSBjb21wb25lbnQgdHlwZSB3aXRob3V0IHZpc2l0aW5nIGEgZGlmZmVyZW50XG4gKiBjb21wb25lbnQgZmlyc3QuIFRoaXMgYmVoYXZpb3IgaXMgY29uZmlndXJlZCBieSB0aGUgYFJvdXRlUmV1c2VTdHJhdGVneWAuIEluIG9yZGVyIHRvIHJlbG9hZFxuICogcm91dGVkIGNvbXBvbmVudHMgb24gc2FtZSB1cmwgbmF2aWdhdGlvbiwgeW91IG5lZWQgdG8gc2V0IGBvblNhbWVVcmxOYXZpZ2F0aW9uYCB0byBgJ3JlbG9hZCdgXG4gKiBfYW5kXyBwcm92aWRlIGEgYFJvdXRlUmV1c2VTdHJhdGVneWAgd2hpY2ggcmV0dXJucyBgZmFsc2VgIGZvciBgc2hvdWxkUmV1c2VSb3V0ZWAuIEFkZGl0aW9uYWxseSxcbiAqIHJlc29sdmVycyBhbmQgbW9zdCBndWFyZHMgZm9yIHJvdXRlcyBkbyBub3QgcnVuIHVubGVzcyB0aGUgcGF0aCBvciBwYXRoIHBhcmFtcyBjaGFuZ2VkXG4gKiAoY29uZmlndXJlZCBieSBgcnVuR3VhcmRzQW5kUmVzb2x2ZXJzYCkuXG4gKlxuICogQHB1YmxpY0FwaVxuICogQHNlZSB7QGxpbmsgUm91dGVSZXVzZVN0cmF0ZWd5fVxuICogQHNlZSB7QGxpbmsgUnVuR3VhcmRzQW5kUmVzb2x2ZXJzfVxuICogQHNlZSB7QGxpbmsgTmF2aWdhdGlvbkJlaGF2aW9yT3B0aW9uc31cbiAqIEBzZWUge0BsaW5rIFJvdXRlckNvbmZpZ09wdGlvbnN9XG4gKi9cbmV4cG9ydCB0eXBlIE9uU2FtZVVybE5hdmlnYXRpb24gPSAncmVsb2FkJyB8ICdpZ25vcmUnO1xuXG4vKipcbiAqIFRoZSBgSW5qZWN0aW9uVG9rZW5gIGFuZCBgQEluamVjdGFibGVgIGNsYXNzZXMgZm9yIGd1YXJkcyBhbmQgcmVzb2x2ZXJzIGFyZSBkZXByZWNhdGVkIGluIGZhdm9yXG4gKiBvZiBwbGFpbiBKYXZhU2NyaXB0IGZ1bmN0aW9ucyBpbnN0ZWFkLi4gRGVwZW5kZW5jeSBpbmplY3Rpb24gY2FuIHN0aWxsIGJlIGFjaGlldmVkIHVzaW5nIHRoZVxuICogW2BpbmplY3RgXShhcGkvY29yZS9pbmplY3QpIGZ1bmN0aW9uIGZyb20gYEBhbmd1bGFyL2NvcmVgIGFuZCBhbiBpbmplY3RhYmxlIGNsYXNzIGNhbiBiZSB1c2VkIGFzXG4gKiBhIGZ1bmN0aW9uYWwgZ3VhcmQgdXNpbmcgW2BpbmplY3RgXShhcGkvY29yZS9pbmplY3QpOiBgY2FuQWN0aXZhdGU6IFsoKSA9PlxuICogaW5qZWN0KG15R3VhcmQpLmNhbkFjdGl2YXRlKCldYC5cbiAqXG4gKiBAZGVwcmVjYXRlZFxuICogQHNlZSB7QGxpbmsgQ2FuTWF0Y2hGbn1cbiAqIEBzZWUge0BsaW5rIENhbkxvYWRGbn1cbiAqIEBzZWUge0BsaW5rIENhbkFjdGl2YXRlRm59XG4gKiBAc2VlIHtAbGluayBDYW5BY3RpdmF0ZUNoaWxkRm59XG4gKiBAc2VlIHtAbGluayBDYW5EZWFjdGl2YXRlRm59XG4gKiBAc2VlIHtAbGluayBSZXNvbHZlRm59XG4gKiBAc2VlIHtAbGluayBjb3JlL2luamVjdH1cbiAqIEBwdWJsaWNBcGlcbiAqL1xuZXhwb3J0IHR5cGUgRGVwcmVjYXRlZEd1YXJkID0gUHJvdmlkZXJUb2tlbjxhbnk+IHwgYW55O1xuXG4vKipcbiAqIFRoZSBzdXBwb3J0ZWQgdHlwZXMgdGhhdCBjYW4gYmUgcmV0dXJuZWQgZnJvbSBhIGBSb3V0ZXJgIGd1YXJkLlxuICpcbiAqIEBzZWUgW1JvdXRpbmcgZ3VpZGVdKGd1aWRlL3JvdXRpbmcvY29tbW9uLXJvdXRlci10YXNrcyNwcmV2ZW50aW5nLXVuYXV0aG9yaXplZC1hY2Nlc3MpXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCB0eXBlIEd1YXJkUmVzdWx0ID0gYm9vbGVhbiB8IFVybFRyZWUgfCBSZWRpcmVjdENvbW1hbmQ7XG5cbi8qKlxuICogQ2FuIGJlIHJldHVybmVkIGJ5IGEgYFJvdXRlcmAgZ3VhcmQgdG8gaW5zdHJ1Y3QgdGhlIGBSb3V0ZXJgIHRvIHJlZGlyZWN0IHJhdGhlciB0aGFuIGNvbnRpbnVlXG4gKiBwcm9jZXNzaW5nIHRoZSBwYXRoIG9mIHRoZSBpbi1mbGlnaHQgbmF2aWdhdGlvbi4gVGhlIGByZWRpcmVjdFRvYCBpbmRpY2F0ZXMgX3doZXJlXyB0aGUgbmV3XG4gKiBuYXZpZ2F0aW9uIHNob3VsZCBnbyB0byBhbmQgdGhlIG9wdGlvbmFsIGBuYXZpZ2F0aW9uQmVoYXZpb3JPcHRpb25zYCBjYW4gcHJvdmlkZSBtb3JlIGluZm9ybWF0aW9uXG4gKiBhYm91dCBfaG93XyB0byBwZXJmb3JtIHRoZSBuYXZpZ2F0aW9uLlxuICpcbiAqIGBgYHRzXG4gKiBjb25zdCByb3V0ZTogUm91dGUgPSB7XG4gKiAgIHBhdGg6IFwidXNlci86dXNlcklkXCIsXG4gKiAgIGNvbXBvbmVudDogVXNlcixcbiAqICAgY2FuQWN0aXZhdGU6IFtcbiAqICAgICAoKSA9PiB7XG4gKiAgICAgICBjb25zdCByb3V0ZXIgPSBpbmplY3QoUm91dGVyKTtcbiAqICAgICAgIGNvbnN0IGF1dGhTZXJ2aWNlID0gaW5qZWN0KEF1dGhlbnRpY2F0aW9uU2VydmljZSk7XG4gKlxuICogICAgICAgaWYgKCFhdXRoU2VydmljZS5pc0xvZ2dlZEluKCkpIHtcbiAqICAgICAgICAgY29uc3QgbG9naW5QYXRoID0gcm91dGVyLnBhcnNlVXJsKFwiL2xvZ2luXCIpO1xuICogICAgICAgICByZXR1cm4gbmV3IFJlZGlyZWN0Q29tbWFuZChsb2dpblBhdGgsIHtcbiAqICAgICAgICAgICBza2lwTG9jYXRpb25DaGFuZ2U6IFwidHJ1ZVwiLFxuICogICAgICAgICB9KTtcbiAqICAgICAgIH1cbiAqXG4gKiAgICAgICByZXR1cm4gdHJ1ZTtcbiAqICAgICB9LFxuICogICBdLFxuICogfTtcbiAqIGBgYFxuICogQHNlZSBbUm91dGluZyBndWlkZV0oZ3VpZGUvcm91dGluZy9jb21tb24tcm91dGVyLXRhc2tzI3ByZXZlbnRpbmctdW5hdXRob3JpemVkLWFjY2VzcylcbiAqXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBjbGFzcyBSZWRpcmVjdENvbW1hbmQge1xuICBjb25zdHJ1Y3RvcihcbiAgICByZWFkb25seSByZWRpcmVjdFRvOiBVcmxUcmVlLFxuICAgIHJlYWRvbmx5IG5hdmlnYXRpb25CZWhhdmlvck9wdGlvbnM/OiBOYXZpZ2F0aW9uQmVoYXZpb3JPcHRpb25zLFxuICApIHt9XG59XG5cbi8qKlxuICogVHlwZSB1c2VkIHRvIHJlcHJlc2VudCBhIHZhbHVlIHdoaWNoIG1heSBiZSBzeW5jaHJvbm91cyBvciBhc3luYy5cbiAqXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCB0eXBlIE1heWJlQXN5bmM8VD4gPSBUIHwgT2JzZXJ2YWJsZTxUPiB8IFByb21pc2U8VD47XG5cbi8qKlxuICogUmVwcmVzZW50cyBhIHJvdXRlIGNvbmZpZ3VyYXRpb24gZm9yIHRoZSBSb3V0ZXIgc2VydmljZS5cbiAqIEFuIGFycmF5IG9mIGBSb3V0ZWAgb2JqZWN0cywgdXNlZCBpbiBgUm91dGVyLmNvbmZpZ2AgYW5kIGZvciBuZXN0ZWQgcm91dGUgY29uZmlndXJhdGlvbnNcbiAqIGluIGBSb3V0ZS5jaGlsZHJlbmAuXG4gKlxuICogQHNlZSB7QGxpbmsgUm91dGV9XG4gKiBAc2VlIHtAbGluayBSb3V0ZXJ9XG4gKiBAc2VlIFtSb3V0ZXIgY29uZmlndXJhdGlvbiBndWlkZV0oZ3VpZGUvcm91dGluZy9yb3V0ZXItcmVmZXJlbmNlI2NvbmZpZ3VyYXRpb24pXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCB0eXBlIFJvdXRlcyA9IFJvdXRlW107XG5cbi8qKlxuICogUmVwcmVzZW50cyB0aGUgcmVzdWx0IG9mIG1hdGNoaW5nIFVSTHMgd2l0aCBhIGN1c3RvbSBtYXRjaGluZyBmdW5jdGlvbi5cbiAqXG4gKiAqIGBjb25zdW1lZGAgaXMgYW4gYXJyYXkgb2YgdGhlIGNvbnN1bWVkIFVSTCBzZWdtZW50cy5cbiAqICogYHBvc1BhcmFtc2AgaXMgYSBtYXAgb2YgcG9zaXRpb25hbCBwYXJhbWV0ZXJzLlxuICpcbiAqIEBzZWUge0BsaW5rIFVybE1hdGNoZXJ9XG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCB0eXBlIFVybE1hdGNoUmVzdWx0ID0ge1xuICBjb25zdW1lZDogVXJsU2VnbWVudFtdO1xuICBwb3NQYXJhbXM/OiB7W25hbWU6IHN0cmluZ106IFVybFNlZ21lbnR9O1xufTtcblxuLyoqXG4gKiBBIGZ1bmN0aW9uIGZvciBtYXRjaGluZyBhIHJvdXRlIGFnYWluc3QgVVJMcy4gSW1wbGVtZW50IGEgY3VzdG9tIFVSTCBtYXRjaGVyXG4gKiBmb3IgYFJvdXRlLm1hdGNoZXJgIHdoZW4gYSBjb21iaW5hdGlvbiBvZiBgcGF0aGAgYW5kIGBwYXRoTWF0Y2hgXG4gKiBpcyBub3QgZXhwcmVzc2l2ZSBlbm91Z2guIENhbm5vdCBiZSB1c2VkIHRvZ2V0aGVyIHdpdGggYHBhdGhgIGFuZCBgcGF0aE1hdGNoYC5cbiAqXG4gKiBUaGUgZnVuY3Rpb24gdGFrZXMgdGhlIGZvbGxvd2luZyBhcmd1bWVudHMgYW5kIHJldHVybnMgYSBgVXJsTWF0Y2hSZXN1bHRgIG9iamVjdC5cbiAqICogKnNlZ21lbnRzKiA6IEFuIGFycmF5IG9mIFVSTCBzZWdtZW50cy5cbiAqICogKmdyb3VwKiA6IEEgc2VnbWVudCBncm91cC5cbiAqICogKnJvdXRlKiA6IFRoZSByb3V0ZSB0byBtYXRjaCBhZ2FpbnN0LlxuICpcbiAqIFRoZSBmb2xsb3dpbmcgZXhhbXBsZSBpbXBsZW1lbnRhdGlvbiBtYXRjaGVzIEhUTUwgZmlsZXMuXG4gKlxuICogYGBgXG4gKiBleHBvcnQgZnVuY3Rpb24gaHRtbEZpbGVzKHVybDogVXJsU2VnbWVudFtdKSB7XG4gKiAgIHJldHVybiB1cmwubGVuZ3RoID09PSAxICYmIHVybFswXS5wYXRoLmVuZHNXaXRoKCcuaHRtbCcpID8gKHtjb25zdW1lZDogdXJsfSkgOiBudWxsO1xuICogfVxuICpcbiAqIGV4cG9ydCBjb25zdCByb3V0ZXMgPSBbeyBtYXRjaGVyOiBodG1sRmlsZXMsIGNvbXBvbmVudDogQW55Q29tcG9uZW50IH1dO1xuICogYGBgXG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgdHlwZSBVcmxNYXRjaGVyID0gKFxuICBzZWdtZW50czogVXJsU2VnbWVudFtdLFxuICBncm91cDogVXJsU2VnbWVudEdyb3VwLFxuICByb3V0ZTogUm91dGUsXG4pID0+IFVybE1hdGNoUmVzdWx0IHwgbnVsbDtcblxuLyoqXG4gKlxuICogUmVwcmVzZW50cyBzdGF0aWMgZGF0YSBhc3NvY2lhdGVkIHdpdGggYSBwYXJ0aWN1bGFyIHJvdXRlLlxuICpcbiAqIEBzZWUge0BsaW5rIFJvdXRlI2RhdGF9XG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgdHlwZSBEYXRhID0ge1xuICBba2V5OiBzdHJpbmcgfCBzeW1ib2xdOiBhbnk7XG59O1xuXG4vKipcbiAqXG4gKiBSZXByZXNlbnRzIHRoZSByZXNvbHZlZCBkYXRhIGFzc29jaWF0ZWQgd2l0aCBhIHBhcnRpY3VsYXIgcm91dGUuXG4gKlxuICogUmV0dXJuaW5nIGEgYFJlZGlyZWN0Q29tbWFuZGAgZGlyZWN0cyB0aGUgcm91dGVyIHRvIGNhbmNlbCB0aGUgY3VycmVudCBuYXZpZ2F0aW9uIGFuZCByZWRpcmVjdCB0b1xuICogdGhlIGxvY2F0aW9uIHByb3ZpZGVkIGluIHRoZSBgUmVkaXJlY3RDb21tYW5kYC4gTm90ZSB0aGF0IHRoZXJlIGFyZSBubyBvcmRlcmluZyBndWFyYW50ZWVzIHdoZW5cbiAqIHJlc29sdmVycyBleGVjdXRlLiBJZiBtdWx0aXBsZSByZXNvbHZlcnMgd291bGQgcmV0dXJuIGEgYFJlZGlyZWN0Q29tbWFuZGAsIG9ubHkgdGhlIGZpcnN0IG9uZVxuICogcmV0dXJuZWQgd2lsbCBiZSB1c2VkLlxuICpcbiAqIEBzZWUge0BsaW5rIFJvdXRlI3Jlc29sdmV9XG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgdHlwZSBSZXNvbHZlRGF0YSA9IHtcbiAgW2tleTogc3RyaW5nIHwgc3ltYm9sXTogUmVzb2x2ZUZuPHVua25vd24+IHwgRGVwcmVjYXRlZEd1YXJkO1xufTtcblxuLyoqXG4gKiBBbiBFUyBNb2R1bGUgb2JqZWN0IHdpdGggYSBkZWZhdWx0IGV4cG9ydCBvZiB0aGUgZ2l2ZW4gdHlwZS5cbiAqXG4gKiBAc2VlIHtAbGluayBSb3V0ZSNsb2FkQ29tcG9uZW50fVxuICogQHNlZSB7QGxpbmsgTG9hZENoaWxkcmVuQ2FsbGJhY2t9XG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgaW50ZXJmYWNlIERlZmF1bHRFeHBvcnQ8VD4ge1xuICAvKipcbiAgICogRGVmYXVsdCBleHBvcnRzIGFyZSBib3VuZCB1bmRlciB0aGUgbmFtZSBgXCJkZWZhdWx0XCJgLCBwZXIgdGhlIEVTIE1vZHVsZSBzcGVjOlxuICAgKiBodHRwczovL3RjMzkuZXMvZWNtYTI2Mi8jdGFibGUtZXhwb3J0LWZvcm1zLW1hcHBpbmctdG8tZXhwb3J0ZW50cnktcmVjb3Jkc1xuICAgKi9cbiAgZGVmYXVsdDogVDtcbn1cblxuLyoqXG4gKlxuICogQSBmdW5jdGlvbiB0aGF0IGlzIGNhbGxlZCB0byByZXNvbHZlIGEgY29sbGVjdGlvbiBvZiBsYXp5LWxvYWRlZCByb3V0ZXMuXG4gKiBNdXN0IGJlIGFuIGFycm93IGZ1bmN0aW9uIG9mIHRoZSBmb2xsb3dpbmcgZm9ybTpcbiAqIGAoKSA9PiBpbXBvcnQoJy4uLicpLnRoZW4obW9kID0+IG1vZC5NT0RVTEUpYFxuICogb3JcbiAqIGAoKSA9PiBpbXBvcnQoJy4uLicpLnRoZW4obW9kID0+IG1vZC5ST1VURVMpYFxuICpcbiAqIEZvciBleGFtcGxlOlxuICpcbiAqIGBgYFxuICogW3tcbiAqICAgcGF0aDogJ2xhenknLFxuICogICBsb2FkQ2hpbGRyZW46ICgpID0+IGltcG9ydCgnLi9sYXp5LXJvdXRlL2xhenkubW9kdWxlJykudGhlbihtb2QgPT4gbW9kLkxhenlNb2R1bGUpLFxuICogfV07XG4gKiBgYGBcbiAqIG9yXG4gKiBgYGBcbiAqIFt7XG4gKiAgIHBhdGg6ICdsYXp5JyxcbiAqICAgbG9hZENoaWxkcmVuOiAoKSA9PiBpbXBvcnQoJy4vbGF6eS1yb3V0ZS9sYXp5LnJvdXRlcycpLnRoZW4obW9kID0+IG1vZC5ST1VURVMpLFxuICogfV07XG4gKiBgYGBcbiAqXG4gKiBJZiB0aGUgbGF6eS1sb2FkZWQgcm91dGVzIGFyZSBleHBvcnRlZCB2aWEgYSBgZGVmYXVsdGAgZXhwb3J0LCB0aGUgYC50aGVuYCBjYW4gYmUgb21pdHRlZDpcbiAqIGBgYFxuICogW3tcbiAqICAgcGF0aDogJ2xhenknLFxuICogICBsb2FkQ2hpbGRyZW46ICgpID0+IGltcG9ydCgnLi9sYXp5LXJvdXRlL2xhenkucm91dGVzJyksXG4gKiB9XTtcbiAqIGBgYFxuICpcbiAqIEBzZWUge0BsaW5rIFJvdXRlI2xvYWRDaGlsZHJlbn1cbiAqIEBwdWJsaWNBcGlcbiAqL1xuZXhwb3J0IHR5cGUgTG9hZENoaWxkcmVuQ2FsbGJhY2sgPSAoKSA9PlxuICB8IFR5cGU8YW55PlxuICB8IE5nTW9kdWxlRmFjdG9yeTxhbnk+XG4gIHwgUm91dGVzXG4gIHwgT2JzZXJ2YWJsZTxUeXBlPGFueT4gfCBSb3V0ZXMgfCBEZWZhdWx0RXhwb3J0PFR5cGU8YW55Pj4gfCBEZWZhdWx0RXhwb3J0PFJvdXRlcz4+XG4gIHwgUHJvbWlzZTxcbiAgICAgIE5nTW9kdWxlRmFjdG9yeTxhbnk+IHwgVHlwZTxhbnk+IHwgUm91dGVzIHwgRGVmYXVsdEV4cG9ydDxUeXBlPGFueT4+IHwgRGVmYXVsdEV4cG9ydDxSb3V0ZXM+XG4gICAgPjtcblxuLyoqXG4gKlxuICogQSBmdW5jdGlvbiB0aGF0IHJldHVybnMgYSBzZXQgb2Ygcm91dGVzIHRvIGxvYWQuXG4gKlxuICogQHNlZSB7QGxpbmsgTG9hZENoaWxkcmVuQ2FsbGJhY2t9XG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCB0eXBlIExvYWRDaGlsZHJlbiA9IExvYWRDaGlsZHJlbkNhbGxiYWNrO1xuXG4vKipcbiAqXG4gKiBIb3cgdG8gaGFuZGxlIHF1ZXJ5IHBhcmFtZXRlcnMgaW4gYSByb3V0ZXIgbGluay5cbiAqIE9uZSBvZjpcbiAqIC0gYFwibWVyZ2VcImAgOiBNZXJnZSBuZXcgcGFyYW1ldGVycyB3aXRoIGN1cnJlbnQgcGFyYW1ldGVycy5cbiAqIC0gYFwicHJlc2VydmVcImAgOiBQcmVzZXJ2ZSBjdXJyZW50IHBhcmFtZXRlcnMuXG4gKiAtIGBcIlwiYCA6IFJlcGxhY2UgY3VycmVudCBwYXJhbWV0ZXJzIHdpdGggbmV3IHBhcmFtZXRlcnMuIFRoaXMgaXMgdGhlIGRlZmF1bHQgYmVoYXZpb3IuXG4gKlxuICogQHNlZSB7QGxpbmsgVXJsQ3JlYXRpb25PcHRpb25zI3F1ZXJ5UGFyYW1zSGFuZGxpbmd9XG4gKiBAc2VlIHtAbGluayBSb3V0ZXJMaW5rfVxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgdHlwZSBRdWVyeVBhcmFtc0hhbmRsaW5nID0gJ21lcmdlJyB8ICdwcmVzZXJ2ZScgfCAnJztcblxuLyoqXG4gKiBUaGUgdHlwZSBmb3IgdGhlIGZ1bmN0aW9uIHRoYXQgY2FuIGJlIHVzZWQgdG8gaGFuZGxlIHJlZGlyZWN0cyB3aGVuIHRoZSBwYXRoIG1hdGNoZXMgYSBgUm91dGVgIGNvbmZpZy5cbiAqXG4gKiBUaGUgYFJlZGlyZWN0RnVuY3Rpb25gIGRvZXMgaGF2ZSBhY2Nlc3MgdG8gdGhlIGZ1bGxcbiAqIGBBY3RpdmF0ZWRSb3V0ZVNuYXBzaG90YCBpbnRlcmZhY2UuIFNvbWUgZGF0YSBhcmUgbm90IGFjY3VyYXRlbHkga25vd25cbiAqIGF0IHRoZSByb3V0ZSBtYXRjaGluZyBwaGFzZS4gRm9yIGV4YW1wbGUsIHJlc29sdmVycyBhcmUgbm90IHJ1biB1bnRpbFxuICogbGF0ZXIsIHNvIGFueSByZXNvbHZlZCB0aXRsZSB3b3VsZCBub3QgYmUgcG9wdWxhdGVkLiBUaGUgc2FtZSBnb2VzIGZvciBsYXp5XG4gKiBsb2FkZWQgY29tcG9uZW50cy4gVGhpcyBpcyBhbHNvIHRydWUgZm9yIGFsbCB0aGUgc25hcHNob3RzIHVwIHRvIHRoZVxuICogcm9vdCwgc28gcHJvcGVydGllcyB0aGF0IGluY2x1ZGUgcGFyZW50cyAocm9vdCwgcGFyZW50LCBwYXRoRnJvbVJvb3QpXG4gKiBhcmUgYWxzbyBleGNsdWRlZC4gQW5kIG5hdHVyYWxseSwgdGhlIGZ1bGwgcm91dGUgbWF0Y2hpbmcgaGFzbid0IHlldFxuICogaGFwcGVuZWQgc28gZmlyc3RDaGlsZCBhbmQgY2hpbGRyZW4gYXJlIG5vdCBhdmFpbGFibGUgZWl0aGVyLlxuICpcbiAqIEBzZWUge0BsaW5rIFJvdXRlI3JlZGlyZWN0VG99XG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCB0eXBlIFJlZGlyZWN0RnVuY3Rpb24gPSAoXG4gIHJlZGlyZWN0RGF0YTogUGljazxcbiAgICBBY3RpdmF0ZWRSb3V0ZVNuYXBzaG90LFxuICAgICdyb3V0ZUNvbmZpZycgfCAndXJsJyB8ICdwYXJhbXMnIHwgJ3F1ZXJ5UGFyYW1zJyB8ICdmcmFnbWVudCcgfCAnZGF0YScgfCAnb3V0bGV0JyB8ICd0aXRsZSdcbiAgPixcbikgPT4gc3RyaW5nIHwgVXJsVHJlZTtcblxuLyoqXG4gKiBBIHBvbGljeSBmb3Igd2hlbiB0byBydW4gZ3VhcmRzIGFuZCByZXNvbHZlcnMgb24gYSByb3V0ZS5cbiAqXG4gKiBHdWFyZHMgYW5kL29yIHJlc29sdmVycyB3aWxsIGFsd2F5cyBydW4gd2hlbiBhIHJvdXRlIGlzIGFjdGl2YXRlZCBvciBkZWFjdGl2YXRlZC4gV2hlbiBhIHJvdXRlIGlzXG4gKiB1bmNoYW5nZWQsIHRoZSBkZWZhdWx0IGJlaGF2aW9yIGlzIHRoZSBzYW1lIGFzIGBwYXJhbXNDaGFuZ2VgLlxuICpcbiAqIGBwYXJhbXNDaGFuZ2VgIDogUmVydW4gdGhlIGd1YXJkcyBhbmQgcmVzb2x2ZXJzIHdoZW4gcGF0aCBvclxuICogcGF0aCBwYXJhbSBjaGFuZ2VzLiBUaGlzIGRvZXMgbm90IGluY2x1ZGUgcXVlcnkgcGFyYW1ldGVycy4gVGhpcyBvcHRpb24gaXMgdGhlIGRlZmF1bHQuXG4gKiAtIGBhbHdheXNgIDogUnVuIG9uIGV2ZXJ5IGV4ZWN1dGlvbi5cbiAqIC0gYHBhdGhQYXJhbXNDaGFuZ2VgIDogUmVydW4gZ3VhcmRzIGFuZCByZXNvbHZlcnMgd2hlbiB0aGUgcGF0aCBwYXJhbXNcbiAqIGNoYW5nZS4gVGhpcyBkb2VzIG5vdCBjb21wYXJlIG1hdHJpeCBvciBxdWVyeSBwYXJhbWV0ZXJzLlxuICogLSBgcGFyYW1zT3JRdWVyeVBhcmFtc0NoYW5nZWAgOiBSdW4gd2hlbiBwYXRoLCBtYXRyaXgsIG9yIHF1ZXJ5IHBhcmFtZXRlcnMgY2hhbmdlLlxuICogLSBgcGF0aFBhcmFtc09yUXVlcnlQYXJhbXNDaGFuZ2VgIDogUmVydW4gZ3VhcmRzIGFuZCByZXNvbHZlcnMgd2hlbiB0aGUgcGF0aCBwYXJhbXNcbiAqIGNoYW5nZSBvciBxdWVyeSBwYXJhbXMgaGF2ZSBjaGFuZ2VkLiBUaGlzIGRvZXMgbm90IGluY2x1ZGUgbWF0cml4IHBhcmFtZXRlcnMuXG4gKlxuICogQHNlZSB7QGxpbmsgUm91dGUjcnVuR3VhcmRzQW5kUmVzb2x2ZXJzfVxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgdHlwZSBSdW5HdWFyZHNBbmRSZXNvbHZlcnMgPVxuICB8ICdwYXRoUGFyYW1zQ2hhbmdlJ1xuICB8ICdwYXRoUGFyYW1zT3JRdWVyeVBhcmFtc0NoYW5nZSdcbiAgfCAncGFyYW1zQ2hhbmdlJ1xuICB8ICdwYXJhbXNPclF1ZXJ5UGFyYW1zQ2hhbmdlJ1xuICB8ICdhbHdheXMnXG4gIHwgKChmcm9tOiBBY3RpdmF0ZWRSb3V0ZVNuYXBzaG90LCB0bzogQWN0aXZhdGVkUm91dGVTbmFwc2hvdCkgPT4gYm9vbGVhbik7XG5cbi8qKlxuICogQSBjb25maWd1cmF0aW9uIG9iamVjdCB0aGF0IGRlZmluZXMgYSBzaW5nbGUgcm91dGUuXG4gKiBBIHNldCBvZiByb3V0ZXMgYXJlIGNvbGxlY3RlZCBpbiBhIGBSb3V0ZXNgIGFycmF5IHRvIGRlZmluZSBhIGBSb3V0ZXJgIGNvbmZpZ3VyYXRpb24uXG4gKiBUaGUgcm91dGVyIGF0dGVtcHRzIHRvIG1hdGNoIHNlZ21lbnRzIG9mIGEgZ2l2ZW4gVVJMIGFnYWluc3QgZWFjaCByb3V0ZSxcbiAqIHVzaW5nIHRoZSBjb25maWd1cmF0aW9uIG9wdGlvbnMgZGVmaW5lZCBpbiB0aGlzIG9iamVjdC5cbiAqXG4gKiBTdXBwb3J0cyBzdGF0aWMsIHBhcmFtZXRlcml6ZWQsIHJlZGlyZWN0LCBhbmQgd2lsZGNhcmQgcm91dGVzLCBhcyB3ZWxsIGFzXG4gKiBjdXN0b20gcm91dGUgZGF0YSBhbmQgcmVzb2x2ZSBtZXRob2RzLlxuICpcbiAqIEZvciBkZXRhaWxlZCB1c2FnZSBpbmZvcm1hdGlvbiwgc2VlIHRoZSBbUm91dGluZyBHdWlkZV0oZ3VpZGUvcm91dGluZy9jb21tb24tcm91dGVyLXRhc2tzKS5cbiAqXG4gKiBAdXNhZ2VOb3Rlc1xuICpcbiAqICMjIyBTaW1wbGUgQ29uZmlndXJhdGlvblxuICpcbiAqIFRoZSBmb2xsb3dpbmcgcm91dGUgc3BlY2lmaWVzIHRoYXQgd2hlbiBuYXZpZ2F0aW5nIHRvLCBmb3IgZXhhbXBsZSxcbiAqIGAvdGVhbS8xMS91c2VyL2JvYmAsIHRoZSByb3V0ZXIgY3JlYXRlcyB0aGUgJ1RlYW0nIGNvbXBvbmVudFxuICogd2l0aCB0aGUgJ1VzZXInIGNoaWxkIGNvbXBvbmVudCBpbiBpdC5cbiAqXG4gKiBgYGBcbiAqIFt7XG4gKiAgIHBhdGg6ICd0ZWFtLzppZCcsXG4gKiAgY29tcG9uZW50OiBUZWFtLFxuICogICBjaGlsZHJlbjogW3tcbiAqICAgICBwYXRoOiAndXNlci86bmFtZScsXG4gKiAgICAgY29tcG9uZW50OiBVc2VyXG4gKiAgIH1dXG4gKiB9XVxuICogYGBgXG4gKlxuICogIyMjIE11bHRpcGxlIE91dGxldHNcbiAqXG4gKiBUaGUgZm9sbG93aW5nIHJvdXRlIGNyZWF0ZXMgc2libGluZyBjb21wb25lbnRzIHdpdGggbXVsdGlwbGUgb3V0bGV0cy5cbiAqIFdoZW4gbmF2aWdhdGluZyB0byBgL3RlYW0vMTEoYXV4OmNoYXQvamltKWAsIHRoZSByb3V0ZXIgY3JlYXRlcyB0aGUgJ1RlYW0nIGNvbXBvbmVudCBuZXh0IHRvXG4gKiB0aGUgJ0NoYXQnIGNvbXBvbmVudC4gVGhlICdDaGF0JyBjb21wb25lbnQgaXMgcGxhY2VkIGludG8gdGhlICdhdXgnIG91dGxldC5cbiAqXG4gKiBgYGBcbiAqIFt7XG4gKiAgIHBhdGg6ICd0ZWFtLzppZCcsXG4gKiAgIGNvbXBvbmVudDogVGVhbVxuICogfSwge1xuICogICBwYXRoOiAnY2hhdC86dXNlcicsXG4gKiAgIGNvbXBvbmVudDogQ2hhdFxuICogICBvdXRsZXQ6ICdhdXgnXG4gKiB9XVxuICogYGBgXG4gKlxuICogIyMjIFdpbGQgQ2FyZHNcbiAqXG4gKiBUaGUgZm9sbG93aW5nIHJvdXRlIHVzZXMgd2lsZC1jYXJkIG5vdGF0aW9uIHRvIHNwZWNpZnkgYSBjb21wb25lbnRcbiAqIHRoYXQgaXMgYWx3YXlzIGluc3RhbnRpYXRlZCByZWdhcmRsZXNzIG9mIHdoZXJlIHlvdSBuYXZpZ2F0ZSB0by5cbiAqXG4gKiBgYGBcbiAqIFt7XG4gKiAgIHBhdGg6ICcqKicsXG4gKiAgIGNvbXBvbmVudDogV2lsZGNhcmRDb21wb25lbnRcbiAqIH1dXG4gKiBgYGBcbiAqXG4gKiAjIyMgUmVkaXJlY3RzXG4gKlxuICogVGhlIGZvbGxvd2luZyByb3V0ZSB1c2VzIHRoZSBgcmVkaXJlY3RUb2AgcHJvcGVydHkgdG8gaWdub3JlIGEgc2VnbWVudCBvZlxuICogYSBnaXZlbiBVUkwgd2hlbiBsb29raW5nIGZvciBhIGNoaWxkIHBhdGguXG4gKlxuICogV2hlbiBuYXZpZ2F0aW5nIHRvICcvdGVhbS8xMS9sZWdhY3kvdXNlci9qaW0nLCB0aGUgcm91dGVyIGNoYW5nZXMgdGhlIFVSTCBzZWdtZW50XG4gKiAnL3RlYW0vMTEvbGVnYWN5L3VzZXIvamltJyB0byAnL3RlYW0vMTEvdXNlci9qaW0nLCBhbmQgdGhlbiBpbnN0YW50aWF0ZXNcbiAqIHRoZSBUZWFtIGNvbXBvbmVudCB3aXRoIHRoZSBVc2VyIGNoaWxkIGNvbXBvbmVudCBpbiBpdC5cbiAqXG4gKiBgYGBcbiAqIFt7XG4gKiAgIHBhdGg6ICd0ZWFtLzppZCcsXG4gKiAgIGNvbXBvbmVudDogVGVhbSxcbiAqICAgY2hpbGRyZW46IFt7XG4gKiAgICAgcGF0aDogJ2xlZ2FjeS91c2VyLzpuYW1lJyxcbiAqICAgICByZWRpcmVjdFRvOiAndXNlci86bmFtZSdcbiAqICAgfSwge1xuICogICAgIHBhdGg6ICd1c2VyLzpuYW1lJyxcbiAqICAgICBjb21wb25lbnQ6IFVzZXJcbiAqICAgfV1cbiAqIH1dXG4gKiBgYGBcbiAqXG4gKiBUaGUgcmVkaXJlY3QgcGF0aCBjYW4gYmUgcmVsYXRpdmUsIGFzIHNob3duIGluIHRoaXMgZXhhbXBsZSwgb3IgYWJzb2x1dGUuXG4gKiBJZiB3ZSBjaGFuZ2UgdGhlIGByZWRpcmVjdFRvYCB2YWx1ZSBpbiB0aGUgZXhhbXBsZSB0byB0aGUgYWJzb2x1dGUgVVJMIHNlZ21lbnQgJy91c2VyLzpuYW1lJyxcbiAqIHRoZSByZXN1bHQgVVJMIGlzIGFsc28gYWJzb2x1dGUsICcvdXNlci9qaW0nLlxuXG4gKiAjIyMgRW1wdHkgUGF0aFxuICpcbiAqIEVtcHR5LXBhdGggcm91dGUgY29uZmlndXJhdGlvbnMgY2FuIGJlIHVzZWQgdG8gaW5zdGFudGlhdGUgY29tcG9uZW50cyB0aGF0IGRvIG5vdCAnY29uc3VtZSdcbiAqIGFueSBVUkwgc2VnbWVudHMuXG4gKlxuICogSW4gdGhlIGZvbGxvd2luZyBjb25maWd1cmF0aW9uLCB3aGVuIG5hdmlnYXRpbmcgdG9cbiAqIGAvdGVhbS8xMWAsIHRoZSByb3V0ZXIgaW5zdGFudGlhdGVzIHRoZSAnQWxsVXNlcnMnIGNvbXBvbmVudC5cbiAqXG4gKiBgYGBcbiAqIFt7XG4gKiAgIHBhdGg6ICd0ZWFtLzppZCcsXG4gKiAgIGNvbXBvbmVudDogVGVhbSxcbiAqICAgY2hpbGRyZW46IFt7XG4gKiAgICAgcGF0aDogJycsXG4gKiAgICAgY29tcG9uZW50OiBBbGxVc2Vyc1xuICogICB9LCB7XG4gKiAgICAgcGF0aDogJ3VzZXIvOm5hbWUnLFxuICogICAgIGNvbXBvbmVudDogVXNlclxuICogICB9XVxuICogfV1cbiAqIGBgYFxuICpcbiAqIEVtcHR5LXBhdGggcm91dGVzIGNhbiBoYXZlIGNoaWxkcmVuLiBJbiB0aGUgZm9sbG93aW5nIGV4YW1wbGUsIHdoZW4gbmF2aWdhdGluZ1xuICogdG8gYC90ZWFtLzExL3VzZXIvamltYCwgdGhlIHJvdXRlciBpbnN0YW50aWF0ZXMgdGhlIHdyYXBwZXIgY29tcG9uZW50IHdpdGhcbiAqIHRoZSB1c2VyIGNvbXBvbmVudCBpbiBpdC5cbiAqXG4gKiBOb3RlIHRoYXQgYW4gZW1wdHkgcGF0aCByb3V0ZSBpbmhlcml0cyBpdHMgcGFyZW50J3MgcGFyYW1ldGVycyBhbmQgZGF0YS5cbiAqXG4gKiBgYGBcbiAqIFt7XG4gKiAgIHBhdGg6ICd0ZWFtLzppZCcsXG4gKiAgIGNvbXBvbmVudDogVGVhbSxcbiAqICAgY2hpbGRyZW46IFt7XG4gKiAgICAgcGF0aDogJycsXG4gKiAgICAgY29tcG9uZW50OiBXcmFwcGVyQ21wLFxuICogICAgIGNoaWxkcmVuOiBbe1xuICogICAgICAgcGF0aDogJ3VzZXIvOm5hbWUnLFxuICogICAgICAgY29tcG9uZW50OiBVc2VyXG4gKiAgICAgfV1cbiAqICAgfV1cbiAqIH1dXG4gKiBgYGBcbiAqXG4gKiAjIyMgTWF0Y2hpbmcgU3RyYXRlZ3lcbiAqXG4gKiBUaGUgZGVmYXVsdCBwYXRoLW1hdGNoIHN0cmF0ZWd5IGlzICdwcmVmaXgnLCB3aGljaCBtZWFucyB0aGF0IHRoZSByb3V0ZXJcbiAqIGNoZWNrcyBVUkwgZWxlbWVudHMgZnJvbSB0aGUgbGVmdCB0byBzZWUgaWYgdGhlIFVSTCBtYXRjaGVzIGEgc3BlY2lmaWVkIHBhdGguXG4gKiBGb3IgZXhhbXBsZSwgJy90ZWFtLzExL3VzZXInIG1hdGNoZXMgJ3RlYW0vOmlkJy5cbiAqXG4gKiBgYGBcbiAqIFt7XG4gKiAgIHBhdGg6ICcnLFxuICogICBwYXRoTWF0Y2g6ICdwcmVmaXgnLCAvL2RlZmF1bHRcbiAqICAgcmVkaXJlY3RUbzogJ21haW4nXG4gKiB9LCB7XG4gKiAgIHBhdGg6ICdtYWluJyxcbiAqICAgY29tcG9uZW50OiBNYWluXG4gKiB9XVxuICogYGBgXG4gKlxuICogWW91IGNhbiBzcGVjaWZ5IHRoZSBwYXRoLW1hdGNoIHN0cmF0ZWd5ICdmdWxsJyB0byBtYWtlIHN1cmUgdGhhdCB0aGUgcGF0aFxuICogY292ZXJzIHRoZSB3aG9sZSB1bmNvbnN1bWVkIFVSTC4gSXQgaXMgaW1wb3J0YW50IHRvIGRvIHRoaXMgd2hlbiByZWRpcmVjdGluZ1xuICogZW1wdHktcGF0aCByb3V0ZXMuIE90aGVyd2lzZSwgYmVjYXVzZSBhbiBlbXB0eSBwYXRoIGlzIGEgcHJlZml4IG9mIGFueSBVUkwsXG4gKiB0aGUgcm91dGVyIHdvdWxkIGFwcGx5IHRoZSByZWRpcmVjdCBldmVuIHdoZW4gbmF2aWdhdGluZyB0byB0aGUgcmVkaXJlY3QgZGVzdGluYXRpb24sXG4gKiBjcmVhdGluZyBhbiBlbmRsZXNzIGxvb3AuXG4gKlxuICogSW4gdGhlIGZvbGxvd2luZyBleGFtcGxlLCBzdXBwbHlpbmcgdGhlICdmdWxsJyBgcGF0aE1hdGNoYCBzdHJhdGVneSBlbnN1cmVzXG4gKiB0aGF0IHRoZSByb3V0ZXIgYXBwbGllcyB0aGUgcmVkaXJlY3QgaWYgYW5kIG9ubHkgaWYgbmF2aWdhdGluZyB0byAnLycuXG4gKlxuICogYGBgXG4gKiBbe1xuICogICBwYXRoOiAnJyxcbiAqICAgcGF0aE1hdGNoOiAnZnVsbCcsXG4gKiAgIHJlZGlyZWN0VG86ICdtYWluJ1xuICogfSwge1xuICogICBwYXRoOiAnbWFpbicsXG4gKiAgIGNvbXBvbmVudDogTWFpblxuICogfV1cbiAqIGBgYFxuICpcbiAqICMjIyBDb21wb25lbnRsZXNzIFJvdXRlc1xuICpcbiAqIFlvdSBjYW4gc2hhcmUgcGFyYW1ldGVycyBiZXR3ZWVuIHNpYmxpbmcgY29tcG9uZW50cy5cbiAqIEZvciBleGFtcGxlLCBzdXBwb3NlIHRoYXQgdHdvIHNpYmxpbmcgY29tcG9uZW50cyBzaG91bGQgZ28gbmV4dCB0byBlYWNoIG90aGVyLFxuICogYW5kIGJvdGggb2YgdGhlbSByZXF1aXJlIGFuIElEIHBhcmFtZXRlci4gWW91IGNhbiBhY2NvbXBsaXNoIHRoaXMgdXNpbmcgYSByb3V0ZVxuICogdGhhdCBkb2VzIG5vdCBzcGVjaWZ5IGEgY29tcG9uZW50IGF0IHRoZSB0b3AgbGV2ZWwuXG4gKlxuICogSW4gdGhlIGZvbGxvd2luZyBleGFtcGxlLCAnTWFpbkNoaWxkJyBhbmQgJ0F1eENoaWxkJyBhcmUgc2libGluZ3MuXG4gKiBXaGVuIG5hdmlnYXRpbmcgdG8gJ3BhcmVudC8xMC8oYS8vYXV4OmIpJywgdGhlIHJvdXRlIGluc3RhbnRpYXRlc1xuICogdGhlIG1haW4gY2hpbGQgYW5kIGF1eCBjaGlsZCBjb21wb25lbnRzIG5leHQgdG8gZWFjaCBvdGhlci5cbiAqIEZvciB0aGlzIHRvIHdvcmssIHRoZSBhcHBsaWNhdGlvbiBjb21wb25lbnQgbXVzdCBoYXZlIHRoZSBwcmltYXJ5IGFuZCBhdXggb3V0bGV0cyBkZWZpbmVkLlxuICpcbiAqIGBgYFxuICogW3tcbiAqICAgIHBhdGg6ICdwYXJlbnQvOmlkJyxcbiAqICAgIGNoaWxkcmVuOiBbXG4gKiAgICAgIHsgcGF0aDogJ2EnLCBjb21wb25lbnQ6IE1haW5DaGlsZCB9LFxuICogICAgICB7IHBhdGg6ICdiJywgY29tcG9uZW50OiBBdXhDaGlsZCwgb3V0bGV0OiAnYXV4JyB9XG4gKiAgICBdXG4gKiB9XVxuICogYGBgXG4gKlxuICogVGhlIHJvdXRlciBtZXJnZXMgdGhlIHBhcmFtZXRlcnMsIGRhdGEsIGFuZCByZXNvbHZlIG9mIHRoZSBjb21wb25lbnRsZXNzXG4gKiBwYXJlbnQgaW50byB0aGUgcGFyYW1ldGVycywgZGF0YSwgYW5kIHJlc29sdmUgb2YgdGhlIGNoaWxkcmVuLlxuICpcbiAqIFRoaXMgaXMgZXNwZWNpYWxseSB1c2VmdWwgd2hlbiBjaGlsZCBjb21wb25lbnRzIGFyZSBkZWZpbmVkXG4gKiB3aXRoIGFuIGVtcHR5IHBhdGggc3RyaW5nLCBhcyBpbiB0aGUgZm9sbG93aW5nIGV4YW1wbGUuXG4gKiBXaXRoIHRoaXMgY29uZmlndXJhdGlvbiwgbmF2aWdhdGluZyB0byAnL3BhcmVudC8xMCcgY3JlYXRlc1xuICogdGhlIG1haW4gY2hpbGQgYW5kIGF1eCBjb21wb25lbnRzLlxuICpcbiAqIGBgYFxuICogW3tcbiAqICAgIHBhdGg6ICdwYXJlbnQvOmlkJyxcbiAqICAgIGNoaWxkcmVuOiBbXG4gKiAgICAgIHsgcGF0aDogJycsIGNvbXBvbmVudDogTWFpbkNoaWxkIH0sXG4gKiAgICAgIHsgcGF0aDogJycsIGNvbXBvbmVudDogQXV4Q2hpbGQsIG91dGxldDogJ2F1eCcgfVxuICogICAgXVxuICogfV1cbiAqIGBgYFxuICpcbiAqICMjIyBMYXp5IExvYWRpbmdcbiAqXG4gKiBMYXp5IGxvYWRpbmcgc3BlZWRzIHVwIGFwcGxpY2F0aW9uIGxvYWQgdGltZSBieSBzcGxpdHRpbmcgdGhlIGFwcGxpY2F0aW9uXG4gKiBpbnRvIG11bHRpcGxlIGJ1bmRsZXMgYW5kIGxvYWRpbmcgdGhlbSBvbiBkZW1hbmQuXG4gKiBUbyB1c2UgbGF6eSBsb2FkaW5nLCBwcm92aWRlIHRoZSBgbG9hZENoaWxkcmVuYCBwcm9wZXJ0eSBpbiB0aGUgYFJvdXRlYCBvYmplY3QsXG4gKiBpbnN0ZWFkIG9mIHRoZSBgY2hpbGRyZW5gIHByb3BlcnR5LlxuICpcbiAqIEdpdmVuIHRoZSBmb2xsb3dpbmcgZXhhbXBsZSByb3V0ZSwgdGhlIHJvdXRlciB3aWxsIGxhenkgbG9hZFxuICogdGhlIGFzc29jaWF0ZWQgbW9kdWxlIG9uIGRlbWFuZCB1c2luZyB0aGUgYnJvd3NlciBuYXRpdmUgaW1wb3J0IHN5c3RlbS5cbiAqXG4gKiBgYGBcbiAqIFt7XG4gKiAgIHBhdGg6ICdsYXp5JyxcbiAqICAgbG9hZENoaWxkcmVuOiAoKSA9PiBpbXBvcnQoJy4vbGF6eS1yb3V0ZS9sYXp5Lm1vZHVsZScpLnRoZW4obW9kID0+IG1vZC5MYXp5TW9kdWxlKSxcbiAqIH1dO1xuICogYGBgXG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJvdXRlIHtcbiAgLyoqXG4gICAqIFVzZWQgdG8gZGVmaW5lIGEgcGFnZSB0aXRsZSBmb3IgdGhlIHJvdXRlLiBUaGlzIGNhbiBiZSBhIHN0YXRpYyBzdHJpbmcgb3IgYW4gYEluamVjdGFibGVgIHRoYXRcbiAgICogaW1wbGVtZW50cyBgUmVzb2x2ZWAuXG4gICAqXG4gICAqIEBzZWUge0BsaW5rIFRpdGxlU3RyYXRlZ3l9XG4gICAqL1xuICB0aXRsZT86IHN0cmluZyB8IFR5cGU8UmVzb2x2ZTxzdHJpbmc+PiB8IFJlc29sdmVGbjxzdHJpbmc+O1xuXG4gIC8qKlxuICAgKiBUaGUgcGF0aCB0byBtYXRjaCBhZ2FpbnN0LiBDYW5ub3QgYmUgdXNlZCB0b2dldGhlciB3aXRoIGEgY3VzdG9tIGBtYXRjaGVyYCBmdW5jdGlvbi5cbiAgICogQSBVUkwgc3RyaW5nIHRoYXQgdXNlcyByb3V0ZXIgbWF0Y2hpbmcgbm90YXRpb24uXG4gICAqIENhbiBiZSBhIHdpbGQgY2FyZCAoYCoqYCkgdGhhdCBtYXRjaGVzIGFueSBVUkwgKHNlZSBVc2FnZSBOb3RlcyBiZWxvdykuXG4gICAqIERlZmF1bHQgaXMgXCIvXCIgKHRoZSByb290IHBhdGgpLlxuICAgKlxuICAgKi9cbiAgcGF0aD86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBwYXRoLW1hdGNoaW5nIHN0cmF0ZWd5LCBvbmUgb2YgJ3ByZWZpeCcgb3IgJ2Z1bGwnLlxuICAgKiBEZWZhdWx0IGlzICdwcmVmaXgnLlxuICAgKlxuICAgKiBCeSBkZWZhdWx0LCB0aGUgcm91dGVyIGNoZWNrcyBVUkwgZWxlbWVudHMgZnJvbSB0aGUgbGVmdCB0byBzZWUgaWYgdGhlIFVSTFxuICAgKiBtYXRjaGVzIGEgZ2l2ZW4gcGF0aCBhbmQgc3RvcHMgd2hlbiB0aGVyZSBpcyBhIGNvbmZpZyBtYXRjaC4gSW1wb3J0YW50bHkgdGhlcmUgbXVzdCBzdGlsbCBiZSBhXG4gICAqIGNvbmZpZyBtYXRjaCBmb3IgZWFjaCBzZWdtZW50IG9mIHRoZSBVUkwuIEZvciBleGFtcGxlLCAnL3RlYW0vMTEvdXNlcicgbWF0Y2hlcyB0aGUgcHJlZml4XG4gICAqICd0ZWFtLzppZCcgaWYgb25lIG9mIHRoZSByb3V0ZSdzIGNoaWxkcmVuIG1hdGNoZXMgdGhlIHNlZ21lbnQgJ3VzZXInLiBUaGF0IGlzLCB0aGUgVVJMXG4gICAqICcvdGVhbS8xMS91c2VyJyBtYXRjaGVzIHRoZSBjb25maWdcbiAgICogYHtwYXRoOiAndGVhbS86aWQnLCBjaGlsZHJlbjogW3twYXRoOiAnOnVzZXInLCBjb21wb25lbnQ6IFVzZXJ9XX1gXG4gICAqIGJ1dCBkb2VzIG5vdCBtYXRjaCB3aGVuIHRoZXJlIGFyZSBubyBjaGlsZHJlbiBhcyBpbiBge3BhdGg6ICd0ZWFtLzppZCcsIGNvbXBvbmVudDogVGVhbX1gLlxuICAgKlxuICAgKiBUaGUgcGF0aC1tYXRjaCBzdHJhdGVneSAnZnVsbCcgbWF0Y2hlcyBhZ2FpbnN0IHRoZSBlbnRpcmUgVVJMLlxuICAgKiBJdCBpcyBpbXBvcnRhbnQgdG8gZG8gdGhpcyB3aGVuIHJlZGlyZWN0aW5nIGVtcHR5LXBhdGggcm91dGVzLlxuICAgKiBPdGhlcndpc2UsIGJlY2F1c2UgYW4gZW1wdHkgcGF0aCBpcyBhIHByZWZpeCBvZiBhbnkgVVJMLFxuICAgKiB0aGUgcm91dGVyIHdvdWxkIGFwcGx5IHRoZSByZWRpcmVjdCBldmVuIHdoZW4gbmF2aWdhdGluZ1xuICAgKiB0byB0aGUgcmVkaXJlY3QgZGVzdGluYXRpb24sIGNyZWF0aW5nIGFuIGVuZGxlc3MgbG9vcC5cbiAgICpcbiAgICovXG4gIHBhdGhNYXRjaD86ICdwcmVmaXgnIHwgJ2Z1bGwnO1xuICAvKipcbiAgICogQSBjdXN0b20gVVJMLW1hdGNoaW5nIGZ1bmN0aW9uLiBDYW5ub3QgYmUgdXNlZCB0b2dldGhlciB3aXRoIGBwYXRoYC5cbiAgICovXG4gIG1hdGNoZXI/OiBVcmxNYXRjaGVyO1xuICAvKipcbiAgICogVGhlIGNvbXBvbmVudCB0byBpbnN0YW50aWF0ZSB3aGVuIHRoZSBwYXRoIG1hdGNoZXMuXG4gICAqIENhbiBiZSBlbXB0eSBpZiBjaGlsZCByb3V0ZXMgc3BlY2lmeSBjb21wb25lbnRzLlxuICAgKi9cbiAgY29tcG9uZW50PzogVHlwZTxhbnk+O1xuXG4gIC8qKlxuICAgKiBBbiBvYmplY3Qgc3BlY2lmeWluZyBhIGxhenktbG9hZGVkIGNvbXBvbmVudC5cbiAgICovXG4gIGxvYWRDb21wb25lbnQ/OiAoKSA9PlxuICAgIHwgVHlwZTx1bmtub3duPlxuICAgIHwgT2JzZXJ2YWJsZTxUeXBlPHVua25vd24+IHwgRGVmYXVsdEV4cG9ydDxUeXBlPHVua25vd24+Pj5cbiAgICB8IFByb21pc2U8VHlwZTx1bmtub3duPiB8IERlZmF1bHRFeHBvcnQ8VHlwZTx1bmtub3duPj4+O1xuICAvKipcbiAgICogRmlsbGVkIGZvciByb3V0ZXMgYGxvYWRDb21wb25lbnRgIG9uY2UgdGhlIGNvbXBvbmVudCBpcyBsb2FkZWQuXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgX2xvYWRlZENvbXBvbmVudD86IFR5cGU8dW5rbm93bj47XG5cbiAgLyoqXG4gICAqIEEgVVJMIG9yIGZ1bmN0aW9uIHRoYXQgcmV0dXJucyBhIFVSTCB0byByZWRpcmVjdCB0byB3aGVuIHRoZSBwYXRoIG1hdGNoZXMuXG4gICAqXG4gICAqIEFic29sdXRlIGlmIHRoZSBVUkwgYmVnaW5zIHdpdGggYSBzbGFzaCAoLykgb3IgdGhlIGZ1bmN0aW9uIHJldHVybnMgYSBgVXJsVHJlZWAsIG90aGVyd2lzZVxuICAgKiByZWxhdGl2ZSB0byB0aGUgcGF0aCBVUkwuXG4gICAqXG4gICAqIFRoZSBgUmVkaXJlY3RGdW5jdGlvbmAgaXMgcnVuIGluIGFuIGluamVjdGlvbiBjb250ZXh0IHNvIGl0IGNhbiBjYWxsIGBpbmplY3RgIHRvIGdldCBhbnlcbiAgICogcmVxdWlyZWQgZGVwZW5kZW5jaWVzLlxuICAgKlxuICAgKiBXaGVuIG5vdCBwcmVzZW50LCByb3V0ZXIgZG9lcyBub3QgcmVkaXJlY3QuXG4gICAqL1xuICByZWRpcmVjdFRvPzogc3RyaW5nIHwgUmVkaXJlY3RGdW5jdGlvbjtcbiAgLyoqXG4gICAqIE5hbWUgb2YgYSBgUm91dGVyT3V0bGV0YCBvYmplY3Qgd2hlcmUgdGhlIGNvbXBvbmVudCBjYW4gYmUgcGxhY2VkXG4gICAqIHdoZW4gdGhlIHBhdGggbWF0Y2hlcy5cbiAgICovXG4gIG91dGxldD86IHN0cmluZztcbiAgLyoqXG4gICAqIEFuIGFycmF5IG9mIGBDYW5BY3RpdmF0ZUZuYCBvciBESSB0b2tlbnMgdXNlZCB0byBsb29rIHVwIGBDYW5BY3RpdmF0ZSgpYFxuICAgKiBoYW5kbGVycywgaW4gb3JkZXIgdG8gZGV0ZXJtaW5lIGlmIHRoZSBjdXJyZW50IHVzZXIgaXMgYWxsb3dlZCB0b1xuICAgKiBhY3RpdmF0ZSB0aGUgY29tcG9uZW50LiBCeSBkZWZhdWx0LCBhbnkgdXNlciBjYW4gYWN0aXZhdGUuXG4gICAqXG4gICAqIFdoZW4gdXNpbmcgYSBmdW5jdGlvbiByYXRoZXIgdGhhbiBESSB0b2tlbnMsIHRoZSBmdW5jdGlvbiBjYW4gY2FsbCBgaW5qZWN0YCB0byBnZXQgYW55IHJlcXVpcmVkXG4gICAqIGRlcGVuZGVuY2llcy4gVGhpcyBgaW5qZWN0YCBjYWxsIG11c3QgYmUgZG9uZSBpbiBhIHN5bmNocm9ub3VzIGNvbnRleHQuXG4gICAqL1xuICBjYW5BY3RpdmF0ZT86IEFycmF5PENhbkFjdGl2YXRlRm4gfCBEZXByZWNhdGVkR3VhcmQ+O1xuICAvKipcbiAgICogQW4gYXJyYXkgb2YgYENhbk1hdGNoRm5gIG9yIERJIHRva2VucyB1c2VkIHRvIGxvb2sgdXAgYENhbk1hdGNoKClgXG4gICAqIGhhbmRsZXJzLCBpbiBvcmRlciB0byBkZXRlcm1pbmUgaWYgdGhlIGN1cnJlbnQgdXNlciBpcyBhbGxvd2VkIHRvXG4gICAqIG1hdGNoIHRoZSBgUm91dGVgLiBCeSBkZWZhdWx0LCBhbnkgcm91dGUgY2FuIG1hdGNoLlxuICAgKlxuICAgKiBXaGVuIHVzaW5nIGEgZnVuY3Rpb24gcmF0aGVyIHRoYW4gREkgdG9rZW5zLCB0aGUgZnVuY3Rpb24gY2FuIGNhbGwgYGluamVjdGAgdG8gZ2V0IGFueSByZXF1aXJlZFxuICAgKiBkZXBlbmRlbmNpZXMuIFRoaXMgYGluamVjdGAgY2FsbCBtdXN0IGJlIGRvbmUgaW4gYSBzeW5jaHJvbm91cyBjb250ZXh0LlxuICAgKi9cbiAgY2FuTWF0Y2g/OiBBcnJheTxDYW5NYXRjaEZuIHwgRGVwcmVjYXRlZEd1YXJkPjtcbiAgLyoqXG4gICAqIEFuIGFycmF5IG9mIGBDYW5BY3RpdmF0ZUNoaWxkRm5gIG9yIERJIHRva2VucyB1c2VkIHRvIGxvb2sgdXAgYENhbkFjdGl2YXRlQ2hpbGQoKWAgaGFuZGxlcnMsXG4gICAqIGluIG9yZGVyIHRvIGRldGVybWluZSBpZiB0aGUgY3VycmVudCB1c2VyIGlzIGFsbG93ZWQgdG8gYWN0aXZhdGVcbiAgICogYSBjaGlsZCBvZiB0aGUgY29tcG9uZW50LiBCeSBkZWZhdWx0LCBhbnkgdXNlciBjYW4gYWN0aXZhdGUgYSBjaGlsZC5cbiAgICpcbiAgICogV2hlbiB1c2luZyBhIGZ1bmN0aW9uIHJhdGhlciB0aGFuIERJIHRva2VucywgdGhlIGZ1bmN0aW9uIGNhbiBjYWxsIGBpbmplY3RgIHRvIGdldCBhbnkgcmVxdWlyZWRcbiAgICogZGVwZW5kZW5jaWVzLiBUaGlzIGBpbmplY3RgIGNhbGwgbXVzdCBiZSBkb25lIGluIGEgc3luY2hyb25vdXMgY29udGV4dC5cbiAgICovXG4gIGNhbkFjdGl2YXRlQ2hpbGQ/OiBBcnJheTxDYW5BY3RpdmF0ZUNoaWxkRm4gfCBEZXByZWNhdGVkR3VhcmQ+O1xuICAvKipcbiAgICogQW4gYXJyYXkgb2YgYENhbkRlYWN0aXZhdGVGbmAgb3IgREkgdG9rZW5zIHVzZWQgdG8gbG9vayB1cCBgQ2FuRGVhY3RpdmF0ZSgpYFxuICAgKiBoYW5kbGVycywgaW4gb3JkZXIgdG8gZGV0ZXJtaW5lIGlmIHRoZSBjdXJyZW50IHVzZXIgaXMgYWxsb3dlZCB0b1xuICAgKiBkZWFjdGl2YXRlIHRoZSBjb21wb25lbnQuIEJ5IGRlZmF1bHQsIGFueSB1c2VyIGNhbiBkZWFjdGl2YXRlLlxuICAgKlxuICAgKiBXaGVuIHVzaW5nIGEgZnVuY3Rpb24gcmF0aGVyIHRoYW4gREkgdG9rZW5zLCB0aGUgZnVuY3Rpb24gY2FuIGNhbGwgYGluamVjdGAgdG8gZ2V0IGFueSByZXF1aXJlZFxuICAgKiBkZXBlbmRlbmNpZXMuIFRoaXMgYGluamVjdGAgY2FsbCBtdXN0IGJlIGRvbmUgaW4gYSBzeW5jaHJvbm91cyBjb250ZXh0LlxuICAgKi9cbiAgY2FuRGVhY3RpdmF0ZT86IEFycmF5PENhbkRlYWN0aXZhdGVGbjxhbnk+IHwgRGVwcmVjYXRlZEd1YXJkPjtcbiAgLyoqXG4gICAqIEFuIGFycmF5IG9mIGBDYW5Mb2FkRm5gIG9yIERJIHRva2VucyB1c2VkIHRvIGxvb2sgdXAgYENhbkxvYWQoKWBcbiAgICogaGFuZGxlcnMsIGluIG9yZGVyIHRvIGRldGVybWluZSBpZiB0aGUgY3VycmVudCB1c2VyIGlzIGFsbG93ZWQgdG9cbiAgICogbG9hZCB0aGUgY29tcG9uZW50LiBCeSBkZWZhdWx0LCBhbnkgdXNlciBjYW4gbG9hZC5cbiAgICpcbiAgICogV2hlbiB1c2luZyBhIGZ1bmN0aW9uIHJhdGhlciB0aGFuIERJIHRva2VucywgdGhlIGZ1bmN0aW9uIGNhbiBjYWxsIGBpbmplY3RgIHRvIGdldCBhbnkgcmVxdWlyZWRcbiAgICogZGVwZW5kZW5jaWVzLiBUaGlzIGBpbmplY3RgIGNhbGwgbXVzdCBiZSBkb25lIGluIGEgc3luY2hyb25vdXMgY29udGV4dC5cbiAgICogQGRlcHJlY2F0ZWQgVXNlIGBjYW5NYXRjaGAgaW5zdGVhZFxuICAgKi9cbiAgY2FuTG9hZD86IEFycmF5PENhbkxvYWRGbiB8IERlcHJlY2F0ZWRHdWFyZD47XG4gIC8qKlxuICAgKiBBZGRpdGlvbmFsIGRldmVsb3Blci1kZWZpbmVkIGRhdGEgcHJvdmlkZWQgdG8gdGhlIGNvbXBvbmVudCB2aWFcbiAgICogYEFjdGl2YXRlZFJvdXRlYC4gQnkgZGVmYXVsdCwgbm8gYWRkaXRpb25hbCBkYXRhIGlzIHBhc3NlZC5cbiAgICovXG4gIGRhdGE/OiBEYXRhO1xuICAvKipcbiAgICogQSBtYXAgb2YgREkgdG9rZW5zIHVzZWQgdG8gbG9vayB1cCBkYXRhIHJlc29sdmVycy4gU2VlIGBSZXNvbHZlYC5cbiAgICovXG4gIHJlc29sdmU/OiBSZXNvbHZlRGF0YTtcbiAgLyoqXG4gICAqIEFuIGFycmF5IG9mIGNoaWxkIGBSb3V0ZWAgb2JqZWN0cyB0aGF0IHNwZWNpZmllcyBhIG5lc3RlZCByb3V0ZVxuICAgKiBjb25maWd1cmF0aW9uLlxuICAgKi9cbiAgY2hpbGRyZW4/OiBSb3V0ZXM7XG4gIC8qKlxuICAgKiBBbiBvYmplY3Qgc3BlY2lmeWluZyBsYXp5LWxvYWRlZCBjaGlsZCByb3V0ZXMuXG4gICAqL1xuICBsb2FkQ2hpbGRyZW4/OiBMb2FkQ2hpbGRyZW47XG5cbiAgLyoqXG4gICAqIEEgcG9saWN5IGZvciB3aGVuIHRvIHJ1biBndWFyZHMgYW5kIHJlc29sdmVycyBvbiBhIHJvdXRlLlxuICAgKlxuICAgKiBHdWFyZHMgYW5kL29yIHJlc29sdmVycyB3aWxsIGFsd2F5cyBydW4gd2hlbiBhIHJvdXRlIGlzIGFjdGl2YXRlZCBvciBkZWFjdGl2YXRlZC4gV2hlbiBhIHJvdXRlXG4gICAqIGlzIHVuY2hhbmdlZCwgdGhlIGRlZmF1bHQgYmVoYXZpb3IgaXMgdGhlIHNhbWUgYXMgYHBhcmFtc0NoYW5nZWAuXG4gICAqXG4gICAqIGBwYXJhbXNDaGFuZ2VgIDogUmVydW4gdGhlIGd1YXJkcyBhbmQgcmVzb2x2ZXJzIHdoZW4gcGF0aCBvclxuICAgKiBwYXRoIHBhcmFtIGNoYW5nZXMuIFRoaXMgZG9lcyBub3QgaW5jbHVkZSBxdWVyeSBwYXJhbWV0ZXJzLiBUaGlzIG9wdGlvbiBpcyB0aGUgZGVmYXVsdC5cbiAgICogLSBgYWx3YXlzYCA6IFJ1biBvbiBldmVyeSBleGVjdXRpb24uXG4gICAqIC0gYHBhdGhQYXJhbXNDaGFuZ2VgIDogUmVydW4gZ3VhcmRzIGFuZCByZXNvbHZlcnMgd2hlbiB0aGUgcGF0aCBwYXJhbXNcbiAgICogY2hhbmdlLiBUaGlzIGRvZXMgbm90IGNvbXBhcmUgbWF0cml4IG9yIHF1ZXJ5IHBhcmFtZXRlcnMuXG4gICAqIC0gYHBhcmFtc09yUXVlcnlQYXJhbXNDaGFuZ2VgIDogUnVuIHdoZW4gcGF0aCwgbWF0cml4LCBvciBxdWVyeSBwYXJhbWV0ZXJzIGNoYW5nZS5cbiAgICogLSBgcGF0aFBhcmFtc09yUXVlcnlQYXJhbXNDaGFuZ2VgIDogUmVydW4gZ3VhcmRzIGFuZCByZXNvbHZlcnMgd2hlbiB0aGUgcGF0aCBwYXJhbXNcbiAgICogY2hhbmdlIG9yIHF1ZXJ5IHBhcmFtcyBoYXZlIGNoYW5nZWQuIFRoaXMgZG9lcyBub3QgaW5jbHVkZSBtYXRyaXggcGFyYW1ldGVycy5cbiAgICpcbiAgICogQHNlZSB7QGxpbmsgUnVuR3VhcmRzQW5kUmVzb2x2ZXJzfVxuICAgKi9cbiAgcnVuR3VhcmRzQW5kUmVzb2x2ZXJzPzogUnVuR3VhcmRzQW5kUmVzb2x2ZXJzO1xuXG4gIC8qKlxuICAgKiBBIGBQcm92aWRlcmAgYXJyYXkgdG8gdXNlIGZvciB0aGlzIGBSb3V0ZWAgYW5kIGl0cyBgY2hpbGRyZW5gLlxuICAgKlxuICAgKiBUaGUgYFJvdXRlcmAgd2lsbCBjcmVhdGUgYSBuZXcgYEVudmlyb25tZW50SW5qZWN0b3JgIGZvciB0aGlzXG4gICAqIGBSb3V0ZWAgYW5kIHVzZSBpdCBmb3IgdGhpcyBgUm91dGVgIGFuZCBpdHMgYGNoaWxkcmVuYC4gSWYgdGhpc1xuICAgKiByb3V0ZSBhbHNvIGhhcyBhIGBsb2FkQ2hpbGRyZW5gIGZ1bmN0aW9uIHdoaWNoIHJldHVybnMgYW4gYE5nTW9kdWxlUmVmYCwgdGhpcyBpbmplY3RvciB3aWxsIGJlXG4gICAqIHVzZWQgYXMgdGhlIHBhcmVudCBvZiB0aGUgbGF6eSBsb2FkZWQgbW9kdWxlLlxuICAgKi9cbiAgcHJvdmlkZXJzPzogQXJyYXk8UHJvdmlkZXIgfCBFbnZpcm9ubWVudFByb3ZpZGVycz47XG5cbiAgLyoqXG4gICAqIEluamVjdG9yIGNyZWF0ZWQgZnJvbSB0aGUgc3RhdGljIHJvdXRlIHByb3ZpZGVyc1xuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIF9pbmplY3Rvcj86IEVudmlyb25tZW50SW5qZWN0b3I7XG5cbiAgLyoqXG4gICAqIEZpbGxlZCBmb3Igcm91dGVzIHdpdGggYGxvYWRDaGlsZHJlbmAgb25jZSB0aGUgcm91dGVzIGFyZSBsb2FkZWQuXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgX2xvYWRlZFJvdXRlcz86IFJvdXRlW107XG5cbiAgLyoqXG4gICAqIEZpbGxlZCBmb3Igcm91dGVzIHdpdGggYGxvYWRDaGlsZHJlbmAgb25jZSB0aGUgcm91dGVzIGFyZSBsb2FkZWRcbiAgICogQGludGVybmFsXG4gICAqL1xuICBfbG9hZGVkSW5qZWN0b3I/OiBFbnZpcm9ubWVudEluamVjdG9yO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIExvYWRlZFJvdXRlckNvbmZpZyB7XG4gIHJvdXRlczogUm91dGVbXTtcbiAgaW5qZWN0b3I6IEVudmlyb25tZW50SW5qZWN0b3IgfCB1bmRlZmluZWQ7XG59XG5cbi8qKlxuICogQGRlc2NyaXB0aW9uXG4gKlxuICogSW50ZXJmYWNlIHRoYXQgYSBjbGFzcyBjYW4gaW1wbGVtZW50IHRvIGJlIGEgZ3VhcmQgZGVjaWRpbmcgaWYgYSByb3V0ZSBjYW4gYmUgYWN0aXZhdGVkLlxuICogSWYgYWxsIGd1YXJkcyByZXR1cm4gYHRydWVgLCBuYXZpZ2F0aW9uIGNvbnRpbnVlcy4gSWYgYW55IGd1YXJkIHJldHVybnMgYGZhbHNlYCxcbiAqIG5hdmlnYXRpb24gaXMgY2FuY2VsbGVkLiBJZiBhbnkgZ3VhcmQgcmV0dXJucyBhIGBVcmxUcmVlYCwgdGhlIGN1cnJlbnQgbmF2aWdhdGlvblxuICogaXMgY2FuY2VsbGVkIGFuZCBhIG5ldyBuYXZpZ2F0aW9uIGJlZ2lucyB0byB0aGUgYFVybFRyZWVgIHJldHVybmVkIGZyb20gdGhlIGd1YXJkLlxuICpcbiAqIFRoZSBmb2xsb3dpbmcgZXhhbXBsZSBpbXBsZW1lbnRzIGEgYENhbkFjdGl2YXRlYCBmdW5jdGlvbiB0aGF0IGNoZWNrcyB3aGV0aGVyIHRoZVxuICogY3VycmVudCB1c2VyIGhhcyBwZXJtaXNzaW9uIHRvIGFjdGl2YXRlIHRoZSByZXF1ZXN0ZWQgcm91dGUuXG4gKlxuICogYGBgXG4gKiBjbGFzcyBVc2VyVG9rZW4ge31cbiAqIGNsYXNzIFBlcm1pc3Npb25zIHtcbiAqICAgY2FuQWN0aXZhdGUoKTogYm9vbGVhbiB7XG4gKiAgICAgcmV0dXJuIHRydWU7XG4gKiAgIH1cbiAqIH1cbiAqXG4gKiBASW5qZWN0YWJsZSgpXG4gKiBjbGFzcyBDYW5BY3RpdmF0ZVRlYW0gaW1wbGVtZW50cyBDYW5BY3RpdmF0ZSB7XG4gKiAgIGNvbnN0cnVjdG9yKHByaXZhdGUgcGVybWlzc2lvbnM6IFBlcm1pc3Npb25zLCBwcml2YXRlIGN1cnJlbnRVc2VyOiBVc2VyVG9rZW4pIHt9XG4gKlxuICogICBjYW5BY3RpdmF0ZShcbiAqICAgICByb3V0ZTogQWN0aXZhdGVkUm91dGVTbmFwc2hvdCxcbiAqICAgICBzdGF0ZTogUm91dGVyU3RhdGVTbmFwc2hvdFxuICogICApOiBNYXliZUFzeW5jPEd1YXJkUmVzdWx0PiB7XG4gKiAgICAgcmV0dXJuIHRoaXMucGVybWlzc2lvbnMuY2FuQWN0aXZhdGUodGhpcy5jdXJyZW50VXNlciwgcm91dGUucGFyYW1zLmlkKTtcbiAqICAgfVxuICogfVxuICogYGBgXG4gKlxuICogSGVyZSwgdGhlIGRlZmluZWQgZ3VhcmQgZnVuY3Rpb24gaXMgcHJvdmlkZWQgYXMgcGFydCBvZiB0aGUgYFJvdXRlYCBvYmplY3RcbiAqIGluIHRoZSByb3V0ZXIgY29uZmlndXJhdGlvbjpcbiAqXG4gKiBgYGBcbiAqIEBOZ01vZHVsZSh7XG4gKiAgIGltcG9ydHM6IFtcbiAqICAgICBSb3V0ZXJNb2R1bGUuZm9yUm9vdChbXG4gKiAgICAgICB7XG4gKiAgICAgICAgIHBhdGg6ICd0ZWFtLzppZCcsXG4gKiAgICAgICAgIGNvbXBvbmVudDogVGVhbUNvbXBvbmVudCxcbiAqICAgICAgICAgY2FuQWN0aXZhdGU6IFtDYW5BY3RpdmF0ZVRlYW1dXG4gKiAgICAgICB9XG4gKiAgICAgXSlcbiAqICAgXSxcbiAqICAgcHJvdmlkZXJzOiBbQ2FuQWN0aXZhdGVUZWFtLCBVc2VyVG9rZW4sIFBlcm1pc3Npb25zXVxuICogfSlcbiAqIGNsYXNzIEFwcE1vZHVsZSB7fVxuICogYGBgXG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgaW50ZXJmYWNlIENhbkFjdGl2YXRlIHtcbiAgY2FuQWN0aXZhdGUocm91dGU6IEFjdGl2YXRlZFJvdXRlU25hcHNob3QsIHN0YXRlOiBSb3V0ZXJTdGF0ZVNuYXBzaG90KTogTWF5YmVBc3luYzxHdWFyZFJlc3VsdD47XG59XG5cbi8qKlxuICogVGhlIHNpZ25hdHVyZSBvZiBhIGZ1bmN0aW9uIHVzZWQgYXMgYSBgY2FuQWN0aXZhdGVgIGd1YXJkIG9uIGEgYFJvdXRlYC5cbiAqXG4gKiBJZiBhbGwgZ3VhcmRzIHJldHVybiBgdHJ1ZWAsIG5hdmlnYXRpb24gY29udGludWVzLiBJZiBhbnkgZ3VhcmQgcmV0dXJucyBgZmFsc2VgLFxuICogbmF2aWdhdGlvbiBpcyBjYW5jZWxsZWQuIElmIGFueSBndWFyZCByZXR1cm5zIGEgYFVybFRyZWVgLCB0aGUgY3VycmVudCBuYXZpZ2F0aW9uXG4gKiBpcyBjYW5jZWxsZWQgYW5kIGEgbmV3IG5hdmlnYXRpb24gYmVnaW5zIHRvIHRoZSBgVXJsVHJlZWAgcmV0dXJuZWQgZnJvbSB0aGUgZ3VhcmQuXG4gKlxuICogVGhlIGZvbGxvd2luZyBleGFtcGxlIGltcGxlbWVudHMgYW5kIHVzZXMgYSBgQ2FuQWN0aXZhdGVGbmAgdGhhdCBjaGVja3Mgd2hldGhlciB0aGVcbiAqIGN1cnJlbnQgdXNlciBoYXMgcGVybWlzc2lvbiB0byBhY3RpdmF0ZSB0aGUgcmVxdWVzdGVkIHJvdXRlLlxuICpcbiAqIGBgYHRzXG4gKiBASW5qZWN0YWJsZSgpXG4gKiBjbGFzcyBVc2VyVG9rZW4ge31cbiAqXG4gKiBASW5qZWN0YWJsZSgpXG4gKiBjbGFzcyBQZXJtaXNzaW9uc1NlcnZpY2Uge1xuICogICBjYW5BY3RpdmF0ZShjdXJyZW50VXNlcjogVXNlclRva2VuLCB1c2VySWQ6IHN0cmluZyk6IGJvb2xlYW4ge1xuICogICAgIHJldHVybiB0cnVlO1xuICogICB9XG4gKiAgIGNhbk1hdGNoKGN1cnJlbnRVc2VyOiBVc2VyVG9rZW4pOiBib29sZWFuIHtcbiAqICAgICByZXR1cm4gdHJ1ZTtcbiAqICAgfVxuICogfVxuICpcbiAqIGNvbnN0IGNhbkFjdGl2YXRlVGVhbTogQ2FuQWN0aXZhdGVGbiA9IChcbiAqICAgcm91dGU6IEFjdGl2YXRlZFJvdXRlU25hcHNob3QsXG4gKiAgIHN0YXRlOiBSb3V0ZXJTdGF0ZVNuYXBzaG90LFxuICogKSA9PiB7XG4gKiAgIHJldHVybiBpbmplY3QoUGVybWlzc2lvbnNTZXJ2aWNlKS5jYW5BY3RpdmF0ZShpbmplY3QoVXNlclRva2VuKSwgcm91dGUucGFyYW1zWydpZCddKTtcbiAqIH07XG4gKiBgYGBcbiAqXG4gKiBIZXJlLCB0aGUgZGVmaW5lZCBndWFyZCBmdW5jdGlvbiBpcyBwcm92aWRlZCBhcyBwYXJ0IG9mIHRoZSBgUm91dGVgIG9iamVjdFxuICogaW4gdGhlIHJvdXRlciBjb25maWd1cmF0aW9uOlxuICpcbiAqIGBgYHRzXG4gKiBib290c3RyYXBBcHBsaWNhdGlvbihBcHAsIHtcbiAqICAgIHByb3ZpZGVyczogW1xuICogICAgICBwcm92aWRlUm91dGVyKFtcbiAqICAgICAgICB7XG4gKiAgICAgICAgICBwYXRoOiAndGVhbS86aWQnLFxuICogICAgICAgICAgY29tcG9uZW50OiBUZWFtQ29tcG9uZW50LFxuICogICAgICAgICAgY2FuQWN0aXZhdGU6IFtjYW5BY3RpdmF0ZVRlYW1dLFxuICogICAgICAgIH0sXG4gKiAgICAgIF0pLFxuICogICAgXSxcbiAqICB9KTtcbiAqIGBgYFxuICpcbiAqIEBwdWJsaWNBcGlcbiAqIEBzZWUge0BsaW5rIFJvdXRlfVxuICovXG5leHBvcnQgdHlwZSBDYW5BY3RpdmF0ZUZuID0gKFxuICByb3V0ZTogQWN0aXZhdGVkUm91dGVTbmFwc2hvdCxcbiAgc3RhdGU6IFJvdXRlclN0YXRlU25hcHNob3QsXG4pID0+IE1heWJlQXN5bmM8R3VhcmRSZXN1bHQ+O1xuXG4vKipcbiAqIEBkZXNjcmlwdGlvblxuICpcbiAqIEludGVyZmFjZSB0aGF0IGEgY2xhc3MgY2FuIGltcGxlbWVudCB0byBiZSBhIGd1YXJkIGRlY2lkaW5nIGlmIGEgY2hpbGQgcm91dGUgY2FuIGJlIGFjdGl2YXRlZC5cbiAqIElmIGFsbCBndWFyZHMgcmV0dXJuIGB0cnVlYCwgbmF2aWdhdGlvbiBjb250aW51ZXMuIElmIGFueSBndWFyZCByZXR1cm5zIGBmYWxzZWAsXG4gKiBuYXZpZ2F0aW9uIGlzIGNhbmNlbGxlZC4gSWYgYW55IGd1YXJkIHJldHVybnMgYSBgVXJsVHJlZWAsIGN1cnJlbnQgbmF2aWdhdGlvblxuICogaXMgY2FuY2VsbGVkIGFuZCBhIG5ldyBuYXZpZ2F0aW9uIGJlZ2lucyB0byB0aGUgYFVybFRyZWVgIHJldHVybmVkIGZyb20gdGhlIGd1YXJkLlxuICpcbiAqIFRoZSBmb2xsb3dpbmcgZXhhbXBsZSBpbXBsZW1lbnRzIGEgYENhbkFjdGl2YXRlQ2hpbGRgIGZ1bmN0aW9uIHRoYXQgY2hlY2tzIHdoZXRoZXIgdGhlXG4gKiBjdXJyZW50IHVzZXIgaGFzIHBlcm1pc3Npb24gdG8gYWN0aXZhdGUgdGhlIHJlcXVlc3RlZCBjaGlsZCByb3V0ZS5cbiAqXG4gKiBgYGBcbiAqIGNsYXNzIFVzZXJUb2tlbiB7fVxuICogY2xhc3MgUGVybWlzc2lvbnMge1xuICogICBjYW5BY3RpdmF0ZSh1c2VyOiBVc2VyVG9rZW4sIGlkOiBzdHJpbmcpOiBib29sZWFuIHtcbiAqICAgICByZXR1cm4gdHJ1ZTtcbiAqICAgfVxuICogfVxuICpcbiAqIEBJbmplY3RhYmxlKClcbiAqIGNsYXNzIENhbkFjdGl2YXRlVGVhbSBpbXBsZW1lbnRzIENhbkFjdGl2YXRlQ2hpbGQge1xuICogICBjb25zdHJ1Y3Rvcihwcml2YXRlIHBlcm1pc3Npb25zOiBQZXJtaXNzaW9ucywgcHJpdmF0ZSBjdXJyZW50VXNlcjogVXNlclRva2VuKSB7fVxuICpcbiAqICAgY2FuQWN0aXZhdGVDaGlsZChcbiAqICAgICByb3V0ZTogQWN0aXZhdGVkUm91dGVTbmFwc2hvdCxcbiAqICAgICBzdGF0ZTogUm91dGVyU3RhdGVTbmFwc2hvdFxuICogICApOiBNYXliZUFzeW5jPEd1YXJkUmVzdWx0PiB7XG4gKiAgICAgcmV0dXJuIHRoaXMucGVybWlzc2lvbnMuY2FuQWN0aXZhdGUodGhpcy5jdXJyZW50VXNlciwgcm91dGUucGFyYW1zLmlkKTtcbiAqICAgfVxuICogfVxuICogYGBgXG4gKlxuICogSGVyZSwgdGhlIGRlZmluZWQgZ3VhcmQgZnVuY3Rpb24gaXMgcHJvdmlkZWQgYXMgcGFydCBvZiB0aGUgYFJvdXRlYCBvYmplY3RcbiAqIGluIHRoZSByb3V0ZXIgY29uZmlndXJhdGlvbjpcbiAqXG4gKiBgYGBcbiAqIEBOZ01vZHVsZSh7XG4gKiAgIGltcG9ydHM6IFtcbiAqICAgICBSb3V0ZXJNb2R1bGUuZm9yUm9vdChbXG4gKiAgICAgICB7XG4gKiAgICAgICAgIHBhdGg6ICdyb290JyxcbiAqICAgICAgICAgY2FuQWN0aXZhdGVDaGlsZDogW0NhbkFjdGl2YXRlVGVhbV0sXG4gKiAgICAgICAgIGNoaWxkcmVuOiBbXG4gKiAgICAgICAgICAge1xuICogICAgICAgICAgICAgIHBhdGg6ICd0ZWFtLzppZCcsXG4gKiAgICAgICAgICAgICAgY29tcG9uZW50OiBUZWFtQ29tcG9uZW50XG4gKiAgICAgICAgICAgfVxuICogICAgICAgICBdXG4gKiAgICAgICB9XG4gKiAgICAgXSlcbiAqICAgXSxcbiAqICAgcHJvdmlkZXJzOiBbQ2FuQWN0aXZhdGVUZWFtLCBVc2VyVG9rZW4sIFBlcm1pc3Npb25zXVxuICogfSlcbiAqIGNsYXNzIEFwcE1vZHVsZSB7fVxuICogYGBgXG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgaW50ZXJmYWNlIENhbkFjdGl2YXRlQ2hpbGQge1xuICBjYW5BY3RpdmF0ZUNoaWxkKFxuICAgIGNoaWxkUm91dGU6IEFjdGl2YXRlZFJvdXRlU25hcHNob3QsXG4gICAgc3RhdGU6IFJvdXRlclN0YXRlU25hcHNob3QsXG4gICk6IE1heWJlQXN5bmM8R3VhcmRSZXN1bHQ+O1xufVxuXG4vKipcbiAqIFRoZSBzaWduYXR1cmUgb2YgYSBmdW5jdGlvbiB1c2VkIGFzIGEgYGNhbkFjdGl2YXRlQ2hpbGRgIGd1YXJkIG9uIGEgYFJvdXRlYC5cbiAqXG4gKiBJZiBhbGwgZ3VhcmRzIHJldHVybiBgdHJ1ZWAsIG5hdmlnYXRpb24gY29udGludWVzLiBJZiBhbnkgZ3VhcmQgcmV0dXJucyBgZmFsc2VgLFxuICogbmF2aWdhdGlvbiBpcyBjYW5jZWxsZWQuIElmIGFueSBndWFyZCByZXR1cm5zIGEgYFVybFRyZWVgLCB0aGUgY3VycmVudCBuYXZpZ2F0aW9uXG4gKiBpcyBjYW5jZWxsZWQgYW5kIGEgbmV3IG5hdmlnYXRpb24gYmVnaW5zIHRvIHRoZSBgVXJsVHJlZWAgcmV0dXJuZWQgZnJvbSB0aGUgZ3VhcmQuXG4gKlxuICogVGhlIGZvbGxvd2luZyBleGFtcGxlIGltcGxlbWVudHMgYSBgY2FuQWN0aXZhdGVgIGZ1bmN0aW9uIHRoYXQgY2hlY2tzIHdoZXRoZXIgdGhlXG4gKiBjdXJyZW50IHVzZXIgaGFzIHBlcm1pc3Npb24gdG8gYWN0aXZhdGUgdGhlIHJlcXVlc3RlZCByb3V0ZS5cbiAqXG4gKiB7QGV4YW1wbGUgcm91dGVyL3JvdXRlX2Z1bmN0aW9uYWxfZ3VhcmRzLnRzIHJlZ2lvbj1cIkNhbkFjdGl2YXRlQ2hpbGRGblwifVxuICpcbiAqIEBwdWJsaWNBcGlcbiAqIEBzZWUge0BsaW5rIFJvdXRlfVxuICovXG5leHBvcnQgdHlwZSBDYW5BY3RpdmF0ZUNoaWxkRm4gPSAoXG4gIGNoaWxkUm91dGU6IEFjdGl2YXRlZFJvdXRlU25hcHNob3QsXG4gIHN0YXRlOiBSb3V0ZXJTdGF0ZVNuYXBzaG90LFxuKSA9PiBNYXliZUFzeW5jPEd1YXJkUmVzdWx0PjtcblxuLyoqXG4gKiBAZGVzY3JpcHRpb25cbiAqXG4gKiBJbnRlcmZhY2UgdGhhdCBhIGNsYXNzIGNhbiBpbXBsZW1lbnQgdG8gYmUgYSBndWFyZCBkZWNpZGluZyBpZiBhIHJvdXRlIGNhbiBiZSBkZWFjdGl2YXRlZC5cbiAqIElmIGFsbCBndWFyZHMgcmV0dXJuIGB0cnVlYCwgbmF2aWdhdGlvbiBjb250aW51ZXMuIElmIGFueSBndWFyZCByZXR1cm5zIGBmYWxzZWAsXG4gKiBuYXZpZ2F0aW9uIGlzIGNhbmNlbGxlZC4gSWYgYW55IGd1YXJkIHJldHVybnMgYSBgVXJsVHJlZWAsIGN1cnJlbnQgbmF2aWdhdGlvblxuICogaXMgY2FuY2VsbGVkIGFuZCBhIG5ldyBuYXZpZ2F0aW9uIGJlZ2lucyB0byB0aGUgYFVybFRyZWVgIHJldHVybmVkIGZyb20gdGhlIGd1YXJkLlxuICpcbiAqIFRoZSBmb2xsb3dpbmcgZXhhbXBsZSBpbXBsZW1lbnRzIGEgYENhbkRlYWN0aXZhdGVgIGZ1bmN0aW9uIHRoYXQgY2hlY2tzIHdoZXRoZXIgdGhlXG4gKiBjdXJyZW50IHVzZXIgaGFzIHBlcm1pc3Npb24gdG8gZGVhY3RpdmF0ZSB0aGUgcmVxdWVzdGVkIHJvdXRlLlxuICpcbiAqIGBgYFxuICogY2xhc3MgVXNlclRva2VuIHt9XG4gKiBjbGFzcyBQZXJtaXNzaW9ucyB7XG4gKiAgIGNhbkRlYWN0aXZhdGUodXNlcjogVXNlclRva2VuLCBpZDogc3RyaW5nKTogYm9vbGVhbiB7XG4gKiAgICAgcmV0dXJuIHRydWU7XG4gKiAgIH1cbiAqIH1cbiAqIGBgYFxuICpcbiAqIEhlcmUsIHRoZSBkZWZpbmVkIGd1YXJkIGZ1bmN0aW9uIGlzIHByb3ZpZGVkIGFzIHBhcnQgb2YgdGhlIGBSb3V0ZWAgb2JqZWN0XG4gKiBpbiB0aGUgcm91dGVyIGNvbmZpZ3VyYXRpb246XG4gKlxuICogYGBgXG4gKlxuICogQEluamVjdGFibGUoKVxuICogY2xhc3MgQ2FuRGVhY3RpdmF0ZVRlYW0gaW1wbGVtZW50cyBDYW5EZWFjdGl2YXRlPFRlYW1Db21wb25lbnQ+IHtcbiAqICAgY29uc3RydWN0b3IocHJpdmF0ZSBwZXJtaXNzaW9uczogUGVybWlzc2lvbnMsIHByaXZhdGUgY3VycmVudFVzZXI6IFVzZXJUb2tlbikge31cbiAqXG4gKiAgIGNhbkRlYWN0aXZhdGUoXG4gKiAgICAgY29tcG9uZW50OiBUZWFtQ29tcG9uZW50LFxuICogICAgIGN1cnJlbnRSb3V0ZTogQWN0aXZhdGVkUm91dGVTbmFwc2hvdCxcbiAqICAgICBjdXJyZW50U3RhdGU6IFJvdXRlclN0YXRlU25hcHNob3QsXG4gKiAgICAgbmV4dFN0YXRlOiBSb3V0ZXJTdGF0ZVNuYXBzaG90XG4gKiAgICk6IE1heWJlQXN5bmM8R3VhcmRSZXN1bHQ+IHtcbiAqICAgICByZXR1cm4gdGhpcy5wZXJtaXNzaW9ucy5jYW5EZWFjdGl2YXRlKHRoaXMuY3VycmVudFVzZXIsIHJvdXRlLnBhcmFtcy5pZCk7XG4gKiAgIH1cbiAqIH1cbiAqXG4gKiBATmdNb2R1bGUoe1xuICogICBpbXBvcnRzOiBbXG4gKiAgICAgUm91dGVyTW9kdWxlLmZvclJvb3QoW1xuICogICAgICAge1xuICogICAgICAgICBwYXRoOiAndGVhbS86aWQnLFxuICogICAgICAgICBjb21wb25lbnQ6IFRlYW1Db21wb25lbnQsXG4gKiAgICAgICAgIGNhbkRlYWN0aXZhdGU6IFtDYW5EZWFjdGl2YXRlVGVhbV1cbiAqICAgICAgIH1cbiAqICAgICBdKVxuICogICBdLFxuICogICBwcm92aWRlcnM6IFtDYW5EZWFjdGl2YXRlVGVhbSwgVXNlclRva2VuLCBQZXJtaXNzaW9uc11cbiAqIH0pXG4gKiBjbGFzcyBBcHBNb2R1bGUge31cbiAqIGBgYFxuICpcbiAqIEBwdWJsaWNBcGlcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDYW5EZWFjdGl2YXRlPFQ+IHtcbiAgY2FuRGVhY3RpdmF0ZShcbiAgICBjb21wb25lbnQ6IFQsXG4gICAgY3VycmVudFJvdXRlOiBBY3RpdmF0ZWRSb3V0ZVNuYXBzaG90LFxuICAgIGN1cnJlbnRTdGF0ZTogUm91dGVyU3RhdGVTbmFwc2hvdCxcbiAgICBuZXh0U3RhdGU6IFJvdXRlclN0YXRlU25hcHNob3QsXG4gICk6IE1heWJlQXN5bmM8R3VhcmRSZXN1bHQ+O1xufVxuXG4vKipcbiAqIFRoZSBzaWduYXR1cmUgb2YgYSBmdW5jdGlvbiB1c2VkIGFzIGEgYGNhbkRlYWN0aXZhdGVgIGd1YXJkIG9uIGEgYFJvdXRlYC5cbiAqXG4gKiBJZiBhbGwgZ3VhcmRzIHJldHVybiBgdHJ1ZWAsIG5hdmlnYXRpb24gY29udGludWVzLiBJZiBhbnkgZ3VhcmQgcmV0dXJucyBgZmFsc2VgLFxuICogbmF2aWdhdGlvbiBpcyBjYW5jZWxsZWQuIElmIGFueSBndWFyZCByZXR1cm5zIGEgYFVybFRyZWVgLCB0aGUgY3VycmVudCBuYXZpZ2F0aW9uXG4gKiBpcyBjYW5jZWxsZWQgYW5kIGEgbmV3IG5hdmlnYXRpb24gYmVnaW5zIHRvIHRoZSBgVXJsVHJlZWAgcmV0dXJuZWQgZnJvbSB0aGUgZ3VhcmQuXG4gKlxuICogVGhlIGZvbGxvd2luZyBleGFtcGxlIGltcGxlbWVudHMgYW5kIHVzZXMgYSBgQ2FuRGVhY3RpdmF0ZUZuYCB0aGF0IGNoZWNrcyB3aGV0aGVyIHRoZVxuICogdXNlciBjb21wb25lbnQgaGFzIHVuc2F2ZWQgY2hhbmdlcyBiZWZvcmUgbmF2aWdhdGluZyBhd2F5IGZyb20gdGhlIHJvdXRlLlxuICpcbiAqIHtAZXhhbXBsZSByb3V0ZXIvcm91dGVfZnVuY3Rpb25hbF9ndWFyZHMudHMgcmVnaW9uPVwiQ2FuRGVhY3RpdmF0ZUZuXCJ9XG4gKlxuICogQHB1YmxpY0FwaVxuICogQHNlZSB7QGxpbmsgUm91dGV9XG4gKi9cbmV4cG9ydCB0eXBlIENhbkRlYWN0aXZhdGVGbjxUPiA9IChcbiAgY29tcG9uZW50OiBULFxuICBjdXJyZW50Um91dGU6IEFjdGl2YXRlZFJvdXRlU25hcHNob3QsXG4gIGN1cnJlbnRTdGF0ZTogUm91dGVyU3RhdGVTbmFwc2hvdCxcbiAgbmV4dFN0YXRlOiBSb3V0ZXJTdGF0ZVNuYXBzaG90LFxuKSA9PiBNYXliZUFzeW5jPEd1YXJkUmVzdWx0PjtcblxuLyoqXG4gKiBAZGVzY3JpcHRpb25cbiAqXG4gKiBJbnRlcmZhY2UgdGhhdCBhIGNsYXNzIGNhbiBpbXBsZW1lbnQgdG8gYmUgYSBndWFyZCBkZWNpZGluZyBpZiBhIGBSb3V0ZWAgY2FuIGJlIG1hdGNoZWQuXG4gKiBJZiBhbGwgZ3VhcmRzIHJldHVybiBgdHJ1ZWAsIG5hdmlnYXRpb24gY29udGludWVzIGFuZCB0aGUgYFJvdXRlcmAgd2lsbCB1c2UgdGhlIGBSb3V0ZWAgZHVyaW5nXG4gKiBhY3RpdmF0aW9uLiBJZiBhbnkgZ3VhcmQgcmV0dXJucyBgZmFsc2VgLCB0aGUgYFJvdXRlYCBpcyBza2lwcGVkIGZvciBtYXRjaGluZyBhbmQgb3RoZXIgYFJvdXRlYFxuICogY29uZmlndXJhdGlvbnMgYXJlIHByb2Nlc3NlZCBpbnN0ZWFkLlxuICpcbiAqIFRoZSBmb2xsb3dpbmcgZXhhbXBsZSBpbXBsZW1lbnRzIGEgYENhbk1hdGNoYCBmdW5jdGlvbiB0aGF0IGRlY2lkZXMgd2hldGhlciB0aGVcbiAqIGN1cnJlbnQgdXNlciBoYXMgcGVybWlzc2lvbiB0byBhY2Nlc3MgdGhlIHVzZXJzIHBhZ2UuXG4gKlxuICpcbiAqIGBgYFxuICogY2xhc3MgVXNlclRva2VuIHt9XG4gKiBjbGFzcyBQZXJtaXNzaW9ucyB7XG4gKiAgIGNhbkFjY2Vzcyh1c2VyOiBVc2VyVG9rZW4sIHJvdXRlOiBSb3V0ZSwgc2VnbWVudHM6IFVybFNlZ21lbnRbXSk6IGJvb2xlYW4ge1xuICogICAgIHJldHVybiB0cnVlO1xuICogICB9XG4gKiB9XG4gKlxuICogQEluamVjdGFibGUoKVxuICogY2xhc3MgQ2FuTWF0Y2hUZWFtU2VjdGlvbiBpbXBsZW1lbnRzIENhbk1hdGNoIHtcbiAqICAgY29uc3RydWN0b3IocHJpdmF0ZSBwZXJtaXNzaW9uczogUGVybWlzc2lvbnMsIHByaXZhdGUgY3VycmVudFVzZXI6IFVzZXJUb2tlbikge31cbiAqXG4gKiAgIGNhbk1hdGNoKHJvdXRlOiBSb3V0ZSwgc2VnbWVudHM6IFVybFNlZ21lbnRbXSk6IE9ic2VydmFibGU8Ym9vbGVhbj58UHJvbWlzZTxib29sZWFuPnxib29sZWFuIHtcbiAqICAgICByZXR1cm4gdGhpcy5wZXJtaXNzaW9ucy5jYW5BY2Nlc3ModGhpcy5jdXJyZW50VXNlciwgcm91dGUsIHNlZ21lbnRzKTtcbiAqICAgfVxuICogfVxuICogYGBgXG4gKlxuICogSGVyZSwgdGhlIGRlZmluZWQgZ3VhcmQgZnVuY3Rpb24gaXMgcHJvdmlkZWQgYXMgcGFydCBvZiB0aGUgYFJvdXRlYCBvYmplY3RcbiAqIGluIHRoZSByb3V0ZXIgY29uZmlndXJhdGlvbjpcbiAqXG4gKiBgYGBcbiAqXG4gKiBATmdNb2R1bGUoe1xuICogICBpbXBvcnRzOiBbXG4gKiAgICAgUm91dGVyTW9kdWxlLmZvclJvb3QoW1xuICogICAgICAge1xuICogICAgICAgICBwYXRoOiAndGVhbS86aWQnLFxuICogICAgICAgICBjb21wb25lbnQ6IFRlYW1Db21wb25lbnQsXG4gKiAgICAgICAgIGxvYWRDaGlsZHJlbjogKCkgPT4gaW1wb3J0KCcuL3RlYW0nKS50aGVuKG1vZCA9PiBtb2QuVGVhbU1vZHVsZSksXG4gKiAgICAgICAgIGNhbk1hdGNoOiBbQ2FuTWF0Y2hUZWFtU2VjdGlvbl1cbiAqICAgICAgIH0sXG4gKiAgICAgICB7XG4gKiAgICAgICAgIHBhdGg6ICcqKicsXG4gKiAgICAgICAgIGNvbXBvbmVudDogTm90Rm91bmRDb21wb25lbnRcbiAqICAgICAgIH1cbiAqICAgICBdKVxuICogICBdLFxuICogICBwcm92aWRlcnM6IFtDYW5NYXRjaFRlYW1TZWN0aW9uLCBVc2VyVG9rZW4sIFBlcm1pc3Npb25zXVxuICogfSlcbiAqIGNsYXNzIEFwcE1vZHVsZSB7fVxuICogYGBgXG4gKlxuICogSWYgdGhlIGBDYW5NYXRjaFRlYW1TZWN0aW9uYCB3ZXJlIHRvIHJldHVybiBgZmFsc2VgLCB0aGUgcm91dGVyIHdvdWxkIGNvbnRpbnVlIG5hdmlnYXRpbmcgdG8gdGhlXG4gKiBgdGVhbS86aWRgIFVSTCwgYnV0IHdvdWxkIGxvYWQgdGhlIGBOb3RGb3VuZENvbXBvbmVudGAgYmVjYXVzZSB0aGUgYFJvdXRlYCBmb3IgYCd0ZWFtLzppZCdgXG4gKiBjb3VsZCBub3QgYmUgdXNlZCBmb3IgYSBVUkwgbWF0Y2ggYnV0IHRoZSBjYXRjaC1hbGwgYCoqYCBgUm91dGVgIGRpZCBpbnN0ZWFkLlxuICpcbiAqIEBwdWJsaWNBcGlcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDYW5NYXRjaCB7XG4gIGNhbk1hdGNoKHJvdXRlOiBSb3V0ZSwgc2VnbWVudHM6IFVybFNlZ21lbnRbXSk6IE1heWJlQXN5bmM8R3VhcmRSZXN1bHQ+O1xufVxuXG4vKipcbiAqIFRoZSBzaWduYXR1cmUgb2YgYSBmdW5jdGlvbiB1c2VkIGFzIGEgYGNhbk1hdGNoYCBndWFyZCBvbiBhIGBSb3V0ZWAuXG4gKlxuICogSWYgYWxsIGd1YXJkcyByZXR1cm4gYHRydWVgLCBuYXZpZ2F0aW9uIGNvbnRpbnVlcyBhbmQgdGhlIGBSb3V0ZXJgIHdpbGwgdXNlIHRoZSBgUm91dGVgIGR1cmluZ1xuICogYWN0aXZhdGlvbi4gSWYgYW55IGd1YXJkIHJldHVybnMgYGZhbHNlYCwgdGhlIGBSb3V0ZWAgaXMgc2tpcHBlZCBmb3IgbWF0Y2hpbmcgYW5kIG90aGVyIGBSb3V0ZWBcbiAqIGNvbmZpZ3VyYXRpb25zIGFyZSBwcm9jZXNzZWQgaW5zdGVhZC5cbiAqXG4gKiBUaGUgZm9sbG93aW5nIGV4YW1wbGUgaW1wbGVtZW50cyBhbmQgdXNlcyBhIGBDYW5NYXRjaEZuYCB0aGF0IGNoZWNrcyB3aGV0aGVyIHRoZVxuICogY3VycmVudCB1c2VyIGhhcyBwZXJtaXNzaW9uIHRvIGFjY2VzcyB0aGUgdGVhbSBwYWdlLlxuICpcbiAqIHtAZXhhbXBsZSByb3V0ZXIvcm91dGVfZnVuY3Rpb25hbF9ndWFyZHMudHMgcmVnaW9uPVwiQ2FuTWF0Y2hGblwifVxuICpcbiAqIEBwdWJsaWNBcGlcbiAqIEBzZWUge0BsaW5rIFJvdXRlfVxuICovXG5leHBvcnQgdHlwZSBDYW5NYXRjaEZuID0gKHJvdXRlOiBSb3V0ZSwgc2VnbWVudHM6IFVybFNlZ21lbnRbXSkgPT4gTWF5YmVBc3luYzxHdWFyZFJlc3VsdD47XG5cbi8qKlxuICogQGRlc2NyaXB0aW9uXG4gKlxuICogSW50ZXJmYWNlIHRoYXQgY2xhc3NlcyBjYW4gaW1wbGVtZW50IHRvIGJlIGEgZGF0YSBwcm92aWRlci5cbiAqIEEgZGF0YSBwcm92aWRlciBjbGFzcyBjYW4gYmUgdXNlZCB3aXRoIHRoZSByb3V0ZXIgdG8gcmVzb2x2ZSBkYXRhIGR1cmluZyBuYXZpZ2F0aW9uLlxuICogVGhlIGludGVyZmFjZSBkZWZpbmVzIGEgYHJlc29sdmUoKWAgbWV0aG9kIHRoYXQgaXMgaW52b2tlZCByaWdodCBhZnRlciB0aGUgYFJlc29sdmVTdGFydGBcbiAqIHJvdXRlciBldmVudC4gVGhlIHJvdXRlciB3YWl0cyBmb3IgdGhlIGRhdGEgdG8gYmUgcmVzb2x2ZWQgYmVmb3JlIHRoZSByb3V0ZSBpcyBmaW5hbGx5IGFjdGl2YXRlZC5cbiAqXG4gKiBUaGUgZm9sbG93aW5nIGV4YW1wbGUgaW1wbGVtZW50cyBhIGByZXNvbHZlKClgIG1ldGhvZCB0aGF0IHJldHJpZXZlcyB0aGUgZGF0YVxuICogbmVlZGVkIHRvIGFjdGl2YXRlIHRoZSByZXF1ZXN0ZWQgcm91dGUuXG4gKlxuICogYGBgXG4gKiBASW5qZWN0YWJsZSh7IHByb3ZpZGVkSW46ICdyb290JyB9KVxuICogZXhwb3J0IGNsYXNzIEhlcm9SZXNvbHZlciBpbXBsZW1lbnRzIFJlc29sdmU8SGVybz4ge1xuICogICBjb25zdHJ1Y3Rvcihwcml2YXRlIHNlcnZpY2U6IEhlcm9TZXJ2aWNlKSB7fVxuICpcbiAqICAgcmVzb2x2ZShcbiAqICAgICByb3V0ZTogQWN0aXZhdGVkUm91dGVTbmFwc2hvdCxcbiAqICAgICBzdGF0ZTogUm91dGVyU3RhdGVTbmFwc2hvdFxuICogICApOiBPYnNlcnZhYmxlPEhlcm8+fFByb21pc2U8SGVybz58SGVybyB7XG4gKiAgICAgcmV0dXJuIHRoaXMuc2VydmljZS5nZXRIZXJvKHJvdXRlLnBhcmFtTWFwLmdldCgnaWQnKSk7XG4gKiAgIH1cbiAqIH1cbiAqIGBgYFxuICpcbiAqIEhlcmUsIHRoZSBkZWZpbmVkIGByZXNvbHZlKClgIGZ1bmN0aW9uIGlzIHByb3ZpZGVkIGFzIHBhcnQgb2YgdGhlIGBSb3V0ZWAgb2JqZWN0XG4gKiBpbiB0aGUgcm91dGVyIGNvbmZpZ3VyYXRpb246XG4gKlxuICogYGBgXG5cbiAqIEBOZ01vZHVsZSh7XG4gKiAgIGltcG9ydHM6IFtcbiAqICAgICBSb3V0ZXJNb2R1bGUuZm9yUm9vdChbXG4gKiAgICAgICB7XG4gKiAgICAgICAgIHBhdGg6ICdkZXRhaWwvOmlkJyxcbiAqICAgICAgICAgY29tcG9uZW50OiBIZXJvRGV0YWlsQ29tcG9uZW50LFxuICogICAgICAgICByZXNvbHZlOiB7XG4gKiAgICAgICAgICAgaGVybzogSGVyb1Jlc29sdmVyXG4gKiAgICAgICAgIH1cbiAqICAgICAgIH1cbiAqICAgICBdKVxuICogICBdLFxuICogICBleHBvcnRzOiBbUm91dGVyTW9kdWxlXVxuICogfSlcbiAqIGV4cG9ydCBjbGFzcyBBcHBSb3V0aW5nTW9kdWxlIHt9XG4gKiBgYGBcbiAqXG4gKiBBbmQgeW91IGNhbiBhY2Nlc3MgdG8geW91ciByZXNvbHZlZCBkYXRhIGZyb20gYEhlcm9Db21wb25lbnRgOlxuICpcbiAqIGBgYFxuICogQENvbXBvbmVudCh7XG4gKiAgc2VsZWN0b3I6IFwiYXBwLWhlcm9cIixcbiAqICB0ZW1wbGF0ZVVybDogXCJoZXJvLmNvbXBvbmVudC5odG1sXCIsXG4gKiB9KVxuICogZXhwb3J0IGNsYXNzIEhlcm9Db21wb25lbnQge1xuICpcbiAqICBjb25zdHJ1Y3Rvcihwcml2YXRlIGFjdGl2YXRlZFJvdXRlOiBBY3RpdmF0ZWRSb3V0ZSkge31cbiAqXG4gKiAgbmdPbkluaXQoKSB7XG4gKiAgICB0aGlzLmFjdGl2YXRlZFJvdXRlLmRhdGEuc3Vic2NyaWJlKCh7IGhlcm8gfSkgPT4ge1xuICogICAgICAvLyBkbyBzb21ldGhpbmcgd2l0aCB5b3VyIHJlc29sdmVkIGRhdGEgLi4uXG4gKiAgICB9KVxuICogIH1cbiAqXG4gKiB9XG4gKiBgYGBcbiAqXG4gKiBAdXNhZ2VOb3Rlc1xuICpcbiAqIFdoZW4gYm90aCBndWFyZCBhbmQgcmVzb2x2ZXJzIGFyZSBzcGVjaWZpZWQsIHRoZSByZXNvbHZlcnMgYXJlIG5vdCBleGVjdXRlZCB1bnRpbFxuICogYWxsIGd1YXJkcyBoYXZlIHJ1biBhbmQgc3VjY2VlZGVkLlxuICogRm9yIGV4YW1wbGUsIGNvbnNpZGVyIHRoZSBmb2xsb3dpbmcgcm91dGUgY29uZmlndXJhdGlvbjpcbiAqXG4gKiBgYGBcbiAqIHtcbiAqICBwYXRoOiAnYmFzZSdcbiAqICBjYW5BY3RpdmF0ZTogW0Jhc2VHdWFyZF0sXG4gKiAgcmVzb2x2ZToge2RhdGE6IEJhc2VEYXRhUmVzb2x2ZXJ9XG4gKiAgY2hpbGRyZW46IFtcbiAqICAge1xuICogICAgIHBhdGg6ICdjaGlsZCcsXG4gKiAgICAgZ3VhcmRzOiBbQ2hpbGRHdWFyZF0sXG4gKiAgICAgY29tcG9uZW50OiBDaGlsZENvbXBvbmVudCxcbiAqICAgICByZXNvbHZlOiB7Y2hpbGREYXRhOiBDaGlsZERhdGFSZXNvbHZlcn1cbiAqICAgIH1cbiAqICBdXG4gKiB9XG4gKiBgYGBcbiAqIFRoZSBvcmRlciBvZiBleGVjdXRpb24gaXM6IEJhc2VHdWFyZCwgQ2hpbGRHdWFyZCwgQmFzZURhdGFSZXNvbHZlciwgQ2hpbGREYXRhUmVzb2x2ZXIuXG4gKlxuICogQHB1YmxpY0FwaVxuICogQHNlZSB7QGxpbmsgUmVzb2x2ZUZufVxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJlc29sdmU8VD4ge1xuICByZXNvbHZlKHJvdXRlOiBBY3RpdmF0ZWRSb3V0ZVNuYXBzaG90LCBzdGF0ZTogUm91dGVyU3RhdGVTbmFwc2hvdCk6IE1heWJlQXN5bmM8VD47XG59XG5cbi8qKlxuICogRnVuY3Rpb24gdHlwZSBkZWZpbml0aW9uIGZvciBhIGRhdGEgcHJvdmlkZXIuXG4gKlxuICogQSBkYXRhIHByb3ZpZGVyIGNhbiBiZSB1c2VkIHdpdGggdGhlIHJvdXRlciB0byByZXNvbHZlIGRhdGEgZHVyaW5nIG5hdmlnYXRpb24uXG4gKiBUaGUgcm91dGVyIHdhaXRzIGZvciB0aGUgZGF0YSB0byBiZSByZXNvbHZlZCBiZWZvcmUgdGhlIHJvdXRlIGlzIGZpbmFsbHkgYWN0aXZhdGVkLlxuICpcbiAqIEEgcmVzb2x2ZXIgY2FuIGFsc28gcmVkaXJlY3QgYSBgUmVkaXJlY3RDb21tYW5kYCBhbmQgdGhlIEFuZ3VsYXIgcm91dGVyIHdpbGwgdXNlXG4gKiBpdCB0byByZWRpcmVjdCB0aGUgY3VycmVudCBuYXZpZ2F0aW9uIHRvIHRoZSBuZXcgZGVzdGluYXRpb24uXG4gKlxuICogQHVzYWdlTm90ZXNcbiAqXG4gKiBUaGUgZm9sbG93aW5nIGV4YW1wbGUgaW1wbGVtZW50cyBhIGZ1bmN0aW9uIHRoYXQgcmV0cmlldmVzIHRoZSBkYXRhXG4gKiBuZWVkZWQgdG8gYWN0aXZhdGUgdGhlIHJlcXVlc3RlZCByb3V0ZS5cbiAqXG4gKiBgYGB0c1xuICogaW50ZXJmYWNlIEhlcm8ge1xuICogICBuYW1lOiBzdHJpbmc7XG4gKiB9XG4gKiBASW5qZWN0YWJsZSgpXG4gKiBleHBvcnQgY2xhc3MgSGVyb1NlcnZpY2Uge1xuICogICBnZXRIZXJvKGlkOiBzdHJpbmcpIHtcbiAqICAgICByZXR1cm4ge25hbWU6IGBTdXBlcm1hbi0ke2lkfWB9O1xuICogICB9XG4gKiB9XG4gKlxuICogZXhwb3J0IGNvbnN0IGhlcm9SZXNvbHZlcjogUmVzb2x2ZUZuPEhlcm8+ID0gKFxuICogICByb3V0ZTogQWN0aXZhdGVkUm91dGVTbmFwc2hvdCxcbiAqICAgc3RhdGU6IFJvdXRlclN0YXRlU25hcHNob3QsXG4gKiApID0+IHtcbiAqICAgcmV0dXJuIGluamVjdChIZXJvU2VydmljZSkuZ2V0SGVybyhyb3V0ZS5wYXJhbU1hcC5nZXQoJ2lkJykhKTtcbiAqIH07XG4gKlxuICogYm9vdHN0cmFwQXBwbGljYXRpb24oQXBwLCB7XG4gKiAgIHByb3ZpZGVyczogW1xuICogICAgIHByb3ZpZGVSb3V0ZXIoW1xuICogICAgICAge1xuICogICAgICAgICBwYXRoOiAnZGV0YWlsLzppZCcsXG4gKiAgICAgICAgIGNvbXBvbmVudDogSGVyb0RldGFpbENvbXBvbmVudCxcbiAqICAgICAgICAgcmVzb2x2ZToge2hlcm86IGhlcm9SZXNvbHZlcn0sXG4gKiAgICAgICB9LFxuICogICAgIF0pLFxuICogICBdLFxuICogfSk7XG4gKiBgYGBcbiAqXG4gKiBBbmQgeW91IGNhbiBhY2Nlc3MgdG8geW91ciByZXNvbHZlZCBkYXRhIGZyb20gYEhlcm9Db21wb25lbnRgOlxuICpcbiAqIGBgYHRzXG4gKiBAQ29tcG9uZW50KHt0ZW1wbGF0ZTogJyd9KVxuICogZXhwb3J0IGNsYXNzIEhlcm9EZXRhaWxDb21wb25lbnQge1xuICogICBwcml2YXRlIGFjdGl2YXRlZFJvdXRlID0gaW5qZWN0KEFjdGl2YXRlZFJvdXRlKTtcbiAqXG4gKiAgIG5nT25Jbml0KCkge1xuICogICAgIHRoaXMuYWN0aXZhdGVkUm91dGUuZGF0YS5zdWJzY3JpYmUoKHtoZXJvfSkgPT4ge1xuICogICAgICAgLy8gZG8gc29tZXRoaW5nIHdpdGggeW91ciByZXNvbHZlZCBkYXRhIC4uLlxuICogICAgIH0pO1xuICogICB9XG4gKiB9XG4gKiBgYGBcbiAqXG4gKiBJZiByZXNvbHZlZCBkYXRhIGNhbm5vdCBiZSByZXRyaWV2ZWQsIHlvdSBtYXkgd2FudCB0byByZWRpcmVjdCB0aGUgdXNlclxuICogdG8gYSBuZXcgcGFnZSBpbnN0ZWFkOlxuICpcbiAqIGBgYHRzXG4gKiBleHBvcnQgY29uc3QgaGVyb1Jlc29sdmVyOiBSZXNvbHZlRm48SGVybz4gPSBhc3luYyAoXG4gKiAgIHJvdXRlOiBBY3RpdmF0ZWRSb3V0ZVNuYXBzaG90LFxuICogICBzdGF0ZTogUm91dGVyU3RhdGVTbmFwc2hvdCxcbiAqICkgPT4ge1xuICogICBjb25zdCByb3V0ZXIgPSBpbmplY3QoUm91dGVyKTtcbiAqICAgY29uc3QgaGVyb1NlcnZpY2UgPSBpbmplY3QoSGVyb1NlcnZpY2UpO1xuICogICB0cnkge1xuICogICAgIHJldHVybiBhd2FpdCBoZXJvU2VydmljZS5nZXRIZXJvKHJvdXRlLnBhcmFtTWFwLmdldCgnaWQnKSEpO1xuICogICB9IGNhdGNoIHtcbiAqICAgICByZXR1cm4gbmV3IFJlZGlyZWN0Q29tbWFuZChyb3V0ZXIucGFyc2VVcmwoJy80MDQnKSk7XG4gKiAgIH1cbiAqIH07XG4gKiBgYGBcbiAqXG4gKiBXaGVuIGJvdGggZ3VhcmQgYW5kIHJlc29sdmVycyBhcmUgc3BlY2lmaWVkLCB0aGUgcmVzb2x2ZXJzIGFyZSBub3QgZXhlY3V0ZWQgdW50aWxcbiAqIGFsbCBndWFyZHMgaGF2ZSBydW4gYW5kIHN1Y2NlZWRlZC5cbiAqIEZvciBleGFtcGxlLCBjb25zaWRlciB0aGUgZm9sbG93aW5nIHJvdXRlIGNvbmZpZ3VyYXRpb246XG4gKlxuICogYGBgXG4gKiB7XG4gKiAgcGF0aDogJ2Jhc2UnXG4gKiAgY2FuQWN0aXZhdGU6IFtiYXNlR3VhcmRdLFxuICogIHJlc29sdmU6IHtkYXRhOiBiYXNlRGF0YVJlc29sdmVyfVxuICogIGNoaWxkcmVuOiBbXG4gKiAgIHtcbiAqICAgICBwYXRoOiAnY2hpbGQnLFxuICogICAgIGNhbkFjdGl2YXRlOiBbY2hpbGRHdWFyZF0sXG4gKiAgICAgY29tcG9uZW50OiBDaGlsZENvbXBvbmVudCxcbiAqICAgICByZXNvbHZlOiB7Y2hpbGREYXRhOiBjaGlsZERhdGFSZXNvbHZlcn1cbiAqICAgIH1cbiAqICBdXG4gKiB9XG4gKiBgYGBcbiAqIFRoZSBvcmRlciBvZiBleGVjdXRpb24gaXM6IGJhc2VHdWFyZCwgY2hpbGRHdWFyZCwgYmFzZURhdGFSZXNvbHZlciwgY2hpbGREYXRhUmVzb2x2ZXIuXG4gKlxuICogQHB1YmxpY0FwaVxuICogQHNlZSB7QGxpbmsgUm91dGV9XG4gKi9cbmV4cG9ydCB0eXBlIFJlc29sdmVGbjxUPiA9IChcbiAgcm91dGU6IEFjdGl2YXRlZFJvdXRlU25hcHNob3QsXG4gIHN0YXRlOiBSb3V0ZXJTdGF0ZVNuYXBzaG90LFxuKSA9PiBNYXliZUFzeW5jPFQgfCBSZWRpcmVjdENvbW1hbmQ+O1xuXG4vKipcbiAqIEBkZXNjcmlwdGlvblxuICpcbiAqIEludGVyZmFjZSB0aGF0IGEgY2xhc3MgY2FuIGltcGxlbWVudCB0byBiZSBhIGd1YXJkIGRlY2lkaW5nIGlmIGNoaWxkcmVuIGNhbiBiZSBsb2FkZWQuXG4gKiBJZiBhbGwgZ3VhcmRzIHJldHVybiBgdHJ1ZWAsIG5hdmlnYXRpb24gY29udGludWVzLiBJZiBhbnkgZ3VhcmQgcmV0dXJucyBgZmFsc2VgLFxuICogbmF2aWdhdGlvbiBpcyBjYW5jZWxsZWQuIElmIGFueSBndWFyZCByZXR1cm5zIGEgYFVybFRyZWVgLCBjdXJyZW50IG5hdmlnYXRpb25cbiAqIGlzIGNhbmNlbGxlZCBhbmQgYSBuZXcgbmF2aWdhdGlvbiBzdGFydHMgdG8gdGhlIGBVcmxUcmVlYCByZXR1cm5lZCBmcm9tIHRoZSBndWFyZC5cbiAqXG4gKiBUaGUgZm9sbG93aW5nIGV4YW1wbGUgaW1wbGVtZW50cyBhIGBDYW5Mb2FkYCBmdW5jdGlvbiB0aGF0IGRlY2lkZXMgd2hldGhlciB0aGVcbiAqIGN1cnJlbnQgdXNlciBoYXMgcGVybWlzc2lvbiB0byBsb2FkIHJlcXVlc3RlZCBjaGlsZCByb3V0ZXMuXG4gKlxuICpcbiAqIGBgYFxuICogY2xhc3MgVXNlclRva2VuIHt9XG4gKiBjbGFzcyBQZXJtaXNzaW9ucyB7XG4gKiAgIGNhbkxvYWRDaGlsZHJlbih1c2VyOiBVc2VyVG9rZW4sIGlkOiBzdHJpbmcsIHNlZ21lbnRzOiBVcmxTZWdtZW50W10pOiBib29sZWFuIHtcbiAqICAgICByZXR1cm4gdHJ1ZTtcbiAqICAgfVxuICogfVxuICpcbiAqIEBJbmplY3RhYmxlKClcbiAqIGNsYXNzIENhbkxvYWRUZWFtU2VjdGlvbiBpbXBsZW1lbnRzIENhbkxvYWQge1xuICogICBjb25zdHJ1Y3Rvcihwcml2YXRlIHBlcm1pc3Npb25zOiBQZXJtaXNzaW9ucywgcHJpdmF0ZSBjdXJyZW50VXNlcjogVXNlclRva2VuKSB7fVxuICpcbiAqICAgY2FuTG9hZChyb3V0ZTogUm91dGUsIHNlZ21lbnRzOiBVcmxTZWdtZW50W10pOiBPYnNlcnZhYmxlPGJvb2xlYW4+fFByb21pc2U8Ym9vbGVhbj58Ym9vbGVhbiB7XG4gKiAgICAgcmV0dXJuIHRoaXMucGVybWlzc2lvbnMuY2FuTG9hZENoaWxkcmVuKHRoaXMuY3VycmVudFVzZXIsIHJvdXRlLCBzZWdtZW50cyk7XG4gKiAgIH1cbiAqIH1cbiAqIGBgYFxuICpcbiAqIEhlcmUsIHRoZSBkZWZpbmVkIGd1YXJkIGZ1bmN0aW9uIGlzIHByb3ZpZGVkIGFzIHBhcnQgb2YgdGhlIGBSb3V0ZWAgb2JqZWN0XG4gKiBpbiB0aGUgcm91dGVyIGNvbmZpZ3VyYXRpb246XG4gKlxuICogYGBgXG4gKlxuICogQE5nTW9kdWxlKHtcbiAqICAgaW1wb3J0czogW1xuICogICAgIFJvdXRlck1vZHVsZS5mb3JSb290KFtcbiAqICAgICAgIHtcbiAqICAgICAgICAgcGF0aDogJ3RlYW0vOmlkJyxcbiAqICAgICAgICAgY29tcG9uZW50OiBUZWFtQ29tcG9uZW50LFxuICogICAgICAgICBsb2FkQ2hpbGRyZW46ICgpID0+IGltcG9ydCgnLi90ZWFtJykudGhlbihtb2QgPT4gbW9kLlRlYW1Nb2R1bGUpLFxuICogICAgICAgICBjYW5Mb2FkOiBbQ2FuTG9hZFRlYW1TZWN0aW9uXVxuICogICAgICAgfVxuICogICAgIF0pXG4gKiAgIF0sXG4gKiAgIHByb3ZpZGVyczogW0NhbkxvYWRUZWFtU2VjdGlvbiwgVXNlclRva2VuLCBQZXJtaXNzaW9uc11cbiAqIH0pXG4gKiBjbGFzcyBBcHBNb2R1bGUge31cbiAqIGBgYFxuICpcbiAqIEBwdWJsaWNBcGlcbiAqIEBkZXByZWNhdGVkIFVzZSB7QGxpbmsgQ2FuTWF0Y2h9IGluc3RlYWRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDYW5Mb2FkIHtcbiAgY2FuTG9hZChyb3V0ZTogUm91dGUsIHNlZ21lbnRzOiBVcmxTZWdtZW50W10pOiBNYXliZUFzeW5jPEd1YXJkUmVzdWx0Pjtcbn1cblxuLyoqXG4gKiBUaGUgc2lnbmF0dXJlIG9mIGEgZnVuY3Rpb24gdXNlZCBhcyBhIGBjYW5Mb2FkYCBndWFyZCBvbiBhIGBSb3V0ZWAuXG4gKlxuICogQHB1YmxpY0FwaVxuICogQHNlZSB7QGxpbmsgQ2FuTG9hZH1cbiAqIEBzZWUge0BsaW5rIFJvdXRlfVxuICogQHNlZSB7QGxpbmsgQ2FuTWF0Y2h9XG4gKiBAZGVwcmVjYXRlZCBVc2UgYFJvdXRlLmNhbk1hdGNoYCBhbmQgYENhbk1hdGNoRm5gIGluc3RlYWRcbiAqL1xuZXhwb3J0IHR5cGUgQ2FuTG9hZEZuID0gKHJvdXRlOiBSb3V0ZSwgc2VnbWVudHM6IFVybFNlZ21lbnRbXSkgPT4gTWF5YmVBc3luYzxHdWFyZFJlc3VsdD47XG5cbi8qKlxuICogQGRlc2NyaXB0aW9uXG4gKlxuICogT3B0aW9ucyB0aGF0IG1vZGlmeSB0aGUgYFJvdXRlcmAgbmF2aWdhdGlvbiBzdHJhdGVneS5cbiAqIFN1cHBseSBhbiBvYmplY3QgY29udGFpbmluZyBhbnkgb2YgdGhlc2UgcHJvcGVydGllcyB0byBhIGBSb3V0ZXJgIG5hdmlnYXRpb24gZnVuY3Rpb24gdG9cbiAqIGNvbnRyb2wgaG93IHRoZSBuYXZpZ2F0aW9uIHNob3VsZCBiZSBoYW5kbGVkLlxuICpcbiAqIEBzZWUge0BsaW5rIFJvdXRlciNuYXZpZ2F0ZX1cbiAqIEBzZWUge0BsaW5rIFJvdXRlciNuYXZpZ2F0ZUJ5VXJsfVxuICogQHNlZSBbUm91dGluZyBhbmQgTmF2aWdhdGlvbiBndWlkZV0oZ3VpZGUvcm91dGluZy9jb21tb24tcm91dGVyLXRhc2tzKVxuICpcbiAqIEBwdWJsaWNBcGlcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBOYXZpZ2F0aW9uQmVoYXZpb3JPcHRpb25zIHtcbiAgLyoqXG4gICAqIEhvdyB0byBoYW5kbGUgYSBuYXZpZ2F0aW9uIHJlcXVlc3QgdG8gdGhlIGN1cnJlbnQgVVJMLlxuICAgKlxuICAgKiBUaGlzIHZhbHVlIGlzIGEgc3Vic2V0IG9mIHRoZSBvcHRpb25zIGF2YWlsYWJsZSBpbiBgT25TYW1lVXJsTmF2aWdhdGlvbmAgYW5kXG4gICAqIHdpbGwgdGFrZSBwcmVjZWRlbmNlIG92ZXIgdGhlIGRlZmF1bHQgdmFsdWUgc2V0IGZvciB0aGUgYFJvdXRlcmAuXG4gICAqXG4gICAqIEBzZWUge0BsaW5rIE9uU2FtZVVybE5hdmlnYXRpb259XG4gICAqIEBzZWUge0BsaW5rIFJvdXRlckNvbmZpZ09wdGlvbnN9XG4gICAqL1xuICBvblNhbWVVcmxOYXZpZ2F0aW9uPzogT25TYW1lVXJsTmF2aWdhdGlvbjtcblxuICAvKipcbiAgICogV2hlbiB0cnVlLCBuYXZpZ2F0ZXMgd2l0aG91dCBwdXNoaW5nIGEgbmV3IHN0YXRlIGludG8gaGlzdG9yeS5cbiAgICpcbiAgICogYGBgXG4gICAqIC8vIE5hdmlnYXRlIHNpbGVudGx5IHRvIC92aWV3XG4gICAqIHRoaXMucm91dGVyLm5hdmlnYXRlKFsnL3ZpZXcnXSwgeyBza2lwTG9jYXRpb25DaGFuZ2U6IHRydWUgfSk7XG4gICAqIGBgYFxuICAgKi9cbiAgc2tpcExvY2F0aW9uQ2hhbmdlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogV2hlbiB0cnVlLCBuYXZpZ2F0ZXMgd2hpbGUgcmVwbGFjaW5nIHRoZSBjdXJyZW50IHN0YXRlIGluIGhpc3RvcnkuXG4gICAqXG4gICAqIGBgYFxuICAgKiAvLyBOYXZpZ2F0ZSB0byAvdmlld1xuICAgKiB0aGlzLnJvdXRlci5uYXZpZ2F0ZShbJy92aWV3J10sIHsgcmVwbGFjZVVybDogdHJ1ZSB9KTtcbiAgICogYGBgXG4gICAqL1xuICByZXBsYWNlVXJsPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogRGV2ZWxvcGVyLWRlZmluZWQgc3RhdGUgdGhhdCBjYW4gYmUgcGFzc2VkIHRvIGFueSBuYXZpZ2F0aW9uLlxuICAgKiBBY2Nlc3MgdGhpcyB2YWx1ZSB0aHJvdWdoIHRoZSBgTmF2aWdhdGlvbi5leHRyYXNgIG9iamVjdFxuICAgKiByZXR1cm5lZCBmcm9tIHRoZSBbUm91dGVyLmdldEN1cnJlbnROYXZpZ2F0aW9uKClcbiAgICogbWV0aG9kXShhcGkvcm91dGVyL1JvdXRlciNnZXRjdXJyZW50bmF2aWdhdGlvbikgd2hpbGUgYSBuYXZpZ2F0aW9uIGlzIGV4ZWN1dGluZy5cbiAgICpcbiAgICogQWZ0ZXIgYSBuYXZpZ2F0aW9uIGNvbXBsZXRlcywgdGhlIHJvdXRlciB3cml0ZXMgYW4gb2JqZWN0IGNvbnRhaW5pbmcgdGhpc1xuICAgKiB2YWx1ZSB0b2dldGhlciB3aXRoIGEgYG5hdmlnYXRpb25JZGAgdG8gYGhpc3Rvcnkuc3RhdGVgLlxuICAgKiBUaGUgdmFsdWUgaXMgd3JpdHRlbiB3aGVuIGBsb2NhdGlvbi5nbygpYCBvciBgbG9jYXRpb24ucmVwbGFjZVN0YXRlKClgXG4gICAqIGlzIGNhbGxlZCBiZWZvcmUgYWN0aXZhdGluZyB0aGlzIHJvdXRlLlxuICAgKlxuICAgKiBOb3RlIHRoYXQgYGhpc3Rvcnkuc3RhdGVgIGRvZXMgbm90IHBhc3MgYW4gb2JqZWN0IGVxdWFsaXR5IHRlc3QgYmVjYXVzZVxuICAgKiB0aGUgcm91dGVyIGFkZHMgdGhlIGBuYXZpZ2F0aW9uSWRgIG9uIGVhY2ggbmF2aWdhdGlvbi5cbiAgICpcbiAgICovXG4gIHN0YXRlPzoge1trOiBzdHJpbmddOiBhbnl9O1xuXG4gIC8qKlxuICAgKiBVc2UgdGhpcyB0byBjb252ZXkgdHJhbnNpZW50IGluZm9ybWF0aW9uIGFib3V0IHRoaXMgcGFydGljdWxhciBuYXZpZ2F0aW9uLCBzdWNoIGFzIGhvdyBpdFxuICAgKiBoYXBwZW5lZC4gSW4gdGhpcyB3YXksIGl0J3MgZGlmZmVyZW50IGZyb20gdGhlIHBlcnNpc3RlZCB2YWx1ZSBgc3RhdGVgIHRoYXQgd2lsbCBiZSBzZXQgdG9cbiAgICogYGhpc3Rvcnkuc3RhdGVgLiBUaGlzIG9iamVjdCBpcyBhc3NpZ25lZCBkaXJlY3RseSB0byB0aGUgUm91dGVyJ3MgY3VycmVudCBgTmF2aWdhdGlvbmBcbiAgICogKGl0IGlzIG5vdCBjb3BpZWQgb3IgY2xvbmVkKSwgc28gaXQgc2hvdWxkIGJlIG11dGF0ZWQgd2l0aCBjYXV0aW9uLlxuICAgKlxuICAgKiBPbmUgZXhhbXBsZSBvZiBob3cgdGhpcyBtaWdodCBiZSB1c2VkIGlzIHRvIHRyaWdnZXIgZGlmZmVyZW50IHNpbmdsZS1wYWdlIG5hdmlnYXRpb24gYW5pbWF0aW9uc1xuICAgKiBkZXBlbmRpbmcgb24gaG93IGEgY2VydGFpbiByb3V0ZSB3YXMgcmVhY2hlZC4gRm9yIGV4YW1wbGUsIGNvbnNpZGVyIGEgcGhvdG8gZ2FsbGVyeSBhcHAsIHdoZXJlXG4gICAqIHlvdSBjYW4gcmVhY2ggdGhlIHNhbWUgcGhvdG8gVVJMIGFuZCBzdGF0ZSB2aWEgdmFyaW91cyByb3V0ZXM6XG4gICAqXG4gICAqIC0gQ2xpY2tpbmcgb24gaXQgaW4gYSBnYWxsZXJ5IHZpZXdcbiAgICogLSBDbGlja2luZ1xuICAgKiAtIFwibmV4dFwiIG9yIFwicHJldmlvdXNcIiB3aGVuIHZpZXdpbmcgYW5vdGhlciBwaG90byBpbiB0aGUgYWxidW1cbiAgICogLSBFdGMuXG4gICAqXG4gICAqIEVhY2ggb2YgdGhlc2Ugd2FudHMgYSBkaWZmZXJlbnQgYW5pbWF0aW9uIGF0IG5hdmlnYXRlIHRpbWUuIFRoaXMgaW5mb3JtYXRpb24gZG9lc24ndCBtYWtlIHNlbnNlXG4gICAqIHRvIHN0b3JlIGluIHRoZSBwZXJzaXN0ZW50IFVSTCBvciBoaXN0b3J5IGVudHJ5IHN0YXRlLCBidXQgaXQncyBzdGlsbCBpbXBvcnRhbnQgdG8gY29tbXVuaWNhdGVcbiAgICogZnJvbSB0aGUgcmVzdCBvZiB0aGUgYXBwbGljYXRpb24sIGludG8gdGhlIHJvdXRlci5cbiAgICpcbiAgICogVGhpcyBpbmZvcm1hdGlvbiBjb3VsZCBiZSB1c2VkIGluIGNvb3JkaW5hdGlvbiB3aXRoIHRoZSBWaWV3IFRyYW5zaXRpb25zIGZlYXR1cmUgYW5kIHRoZVxuICAgKiBgb25WaWV3VHJhbnNpdGlvbkNyZWF0ZWRgIGNhbGxiYWNrLiBUaGUgaW5mb3JtYXRpb24gbWlnaHQgYmUgdXNlZCBpbiB0aGUgY2FsbGJhY2sgdG8gc2V0XG4gICAqIGNsYXNzZXMgb24gdGhlIGRvY3VtZW50IGluIG9yZGVyIHRvIGNvbnRyb2wgdGhlIHRyYW5zaXRpb24gYW5pbWF0aW9ucyBhbmQgcmVtb3ZlIHRoZSBjbGFzc2VzXG4gICAqIHdoZW4gdGhlIHRyYW5zaXRpb24gaGFzIGZpbmlzaGVkIGFuaW1hdGluZy5cbiAgICovXG4gIHJlYWRvbmx5IGluZm8/OiB1bmtub3duO1xuXG4gIC8qKlxuICAgKiBXaGVuIHNldCwgdGhlIFJvdXRlciB3aWxsIHVwZGF0ZSB0aGUgYnJvd3NlcidzIGFkZHJlc3MgYmFyIHRvIG1hdGNoIHRoZSBnaXZlbiBgVXJsVHJlZWAgaW5zdGVhZFxuICAgKiBvZiB0aGUgb25lIHVzZWQgZm9yIHJvdXRlIG1hdGNoaW5nLlxuICAgKlxuICAgKlxuICAgKiBAdXNhZ2VOb3Rlc1xuICAgKlxuICAgKiBUaGlzIGZlYXR1cmUgaXMgdXNlZnVsIGZvciByZWRpcmVjdHMsIHN1Y2ggYXMgcmVkaXJlY3RpbmcgdG8gYW4gZXJyb3IgcGFnZSwgd2l0aG91dCBjaGFuZ2luZ1xuICAgKiB0aGUgdmFsdWUgdGhhdCB3aWxsIGJlIGRpc3BsYXllZCBpbiB0aGUgYnJvd3NlcidzIGFkZHJlc3MgYmFyLlxuICAgKlxuICAgKiBgYGBcbiAgICogY29uc3QgY2FuQWN0aXZhdGU6IENhbkFjdGl2YXRlRm4gPSAocm91dGU6IEFjdGl2YXRlZFJvdXRlU25hcHNob3QpID0+IHtcbiAgICogICBjb25zdCB1c2VyU2VydmljZSA9IGluamVjdChVc2VyU2VydmljZSk7XG4gICAqICAgY29uc3Qgcm91dGVyID0gaW5qZWN0KFJvdXRlcik7XG4gICAqICAgaWYgKCF1c2VyU2VydmljZS5pc0xvZ2dlZEluKCkpIHtcbiAgICogICAgIGNvbnN0IHRhcmdldE9mQ3VycmVudE5hdmlnYXRpb24gPSByb3V0ZXIuZ2V0Q3VycmVudE5hdmlnYXRpb24oKT8uZmluYWxVcmw7XG4gICAqICAgICBjb25zdCByZWRpcmVjdCA9IHJvdXRlci5wYXJzZVVybCgnLzQwNCcpO1xuICAgKiAgICAgcmV0dXJuIG5ldyBSZWRpcmVjdENvbW1hbmQocmVkaXJlY3QsIHticm93c2VyVXJsOiB0YXJnZXRPZkN1cnJlbnROYXZpZ2F0aW9ufSk7XG4gICAqICAgfVxuICAgKiAgIHJldHVybiB0cnVlO1xuICAgKiB9O1xuICAgKiBgYGBcbiAgICpcbiAgICogVGhpcyB2YWx1ZSBpcyB1c2VkIGRpcmVjdGx5LCB3aXRob3V0IGNvbnNpZGVyaW5nIGFueSBgVXJsSGFuZGluZ1N0cmF0ZWd5YC4gSW4gdGhpcyB3YXksXG4gICAqIGBicm93c2VyVXJsYCBjYW4gYWxzbyBiZSB1c2VkIHRvIHVzZSBhIGRpZmZlcmVudCB2YWx1ZSBmb3IgdGhlIGJyb3dzZXIgVVJMIHRoYW4gd2hhdCB3b3VsZCBoYXZlXG4gICAqIGJlZW4gcHJvZHVjZWQgYnkgZnJvbSB0aGUgbmF2aWdhdGlvbiBkdWUgdG8gYFVybEhhbmRsaW5nU3RyYXRlZ3kubWVyZ2VgLlxuICAgKlxuICAgKiBUaGlzIHZhbHVlIG9ubHkgYWZmZWN0cyB0aGUgcGF0aCBwcmVzZW50ZWQgaW4gdGhlIGJyb3dzZXIncyBhZGRyZXNzIGJhci4gSXQgZG9lcyBub3QgYXBwbHkgdG9cbiAgICogdGhlIGludGVybmFsIGBSb3V0ZXJgIHN0YXRlLiBJbmZvcm1hdGlvbiBzdWNoIGFzIGBwYXJhbXNgIGFuZCBgZGF0YWAgd2lsbCBtYXRjaCB0aGUgaW50ZXJuYWxcbiAgICogc3RhdGUgdXNlZCB0byBtYXRjaCByb3V0ZXMgd2hpY2ggd2lsbCBiZSBkaWZmZXJlbnQgZnJvbSB0aGUgYnJvd3NlciBVUkwgd2hlbiB1c2luZyB0aGlzIGZlYXR1cmVcbiAgICogVGhlIHNhbWUgaXMgdHJ1ZSB3aGVuIHVzaW5nIG90aGVyIEFQSXMgdGhhdCBjYXVzZSB0aGUgYnJvd3NlciBVUkwgdGhlIGRpZmZlciBmcm9tIHRoZSBSb3V0ZXJcbiAgICogc3RhdGUsIHN1Y2ggYXMgYHNraXBMb2NhdGlvbkNoYW5nZWAuXG4gICAqL1xuICByZWFkb25seSBicm93c2VyVXJsPzogVXJsVHJlZSB8IHN0cmluZztcbn1cbiJdfQ==