/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Location } from '@angular/common';
import { EnvironmentInjector, inject, Injectable, InjectionToken, runInInjectionContext, } from '@angular/core';
import { BehaviorSubject, combineLatest, EMPTY, from, of, Subject } from 'rxjs';
import { catchError, defaultIfEmpty, filter, finalize, map, switchMap, take, takeUntil, tap, } from 'rxjs/operators';
import { createRouterState } from './create_router_state';
import { INPUT_BINDER } from './directives/router_outlet';
import { BeforeActivateRoutes, GuardsCheckEnd, GuardsCheckStart, IMPERATIVE_NAVIGATION, NavigationCancel, NavigationCancellationCode, NavigationEnd, NavigationError, NavigationSkipped, NavigationSkippedCode, NavigationStart, RedirectRequest, ResolveEnd, ResolveStart, RouteConfigLoadEnd, RouteConfigLoadStart, RoutesRecognized, } from './events';
import { RedirectCommand, } from './models';
import { isNavigationCancelingError, isRedirectingNavigationCancelingError, redirectingNavigationError, } from './navigation_canceling_error';
import { activateRoutes } from './operators/activate_routes';
import { checkGuards } from './operators/check_guards';
import { recognize } from './operators/recognize';
import { resolveData } from './operators/resolve_data';
import { switchTap } from './operators/switch_tap';
import { TitleStrategy } from './page_title_strategy';
import { ROUTER_CONFIGURATION } from './router_config';
import { RouterConfigLoader } from './router_config_loader';
import { ChildrenOutletContexts } from './router_outlet_context';
import { createEmptyState, } from './router_state';
import { UrlHandlingStrategy } from './url_handling_strategy';
import { UrlSerializer } from './url_tree';
import { getAllRouteGuards } from './utils/preactivation';
import { CREATE_VIEW_TRANSITION } from './utils/view_transition';
import * as i0 from "@angular/core";
export const NAVIGATION_ERROR_HANDLER = new InjectionToken(typeof ngDevMode === 'undefined' || ngDevMode ? 'navigation error handler' : '');
export class NavigationTransitions {
    get hasRequestedNavigation() {
        return this.navigationId !== 0;
    }
    constructor() {
        this.currentNavigation = null;
        this.currentTransition = null;
        this.lastSuccessfulNavigation = null;
        /**
         * These events are used to communicate back to the Router about the state of the transition. The
         * Router wants to respond to these events in various ways. Because the `NavigationTransition`
         * class is not public, this event subject is not publicly exposed.
         */
        this.events = new Subject();
        /**
         * Used to abort the current transition with an error.
         */
        this.transitionAbortSubject = new Subject();
        this.configLoader = inject(RouterConfigLoader);
        this.environmentInjector = inject(EnvironmentInjector);
        this.urlSerializer = inject(UrlSerializer);
        this.rootContexts = inject(ChildrenOutletContexts);
        this.location = inject(Location);
        this.inputBindingEnabled = inject(INPUT_BINDER, { optional: true }) !== null;
        this.titleStrategy = inject(TitleStrategy);
        this.options = inject(ROUTER_CONFIGURATION, { optional: true }) || {};
        this.paramsInheritanceStrategy = this.options.paramsInheritanceStrategy || 'emptyOnly';
        this.urlHandlingStrategy = inject(UrlHandlingStrategy);
        this.createViewTransition = inject(CREATE_VIEW_TRANSITION, { optional: true });
        this.navigationErrorHandler = inject(NAVIGATION_ERROR_HANDLER, { optional: true });
        this.navigationId = 0;
        /**
         * Hook that enables you to pause navigation after the preactivation phase.
         * Used by `RouterModule`.
         *
         * @internal
         */
        this.afterPreactivation = () => of(void 0);
        /** @internal */
        this.rootComponentType = null;
        const onLoadStart = (r) => this.events.next(new RouteConfigLoadStart(r));
        const onLoadEnd = (r) => this.events.next(new RouteConfigLoadEnd(r));
        this.configLoader.onLoadEndListener = onLoadEnd;
        this.configLoader.onLoadStartListener = onLoadStart;
    }
    complete() {
        this.transitions?.complete();
    }
    handleNavigationRequest(request) {
        const id = ++this.navigationId;
        this.transitions?.next({ ...this.transitions.value, ...request, id });
    }
    setupNavigations(router, initialUrlTree, initialRouterState) {
        this.transitions = new BehaviorSubject({
            id: 0,
            currentUrlTree: initialUrlTree,
            currentRawUrl: initialUrlTree,
            extractedUrl: this.urlHandlingStrategy.extract(initialUrlTree),
            urlAfterRedirects: this.urlHandlingStrategy.extract(initialUrlTree),
            rawUrl: initialUrlTree,
            extras: {},
            resolve: () => { },
            reject: () => { },
            promise: Promise.resolve(true),
            source: IMPERATIVE_NAVIGATION,
            restoredState: null,
            currentSnapshot: initialRouterState.snapshot,
            targetSnapshot: null,
            currentRouterState: initialRouterState,
            targetRouterState: null,
            guards: { canActivateChecks: [], canDeactivateChecks: [] },
            guardsResult: null,
        });
        return this.transitions.pipe(filter((t) => t.id !== 0), 
        // Extract URL
        map((t) => ({
            ...t,
            extractedUrl: this.urlHandlingStrategy.extract(t.rawUrl),
        })), 
        // Using switchMap so we cancel executing navigations when a new one comes in
        switchMap((overallTransitionState) => {
            let completed = false;
            let errored = false;
            return of(overallTransitionState).pipe(switchMap((t) => {
                // It is possible that `switchMap` fails to cancel previous navigations if a new one happens synchronously while the operator
                // is processing the `next` notification of that previous navigation. This can happen when a new navigation (say 2) cancels a
                // previous one (1) and yet another navigation (3) happens synchronously in response to the `NavigationCancel` event for (1).
                // https://github.com/ReactiveX/rxjs/issues/7455
                if (this.navigationId > overallTransitionState.id) {
                    const cancellationReason = typeof ngDevMode === 'undefined' || ngDevMode
                        ? `Navigation ID ${overallTransitionState.id} is not equal to the current navigation id ${this.navigationId}`
                        : '';
                    this.cancelNavigationTransition(overallTransitionState, cancellationReason, NavigationCancellationCode.SupersededByNewNavigation);
                    return EMPTY;
                }
                this.currentTransition = overallTransitionState;
                // Store the Navigation object
                this.currentNavigation = {
                    id: t.id,
                    initialUrl: t.rawUrl,
                    extractedUrl: t.extractedUrl,
                    targetBrowserUrl: typeof t.extras.browserUrl === 'string'
                        ? this.urlSerializer.parse(t.extras.browserUrl)
                        : t.extras.browserUrl,
                    trigger: t.source,
                    extras: t.extras,
                    previousNavigation: !this.lastSuccessfulNavigation
                        ? null
                        : {
                            ...this.lastSuccessfulNavigation,
                            previousNavigation: null,
                        },
                };
                const urlTransition = !router.navigated || this.isUpdatingInternalState() || this.isUpdatedBrowserUrl();
                const onSameUrlNavigation = t.extras.onSameUrlNavigation ?? router.onSameUrlNavigation;
                if (!urlTransition && onSameUrlNavigation !== 'reload') {
                    const reason = typeof ngDevMode === 'undefined' || ngDevMode
                        ? `Navigation to ${t.rawUrl} was ignored because it is the same as the current Router URL.`
                        : '';
                    this.events.next(new NavigationSkipped(t.id, this.urlSerializer.serialize(t.rawUrl), reason, NavigationSkippedCode.IgnoredSameUrlNavigation));
                    t.resolve(false);
                    return EMPTY;
                }
                if (this.urlHandlingStrategy.shouldProcessUrl(t.rawUrl)) {
                    return of(t).pipe(
                    // Fire NavigationStart event
                    switchMap((t) => {
                        const transition = this.transitions?.getValue();
                        this.events.next(new NavigationStart(t.id, this.urlSerializer.serialize(t.extractedUrl), t.source, t.restoredState));
                        if (transition !== this.transitions?.getValue()) {
                            return EMPTY;
                        }
                        // This delay is required to match old behavior that forced
                        // navigation to always be async
                        return Promise.resolve(t);
                    }), 
                    // Recognize
                    recognize(this.environmentInjector, this.configLoader, this.rootComponentType, router.config, this.urlSerializer, this.paramsInheritanceStrategy), 
                    // Update URL if in `eager` update mode
                    tap((t) => {
                        overallTransitionState.targetSnapshot = t.targetSnapshot;
                        overallTransitionState.urlAfterRedirects = t.urlAfterRedirects;
                        this.currentNavigation = {
                            ...this.currentNavigation,
                            finalUrl: t.urlAfterRedirects,
                        };
                        // Fire RoutesRecognized
                        const routesRecognized = new RoutesRecognized(t.id, this.urlSerializer.serialize(t.extractedUrl), this.urlSerializer.serialize(t.urlAfterRedirects), t.targetSnapshot);
                        this.events.next(routesRecognized);
                    }));
                }
                else if (urlTransition &&
                    this.urlHandlingStrategy.shouldProcessUrl(t.currentRawUrl)) {
                    /* When the current URL shouldn't be processed, but the previous one
                     * was, we handle this "error condition" by navigating to the
                     * previously successful URL, but leaving the URL intact.*/
                    const { id, extractedUrl, source, restoredState, extras } = t;
                    const navStart = new NavigationStart(id, this.urlSerializer.serialize(extractedUrl), source, restoredState);
                    this.events.next(navStart);
                    const targetSnapshot = createEmptyState(this.rootComponentType).snapshot;
                    this.currentTransition = overallTransitionState = {
                        ...t,
                        targetSnapshot,
                        urlAfterRedirects: extractedUrl,
                        extras: { ...extras, skipLocationChange: false, replaceUrl: false },
                    };
                    this.currentNavigation.finalUrl = extractedUrl;
                    return of(overallTransitionState);
                }
                else {
                    /* When neither the current or previous URL can be processed, do
                     * nothing other than update router's internal reference to the
                     * current "settled" URL. This way the next navigation will be coming
                     * from the current URL in the browser.
                     */
                    const reason = typeof ngDevMode === 'undefined' || ngDevMode
                        ? `Navigation was ignored because the UrlHandlingStrategy` +
                            ` indicated neither the current URL ${t.currentRawUrl} nor target URL ${t.rawUrl} should be processed.`
                        : '';
                    this.events.next(new NavigationSkipped(t.id, this.urlSerializer.serialize(t.extractedUrl), reason, NavigationSkippedCode.IgnoredByUrlHandlingStrategy));
                    t.resolve(false);
                    return EMPTY;
                }
            }), 
            // --- GUARDS ---
            tap((t) => {
                const guardsStart = new GuardsCheckStart(t.id, this.urlSerializer.serialize(t.extractedUrl), this.urlSerializer.serialize(t.urlAfterRedirects), t.targetSnapshot);
                this.events.next(guardsStart);
            }), map((t) => {
                this.currentTransition = overallTransitionState = {
                    ...t,
                    guards: getAllRouteGuards(t.targetSnapshot, t.currentSnapshot, this.rootContexts),
                };
                return overallTransitionState;
            }), checkGuards(this.environmentInjector, (evt) => this.events.next(evt)), tap((t) => {
                overallTransitionState.guardsResult = t.guardsResult;
                if (t.guardsResult && typeof t.guardsResult !== 'boolean') {
                    throw redirectingNavigationError(this.urlSerializer, t.guardsResult);
                }
                const guardsEnd = new GuardsCheckEnd(t.id, this.urlSerializer.serialize(t.extractedUrl), this.urlSerializer.serialize(t.urlAfterRedirects), t.targetSnapshot, !!t.guardsResult);
                this.events.next(guardsEnd);
            }), filter((t) => {
                if (!t.guardsResult) {
                    this.cancelNavigationTransition(t, '', NavigationCancellationCode.GuardRejected);
                    return false;
                }
                return true;
            }), 
            // --- RESOLVE ---
            switchTap((t) => {
                if (t.guards.canActivateChecks.length) {
                    return of(t).pipe(tap((t) => {
                        const resolveStart = new ResolveStart(t.id, this.urlSerializer.serialize(t.extractedUrl), this.urlSerializer.serialize(t.urlAfterRedirects), t.targetSnapshot);
                        this.events.next(resolveStart);
                    }), switchMap((t) => {
                        let dataResolved = false;
                        return of(t).pipe(resolveData(this.paramsInheritanceStrategy, this.environmentInjector), tap({
                            next: () => (dataResolved = true),
                            complete: () => {
                                if (!dataResolved) {
                                    this.cancelNavigationTransition(t, typeof ngDevMode === 'undefined' || ngDevMode
                                        ? `At least one route resolver didn't emit any value.`
                                        : '', NavigationCancellationCode.NoDataFromResolver);
                                }
                            },
                        }));
                    }), tap((t) => {
                        const resolveEnd = new ResolveEnd(t.id, this.urlSerializer.serialize(t.extractedUrl), this.urlSerializer.serialize(t.urlAfterRedirects), t.targetSnapshot);
                        this.events.next(resolveEnd);
                    }));
                }
                return undefined;
            }), 
            // --- LOAD COMPONENTS ---
            switchTap((t) => {
                const loadComponents = (route) => {
                    const loaders = [];
                    if (route.routeConfig?.loadComponent && !route.routeConfig._loadedComponent) {
                        loaders.push(this.configLoader.loadComponent(route.routeConfig).pipe(tap((loadedComponent) => {
                            route.component = loadedComponent;
                        }), map(() => void 0)));
                    }
                    for (const child of route.children) {
                        loaders.push(...loadComponents(child));
                    }
                    return loaders;
                };
                return combineLatest(loadComponents(t.targetSnapshot.root)).pipe(defaultIfEmpty(null), take(1));
            }), switchTap(() => this.afterPreactivation()), switchMap(() => {
                const { currentSnapshot, targetSnapshot } = overallTransitionState;
                const viewTransitionStarted = this.createViewTransition?.(this.environmentInjector, currentSnapshot.root, targetSnapshot.root);
                // If view transitions are enabled, block the navigation until the view
                // transition callback starts. Otherwise, continue immediately.
                return viewTransitionStarted
                    ? from(viewTransitionStarted).pipe(map(() => overallTransitionState))
                    : of(overallTransitionState);
            }), map((t) => {
                const targetRouterState = createRouterState(router.routeReuseStrategy, t.targetSnapshot, t.currentRouterState);
                this.currentTransition = overallTransitionState = { ...t, targetRouterState };
                this.currentNavigation.targetRouterState = targetRouterState;
                return overallTransitionState;
            }), tap(() => {
                this.events.next(new BeforeActivateRoutes());
            }), activateRoutes(this.rootContexts, router.routeReuseStrategy, (evt) => this.events.next(evt), this.inputBindingEnabled), 
            // Ensure that if some observable used to drive the transition doesn't
            // complete, the navigation still finalizes This should never happen, but
            // this is done as a safety measure to avoid surfacing this error (#49567).
            take(1), tap({
                next: (t) => {
                    completed = true;
                    this.lastSuccessfulNavigation = this.currentNavigation;
                    this.events.next(new NavigationEnd(t.id, this.urlSerializer.serialize(t.extractedUrl), this.urlSerializer.serialize(t.urlAfterRedirects)));
                    this.titleStrategy?.updateTitle(t.targetRouterState.snapshot);
                    t.resolve(true);
                },
                complete: () => {
                    completed = true;
                },
            }), 
            // There used to be a lot more logic happening directly within the
            // transition Observable. Some of this logic has been refactored out to
            // other places but there may still be errors that happen there. This gives
            // us a way to cancel the transition from the outside. This may also be
            // required in the future to support something like the abort signal of the
            // Navigation API where the navigation gets aborted from outside the
            // transition.
            takeUntil(this.transitionAbortSubject.pipe(tap((err) => {
                throw err;
            }))), finalize(() => {
                /* When the navigation stream finishes either through error or success,
                 * we set the `completed` or `errored` flag. However, there are some
                 * situations where we could get here without either of those being set.
                 * For instance, a redirect during NavigationStart. Therefore, this is a
                 * catch-all to make sure the NavigationCancel event is fired when a
                 * navigation gets cancelled but not caught by other means. */
                if (!completed && !errored) {
                    const cancelationReason = typeof ngDevMode === 'undefined' || ngDevMode
                        ? `Navigation ID ${overallTransitionState.id} is not equal to the current navigation id ${this.navigationId}`
                        : '';
                    this.cancelNavigationTransition(overallTransitionState, cancelationReason, NavigationCancellationCode.SupersededByNewNavigation);
                }
                // Only clear current navigation if it is still set to the one that
                // finalized.
                if (this.currentTransition?.id === overallTransitionState.id) {
                    this.currentNavigation = null;
                    this.currentTransition = null;
                }
            }), catchError((e) => {
                errored = true;
                /* This error type is issued during Redirect, and is handled as a
                 * cancellation rather than an error. */
                if (isNavigationCancelingError(e)) {
                    this.events.next(new NavigationCancel(overallTransitionState.id, this.urlSerializer.serialize(overallTransitionState.extractedUrl), e.message, e.cancellationCode));
                    // When redirecting, we need to delay resolving the navigation
                    // promise and push it to the redirect navigation
                    if (!isRedirectingNavigationCancelingError(e)) {
                        overallTransitionState.resolve(false);
                    }
                    else {
                        this.events.next(new RedirectRequest(e.url, e.navigationBehaviorOptions));
                    }
                    /* All other errors should reset to the router's internal URL reference
                     * to the pre-error state. */
                }
                else {
                    const navigationError = new NavigationError(overallTransitionState.id, this.urlSerializer.serialize(overallTransitionState.extractedUrl), e, overallTransitionState.targetSnapshot ?? undefined);
                    try {
                        const navigationErrorHandlerResult = runInInjectionContext(this.environmentInjector, () => this.navigationErrorHandler?.(navigationError));
                        if (navigationErrorHandlerResult instanceof RedirectCommand) {
                            const { message, cancellationCode } = redirectingNavigationError(this.urlSerializer, navigationErrorHandlerResult);
                            this.events.next(new NavigationCancel(overallTransitionState.id, this.urlSerializer.serialize(overallTransitionState.extractedUrl), message, cancellationCode));
                            this.events.next(new RedirectRequest(navigationErrorHandlerResult.redirectTo, navigationErrorHandlerResult.navigationBehaviorOptions));
                        }
                        else {
                            this.events.next(navigationError);
                            // TODO(atscott): remove deprecation on errorHandler in RouterModule.forRoot and change behavior to provide NAVIGATION_ERROR_HANDLER
                            // Note: Still remove public `Router.errorHandler` property, as this is supposed to be configured in DI.
                            const errorHandlerResult = router.errorHandler(e);
                            overallTransitionState.resolve(!!errorHandlerResult);
                        }
                    }
                    catch (ee) {
                        // TODO(atscott): consider flipping the default behavior of
                        // resolveNavigationPromiseOnError to be `resolve(false)` when
                        // undefined. This is the most sane thing to do given that
                        // applications very rarely handle the promise rejection and, as a
                        // result, would get "unhandled promise rejection" console logs.
                        // The vast majority of applications would not be affected by this
                        // change so omitting a migration seems reasonable. Instead,
                        // applications that rely on rejection can specifically opt-in to the
                        // old behavior.
                        if (this.options.resolveNavigationPromiseOnError) {
                            overallTransitionState.resolve(false);
                        }
                        else {
                            overallTransitionState.reject(ee);
                        }
                    }
                }
                return EMPTY;
            }));
            // casting because `pipe` returns observable({}) when called with 8+ arguments
        }));
    }
    cancelNavigationTransition(t, reason, code) {
        const navCancel = new NavigationCancel(t.id, this.urlSerializer.serialize(t.extractedUrl), reason, code);
        this.events.next(navCancel);
        t.resolve(false);
    }
    /**
     * @returns Whether we're navigating to somewhere that is not what the Router is
     * currently set to.
     */
    isUpdatingInternalState() {
        // TODO(atscott): The serializer should likely be used instead of
        // `UrlTree.toString()`. Custom serializers are often written to handle
        // things better than the default one (objects, for example will be
        // [Object object] with the custom serializer and be "the same" when they
        // aren't).
        // (Same for isUpdatedBrowserUrl)
        return (this.currentTransition?.extractedUrl.toString() !==
            this.currentTransition?.currentUrlTree.toString());
    }
    /**
     * @returns Whether we're updating the browser URL to something new (navigation is going
     * to somewhere not displayed in the URL bar and we will update the URL
     * bar if navigation succeeds).
     */
    isUpdatedBrowserUrl() {
        // The extracted URL is the part of the URL that this application cares about. `extract` may
        // return only part of the browser URL and that part may have not changed even if some other
        // portion of the URL did.
        const currentBrowserUrl = this.urlHandlingStrategy.extract(this.urlSerializer.parse(this.location.path(true)));
        const targetBrowserUrl = this.currentNavigation?.targetBrowserUrl ?? this.currentNavigation?.extractedUrl;
        return (currentBrowserUrl.toString() !== targetBrowserUrl?.toString() &&
            !this.currentNavigation?.extras.skipLocationChange);
    }
    static { this.ɵfac = i0.ɵɵngDeclareFactory({ minVersion: "12.0.0", version: "18.1.2", ngImport: i0, type: NavigationTransitions, deps: [], target: i0.ɵɵFactoryTarget.Injectable }); }
    static { this.ɵprov = i0.ɵɵngDeclareInjectable({ minVersion: "12.0.0", version: "18.1.2", ngImport: i0, type: NavigationTransitions, providedIn: 'root' }); }
}
i0.ɵɵngDeclareClassMetadata({ minVersion: "12.0.0", version: "18.1.2", ngImport: i0, type: NavigationTransitions, decorators: [{
            type: Injectable,
            args: [{ providedIn: 'root' }]
        }], ctorParameters: () => [] });
export function isBrowserTriggeredNavigation(source) {
    return source !== IMPERATIVE_NAVIGATION;
}
//# sourceMappingURL=data:application/json;base64,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