/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { runInInjectionContext } from '@angular/core';
import { concat, defer, from, of, pipe, } from 'rxjs';
import { concatMap, first, map, mergeMap, tap } from 'rxjs/operators';
import { ActivationStart, ChildActivationStart } from '../events';
import { redirectingNavigationError } from '../navigation_canceling_error';
import { wrapIntoObservable } from '../utils/collection';
import { getClosestRouteInjector } from '../utils/config';
import { getCanActivateChild, getTokenOrFunctionIdentity, } from '../utils/preactivation';
import { isBoolean, isCanActivate, isCanActivateChild, isCanDeactivate, isCanLoad, isCanMatch, } from '../utils/type_guards';
import { prioritizedGuardValue } from './prioritized_guard_value';
export function checkGuards(injector, forwardEvent) {
    return mergeMap((t) => {
        const { targetSnapshot, currentSnapshot, guards: { canActivateChecks, canDeactivateChecks }, } = t;
        if (canDeactivateChecks.length === 0 && canActivateChecks.length === 0) {
            return of({ ...t, guardsResult: true });
        }
        return runCanDeactivateChecks(canDeactivateChecks, targetSnapshot, currentSnapshot, injector).pipe(mergeMap((canDeactivate) => {
            return canDeactivate && isBoolean(canDeactivate)
                ? runCanActivateChecks(targetSnapshot, canActivateChecks, injector, forwardEvent)
                : of(canDeactivate);
        }), map((guardsResult) => ({ ...t, guardsResult })));
    });
}
function runCanDeactivateChecks(checks, futureRSS, currRSS, injector) {
    return from(checks).pipe(mergeMap((check) => runCanDeactivate(check.component, check.route, currRSS, futureRSS, injector)), first((result) => {
        return result !== true;
    }, true));
}
function runCanActivateChecks(futureSnapshot, checks, injector, forwardEvent) {
    return from(checks).pipe(concatMap((check) => {
        return concat(fireChildActivationStart(check.route.parent, forwardEvent), fireActivationStart(check.route, forwardEvent), runCanActivateChild(futureSnapshot, check.path, injector), runCanActivate(futureSnapshot, check.route, injector));
    }), first((result) => {
        return result !== true;
    }, true));
}
/**
 * This should fire off `ActivationStart` events for each route being activated at this
 * level.
 * In other words, if you're activating `a` and `b` below, `path` will contain the
 * `ActivatedRouteSnapshot`s for both and we will fire `ActivationStart` for both. Always
 * return
 * `true` so checks continue to run.
 */
function fireActivationStart(snapshot, forwardEvent) {
    if (snapshot !== null && forwardEvent) {
        forwardEvent(new ActivationStart(snapshot));
    }
    return of(true);
}
/**
 * This should fire off `ChildActivationStart` events for each route being activated at this
 * level.
 * In other words, if you're activating `a` and `b` below, `path` will contain the
 * `ActivatedRouteSnapshot`s for both and we will fire `ChildActivationStart` for both. Always
 * return
 * `true` so checks continue to run.
 */
function fireChildActivationStart(snapshot, forwardEvent) {
    if (snapshot !== null && forwardEvent) {
        forwardEvent(new ChildActivationStart(snapshot));
    }
    return of(true);
}
function runCanActivate(futureRSS, futureARS, injector) {
    const canActivate = futureARS.routeConfig ? futureARS.routeConfig.canActivate : null;
    if (!canActivate || canActivate.length === 0)
        return of(true);
    const canActivateObservables = canActivate.map((canActivate) => {
        return defer(() => {
            const closestInjector = getClosestRouteInjector(futureARS) ?? injector;
            const guard = getTokenOrFunctionIdentity(canActivate, closestInjector);
            const guardVal = isCanActivate(guard)
                ? guard.canActivate(futureARS, futureRSS)
                : runInInjectionContext(closestInjector, () => guard(futureARS, futureRSS));
            return wrapIntoObservable(guardVal).pipe(first());
        });
    });
    return of(canActivateObservables).pipe(prioritizedGuardValue());
}
function runCanActivateChild(futureRSS, path, injector) {
    const futureARS = path[path.length - 1];
    const canActivateChildGuards = path
        .slice(0, path.length - 1)
        .reverse()
        .map((p) => getCanActivateChild(p))
        .filter((_) => _ !== null);
    const canActivateChildGuardsMapped = canActivateChildGuards.map((d) => {
        return defer(() => {
            const guardsMapped = d.guards.map((canActivateChild) => {
                const closestInjector = getClosestRouteInjector(d.node) ?? injector;
                const guard = getTokenOrFunctionIdentity(canActivateChild, closestInjector);
                const guardVal = isCanActivateChild(guard)
                    ? guard.canActivateChild(futureARS, futureRSS)
                    : runInInjectionContext(closestInjector, () => guard(futureARS, futureRSS));
                return wrapIntoObservable(guardVal).pipe(first());
            });
            return of(guardsMapped).pipe(prioritizedGuardValue());
        });
    });
    return of(canActivateChildGuardsMapped).pipe(prioritizedGuardValue());
}
function runCanDeactivate(component, currARS, currRSS, futureRSS, injector) {
    const canDeactivate = currARS && currARS.routeConfig ? currARS.routeConfig.canDeactivate : null;
    if (!canDeactivate || canDeactivate.length === 0)
        return of(true);
    const canDeactivateObservables = canDeactivate.map((c) => {
        const closestInjector = getClosestRouteInjector(currARS) ?? injector;
        const guard = getTokenOrFunctionIdentity(c, closestInjector);
        const guardVal = isCanDeactivate(guard)
            ? guard.canDeactivate(component, currARS, currRSS, futureRSS)
            : runInInjectionContext(closestInjector, () => guard(component, currARS, currRSS, futureRSS));
        return wrapIntoObservable(guardVal).pipe(first());
    });
    return of(canDeactivateObservables).pipe(prioritizedGuardValue());
}
export function runCanLoadGuards(injector, route, segments, urlSerializer) {
    const canLoad = route.canLoad;
    if (canLoad === undefined || canLoad.length === 0) {
        return of(true);
    }
    const canLoadObservables = canLoad.map((injectionToken) => {
        const guard = getTokenOrFunctionIdentity(injectionToken, injector);
        const guardVal = isCanLoad(guard)
            ? guard.canLoad(route, segments)
            : runInInjectionContext(injector, () => guard(route, segments));
        return wrapIntoObservable(guardVal);
    });
    return of(canLoadObservables).pipe(prioritizedGuardValue(), redirectIfUrlTree(urlSerializer));
}
function redirectIfUrlTree(urlSerializer) {
    return pipe(tap((result) => {
        if (typeof result === 'boolean')
            return;
        throw redirectingNavigationError(urlSerializer, result);
    }), map((result) => result === true));
}
export function runCanMatchGuards(injector, route, segments, urlSerializer) {
    const canMatch = route.canMatch;
    if (!canMatch || canMatch.length === 0)
        return of(true);
    const canMatchObservables = canMatch.map((injectionToken) => {
        const guard = getTokenOrFunctionIdentity(injectionToken, injector);
        const guardVal = isCanMatch(guard)
            ? guard.canMatch(route, segments)
            : runInInjectionContext(injector, () => guard(route, segments));
        return wrapIntoObservable(guardVal);
    });
    return of(canMatchObservables).pipe(prioritizedGuardValue(), redirectIfUrlTree(urlSerializer));
}
//# sourceMappingURL=data:application/json;base64,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