/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { inject } from '@angular/core';
/**
 * Maps an array of injectable classes with canMatch functions to an array of equivalent
 * `CanMatchFn` for use in a `Route` definition.
 *
 * Usage {@example router/utils/functional_guards.ts region='CanActivate'}
 *
 * @publicApi
 * @see {@link Route}
 */
export function mapToCanMatch(providers) {
    return providers.map((provider) => (...params) => inject(provider).canMatch(...params));
}
/**
 * Maps an array of injectable classes with canActivate functions to an array of equivalent
 * `CanActivateFn` for use in a `Route` definition.
 *
 * Usage {@example router/utils/functional_guards.ts region='CanActivate'}
 *
 * @publicApi
 * @see {@link Route}
 */
export function mapToCanActivate(providers) {
    return providers.map((provider) => (...params) => inject(provider).canActivate(...params));
}
/**
 * Maps an array of injectable classes with canActivateChild functions to an array of equivalent
 * `CanActivateChildFn` for use in a `Route` definition.
 *
 * Usage {@example router/utils/functional_guards.ts region='CanActivate'}
 *
 * @publicApi
 * @see {@link Route}
 */
export function mapToCanActivateChild(providers) {
    return providers.map((provider) => (...params) => inject(provider).canActivateChild(...params));
}
/**
 * Maps an array of injectable classes with canDeactivate functions to an array of equivalent
 * `CanDeactivateFn` for use in a `Route` definition.
 *
 * Usage {@example router/utils/functional_guards.ts region='CanActivate'}
 *
 * @publicApi
 * @see {@link Route}
 */
export function mapToCanDeactivate(providers) {
    return providers.map((provider) => (...params) => inject(provider).canDeactivate(...params));
}
/**
 * Maps an injectable class with a resolve function to an equivalent `ResolveFn`
 * for use in a `Route` definition.
 *
 * Usage {@example router/utils/functional_guards.ts region='Resolve'}
 *
 * @publicApi
 * @see {@link Route}
 */
export function mapToResolve(provider) {
    return (...params) => inject(provider).resolve(...params);
}
//# sourceMappingURL=data:application/json;base64,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